/*
 * Java
 *
 * Copyright 2008-2023 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.kf.elflw;

import com.is2t.kf.MemoryBlock;

/**
 * Section that contains target specific raw binary program bytes (asm instructions, data)
 */
public class ProgAllocationSection extends Section implements AllocationSection {

	/**
	 * the 0 based index of the section within the ELF Unit
	 */
	private final int sectionIndex;

	/**
	 * the section size in bytes
	 */
	private final int size;

	/**
	 * the memory block where is stored the data before relocation. Is equals to {@link #relocatedData} if there is no
	 * relocation to apply.
	 */
	private final MemoryBlock data;

	/**
	 * the memory block where is stored the data after relocation. Is equals to {@link #data} if there is no relocation
	 * to apply.
	 */
	private final MemoryBlock relocatedData;

	public ProgAllocationSection(int sectionIndex, int size, MemoryBlock data, MemoryBlock relocatedData) {
		assert size <= data.getSize();
		assert size <= relocatedData.getSize();
		this.size = size;
		this.sectionIndex = sectionIndex;
		this.data = data;
		this.relocatedData = relocatedData;
	}

	@Override
	public int memSize() {
		return size();
	}

	@Override
	public MemoryBlock getMemoryBlock() {
		return this.data;
	}

	/**
	 * @return the relocated memory block descriptor
	 */
	public MemoryBlock getRelocatedMemoryBlock() {
		return this.relocatedData;
	}

	@Override
	public long getRelocatedAddress() {
		return this.relocatedData.getAddress();
	}

	/**
	 * @return the 0 based index of the section within the ELF Unit
	 */
	public int getSectionIndex() {
		return this.sectionIndex;
	}

	@Override
	public int size() {
		return this.size;
	}
}
