/**
 * Java
 *
 * Copyright 2011-2012 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.bon;

import java.util.Hashtable;

public class WeakHashtable extends Hashtable {

	private ReferenceQueue queue;

	public WeakHashtable() {
		super();
		queue = new ReferenceQueue();
	}

	public WeakHashtable(int initialCapacity) {
		super(initialCapacity);
		queue = new ReferenceQueue();
	}

	public synchronized Object put(Object key, Object value) throws NullPointerException {
		//This is a check to prevent problem on recursive call of put. This may
		//happen if the Hashtable implementation of put recall put (when a rehash
		//occurs for example).
		//In this case, the given key is already the WeakKey linked to the real
		//object we want to put in the table. So we don't need to wrap it again.
		//
		//There is no risk that the application gives a WeakKey instance as key
		//because this class is private. The application cannot create instances
		//of WeakKey.
		if(!(key instanceof WeakKey)){
			clean();
			key = new WeakKey(key, queue);
		}
		//else the object we want to put is already wrapped in a WeakKey
		
		return super.put(key, value);
	}

	public synchronized Object get(Object key) {
		clean();
		return super.get(key);
	}

	private void clean() {
		EnqueuedWeakReference reference;
		while ((reference = queue.poll()) != null) {
			remove(reference);
		}
	}

	class WeakKey extends EnqueuedWeakReference {
		int hashcode;

		public WeakKey(Object key, ReferenceQueue queue) {
			super(key, queue);
			hashcode = key.hashCode();
		}

		public int hashCode() {
			return hashcode;
		}

		public boolean equals(Object obj) {
			Object originalKey = this.get();
			if(originalKey == null) {
				//original reference is garbage collected
				return obj == this;
			} else {
				//call the equals method on the given object because it can
				// be another weak key which will compare its referenced
				// object with originalKey
				return obj.equals(originalKey);
			}
		}
	}

}
