/*
 * Java
 *
 * Copyright 2009-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.bon;

import com.is2t.bon.timer.TimerTaskList;

/**
 *  A task that can be scheduled for one-time or repeated execution by a Timer.
 */
public abstract class TimerTask implements Runnable {

	public static final int NOT_SCHEDULED = -1;//value of scheduleTime when the TimerTask has not been scheduled

	/** Set when the task is scheduled */
	public TimerTaskList list;

	public long absoluteTime;//time at which this timer task should be launched (initialized to NOT_SCHEDULED)
	public long period;//period of the task, 0 if the task is not repeatable
	public boolean fixedRate;
	//set to true when the task is canceled
	public boolean isCanceled;
	//set to true when the task has run or is running
	public boolean hasRun;

	/**
	 * Creates a new timer task.
	 * */
	protected TimerTask(){
		this.absoluteTime = NOT_SCHEDULED;
		//this.isCanceled = false; VM DONE
	}

	/**
	 * The action to be performed by this timer task.
	 * Specified by:
	 *   run in interface Runnable Following copied from interface: java.lang.Runnable
	 * See Also:
	 *   Thread.run()
	 */
	@Override
	public abstract void run();


	/**
	 * Cancels this timer task. If the task has been scheduled for one-time
	 * execution and has not yet run, or has not yet been scheduled, it will
	 * never run. If the task has been scheduled for repeated execution, it
	 * will never run again. (If the task is running when this call occurs, the
	 * task will run to completion, but will never run again.)
	 *
	 * Note that calling this method from within the run method of a repeating
	 * timer task absolutely guarantees that the timer task will not run again.
	 *
	 * This method may be called repeatedly; the second and subsequent calls
	 * have no effect.
	 *
	 * Returns:
	 *   true if this task is scheduled for one-time execution and
	 *   has not yet run, or this task is scheduled for repeated execution.
	 *   Returns false if the task was scheduled for one-time execution and has already run,
	 *   or if the task was never scheduled, or if the task was already canceled (Loosely speaking,
	 *   this method returns true if it prevents one or more scheduled
	 *   executions from taking place.)
	 */
	public boolean cancel() {

		long originalPeriod;
		TimerTaskList list;

		synchronized (this) {
			if(isCanceled) {
				return false;
			}

			this.isCanceled = true;
			originalPeriod = this.period;
			this.period = 0;//avoid reschedule of the task
			list = this.list;
			this.list = null;
		}

		if (list != null) {
			list.removeReferenceTo(this);
		}

		//return true if the task is scheduled. If the task is one-time execution and has
		//run, returns false.
		return absoluteTime != NOT_SCHEDULED && !(hasRun && originalPeriod == 0);
	}


	/**
	 * Returns the scheduled execution time of the most recent actual execution
	 * of this task. (If this method is invoked while task execution is in
	 * progress, the return value is the scheduled execution time of the
	 * ongoing task execution.)
	 *
	 * This method is typically invoked from within a task's run method, to
	 * determine whether the current execution of the task is sufficiently
	 * timely to warrant performing the scheduled activity:
	 *
	 * public void run() { if (System.currentTimeMillis() -
	 * scheduledExecutionTime() >= MAX_TARDINESS) return; // Too late; skip
	 * this execution. // Perform the task }
	 *
	 *
	 * This method is typically not used in conjunction with fixed-delay
	 * execution repeating tasks, as their scheduled execution times are
	 * allowed to drift over time, and so are not terribly significant.
	 *
	 * Returns:
	 *   the time at which the most recent execution of this task was scheduled to occur,
	 *   in the format returned by Date.getTime().
	 *   The return value is undefined if the task has yet to commence its first execution.
	 *
	 * See Also:
	 *   Date.getTime()
	 */
	public long scheduledExecutionTime() {
		return absoluteTime;
	}

	public void uncaughtException(Timer timer, Throwable e){
		list.uncaughtException(timer, this, e);
	}

	/**
	 * Configure this task as scheduled in the given timer.<br>
	 * If the task has already been scheduled this method throws an IllegalStateException.<br>
	 * This method is thread safe.
	 */
	public synchronized void setScheduled(TimerTaskList list, long absoluteTime, long period, boolean fixedRate) {

		if (this.absoluteTime != TimerTask.NOT_SCHEDULED || this.isCanceled)
		{
			throw new IllegalStateException();//the task has already been scheduled or canceled
		}

		this.absoluteTime = absoluteTime;
		this.period = period;
		this.fixedRate = fixedRate;
		this.list = list;
	}

}
