/*
 * Java
 *
 * Copyright 2009-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.bon;

import java.io.PrintStream;
import java.lang.Thread.UncaughtExceptionHandler;
import java.util.Date;

import com.is2t.bon.BONFactory;
import com.is2t.bon.timer.TimerTaskList;
import com.is2t.bon.timer.TimerTaskRef;

import ej.annotation.Nullable;

/**
 * A facility for threads to schedule tasks for future execution in a
 * background thread. Tasks may be scheduled for one-time execution, or for
 * repeated execution at regular intervals.
 *
 * Corresponding to each Timer object is a single background thread that is
 * used to execute all of the timer's tasks, sequentially. Timer tasks should
 * complete quickly. If a timer task takes excessive time to complete, it
 * "hogs" the timer's task execution thread. This can, in turn, delay the
 * execution of subsequent tasks, which may "bunch up" and execute in rapid
 * succession when (and if) the offending task finally completes.
 *
 * After the last live reference to a Timer object goes away and all
 * outstanding tasks have completed execution, the timer's task execution
 * thread terminates gracefully (and becomes subject to garbage collection).
 * However, this can take arbitrarily long to occur. By default, the task
 * execution thread does not run as a daemon thread, so it is capable of
 * keeping an application from terminating. If a caller wants to terminate a
 * timer's task execution thread rapidly, the caller should invoke the timer's
 * cancel method.
 *
 * If the timer's task execution thread terminates unexpectedly, any further
 * attempt to schedule a task on the timer will result in an
 * IllegalStateException, as if the timer's cancel method had been invoked.
 *
 * This class is thread-safe: multiple threads can share a single Timer object
 * without the need for external synchronization.
 *
 * This class does not offer real-time guarantees: it schedules tasks using the
 * Object.wait(long) method. The resolution of the Timer is implementation and
 * device dependent.
 *
 * Timers function only within a single VM and are canceled when the VM exits.
 * When the VM is started no timers exist, they are created only by
 */
public class Timer implements Runnable {

	/*
	 * Important Note: Timer does not manage directly TimerTask instances. This is
	 * necessary for KF extended behavior, to avoid Kernel stale references.
	 * Moreover, TimerTask instances cannot be linked directly together: each
	 * TimerTask instance is wrapped by a TimerTaskRef instance. See also
	 * TimerTaskRef#task field comment.
	 */
	public TimerTaskList list;

	/*
	 * Set to true when run() method is executed
	 */
	public boolean started;

	/**
	 * Default Uncaught Exception Handler is <code>null</code> by default if not
	 * explicitly set.
	 */
	public static Thread.UncaughtExceptionHandler defaultUncaughtExceptionHandler;

	/**
	 * Uncaught Exception Handler is <code>null</code> by default if not explicitly
	 * set.
	 */
	public Thread.UncaughtExceptionHandler uncaughtExceptionHandler;

	/**
	 * Creates a new timer. The associated thread does not run as a daemon thread,
	 * which may prevent an application from terminating.
	 */
	public Timer() {
		this(true);
	}

	/**
	 * <p>
	 * Creates a new timer.
	 * </p>
	 * <p>
	 * If the given boolean is <code>true</code>, a thread is created to execute this timer. The associated thread
	 * does not run as a daemon thread, which may prevent an application from terminating. All the {@link TimerTask}
	 * scheduled by the timer will be executed in the context of the associated thread.
	 * </p>
	 * <p>
	 * Otherwise an applicative thread must call the {@link #run()} method in order
	 * to execute tasks scheduling.
	 * </p>
	 * @param automatic If <code>true</code> a thread is created to run this timer, otherwise the application must manage this timer.
	 * @see #run()
	 */
	public Timer(boolean automatic) {
		this.list = BONFactory.getInstance().newTimerTaskList();
		if(automatic){
			new Thread(this).start();
		}
	}


	/**
	 * Schedules the specified task for execution after the specified delay.
	 * Parameters:
	 *   task - task to be scheduled.
	 *   delay - delay in milliseconds before task is to be executed.
	 *   Note that the actual delay may be different than the amount requested
	 *   since the resolution of the Timer is implementation and
	 *   device dependent.
	 * Throws:
	 *   IllegalArgumentException
	 *    - if delay is negative, or delay + System.currentTimeMillis() is negative.
	 *   IllegalStateException
	 *    - if task was already scheduled or canceled, or timer was canceled.
	 */
	public void schedule(TimerTask task, long delay) {
		long absoluteTime = CurrentTime.get(true) + delay;
		if (delay < 0 || absoluteTime < 0) {
			throw new IllegalArgumentException();
		}

		schedule(task, absoluteTime, 0, false);
	}


	/**
	 * Schedules the specified task for execution at the specified time. If the time is in the past,
	 * the task is scheduled for immediate execution.
	 *
	 * Parameters: task - task to be scheduled. time - time at which task is to be executed. Throws:
	 * IllegalArgumentException - if time.getTime() is negative. IllegalStateException - if task was
	 * already scheduled or canceled, timer was canceled, or timer thread terminated.
	 */
	public void schedule(TimerTask task, Date time) {
		long absoluteTime = time.getTime();
		if(absoluteTime < 0) {
			throw new IllegalArgumentException();
		}

		schedule(task, absoluteTime, 0, false);
	}


	/**
	 * Schedules the specified task for repeated fixed-delay execution, beginning after the
	 * specified delay. Subsequent executions take place at approximately regular intervals
	 * separated by the specified period. Note that the actual delay may be different than the
	 * amount requested since the resolution of the Timer is implementation and device dependent.
	 *
	 * In fixed-delay execution, each execution is scheduled relative to the actual execution time
	 * of the previous execution. If an execution is delayed for any reason (such as garbage
	 * collection or other background activity), subsequent executions will be delayed as well. In
	 * the long run, the frequency of execution will generally be slightly lower than the reciprocal
	 * of the specified period (assuming the system clock underlying Object.wait(long) is accurate).
	 *
	 * Fixed-delay execution is appropriate for recurring activities that require "smoothness." In
	 * other words, it is appropriate for activities where it is more important to keep the
	 * frequency accurate in the short run than in the long run. This includes most animation tasks,
	 * such as blinking a cursor at regular intervals. It also includes tasks wherein regular
	 * activity is performed in response to human input, such as automatically repeating a character
	 * as long as a key is held down.
	 *
	 * Parameters: task - task to be scheduled. delay - delay in milliseconds before task is to be
	 * executed. Note that the actual delay may be different than the amount requested since the
	 * resolution of the Timer is implementation and device dependent. period - time in milliseconds
	 * between successive task executions.
	 *
	 * Throws: IllegalArgumentException - if delay is negative, or delay +
	 * System.currentTimeMillis() is negative. IllegalStateException - if task was already scheduled
	 * or canceled, timer was canceled, or timer thread terminated.
	 */
	public void schedule(TimerTask task, long delay, long period) {
		long absoluteTime = CurrentTime.get(true) + delay;
		if(delay < 0 || period <= 0 || absoluteTime < 0) {
			throw new IllegalArgumentException();
		}

		schedule(task, absoluteTime, period, false);
	}


	/**
	 * Schedules the specified task for repeated fixed-delay execution, beginning at the specified
	 * time. Subsequent executions take place at approximately regular intervals, separated by the
	 * specified period.
	 *
	 * In fixed-delay execution, each execution is scheduled relative to the actual execution time
	 * of the previous execution. If an execution is delayed for any reason (such as garbage
	 * collection or other background activity), subsequent executions will be delayed as well. In
	 * the long run, the frequency of execution will generally be slightly lower than the reciprocal
	 * of the specified period (assuming the system clock underlying Object.wait(long) is accurate).
	 *
	 * Fixed-delay execution is appropriate for recurring activities that require "smoothness." In
	 * other words, it is appropriate for activities where it is more important to keep the
	 * frequency accurate in the short run than in the long run. This includes most animation tasks,
	 * such as blinking a cursor at regular intervals. It also includes tasks wherein regular
	 * activity is performed in response to human input, such as automatically repeating a character
	 * as long as a key is held down.
	 *
	 *
	 * Parameters: task - task to be scheduled. firstTime - First time at which task is to be
	 * executed. period - time in milliseconds between successive task executions.
	 *
	 * Throws: IllegalArgumentException - if time.getTime() is negative.IllegalStateException - if
	 * task was already scheduled or canceled, timer was canceled, or timer thread terminated.
	 */
	public void schedule(TimerTask task, Date firstTime, long period) {
		long absoluteTime = firstTime.getTime();
		if(absoluteTime < 0 || period <= 0) {
			throw new IllegalArgumentException();
		}

		schedule(task, absoluteTime, period, false);
	}



	/**
	 * Schedules the specified task for repeated fixed-rate execution, beginning after the specified
	 * delay. Subsequent executions take place at approximately regular intervals, separated by the
	 * specified period.
	 *
	 * In fixed-rate execution, each execution is scheduled relative to the scheduled execution time
	 * of the initial execution. If an execution is delayed for any reason (such as garbage
	 * collection or other background activity), two or more executions will occur in rapid
	 * succession to "catch up." In the long run, the frequency of execution will be exactly the
	 * reciprocal of the specified period (assuming the system clock underlying Object.wait(long) is
	 * accurate).
	 *
	 * Fixed-rate execution is appropriate for recurring activities that are sensitive to absolute
	 * time, such as ringing a chime every hour on the hour, or running scheduled maintenance every
	 * day at a particular time. It is also appropriate for for recurring activities where the total
	 * time to perform a fixed number of executions is important, such as a countdown timer that
	 * ticks once every second for ten seconds. Finally, fixed-rate execution is appropriate for
	 * scheduling multiple repeating timer tasks that must remain synchronized with respect to one
	 * another.
	 *
	 * Parameters: task - task to be scheduled. delay - delay in milliseconds before task is to be
	 * executed. Note that the actual delay may be different than the amount requested since the
	 * resolution of the Timer is implementation and device dependent. period - time in milliseconds
	 * between successive task executions.
	 *
	 * Throws: IllegalArgumentException - if delay is negative, or delay +
	 * System.currentTimeMillis() is negative.IllegalStateException - if task was already scheduled
	 * or canceled, timer was canceled, or timer thread terminated.
	 */
	public void scheduleAtFixedRate(TimerTask task, long delay, long period) {
		long absoluteTime = CurrentTime.get(true) + delay;
		if(delay < 0 || period <= 0 || absoluteTime < 0) {
			throw new IllegalArgumentException();
		}

		schedule(task, absoluteTime, period, true);
	}


	/**
	 * Schedules the specified task for repeated fixed-rate execution, beginning at the specified
	 * time. Subsequent executions take place at approximately regular intervals, separated by the
	 * specified period.
	 *
	 * In fixed-rate execution, each execution is scheduled relative to the scheduled execution time
	 * of the initial execution. If an execution is delayed for any reason (such as garbage
	 * collection or other background activity), two or more executions will occur in rapid
	 * succession to "catch up." In the long run, the frequency of execution will be exactly the
	 * reciprocal of the specified period (assuming the system clock underlying Object.wait(long) is
	 * accurate).
	 *
	 * Fixed-rate execution is appropriate for recurring activities that are sensitive to absolute
	 * time, such as ringing a chime every hour on the hour, or running scheduled maintenance every
	 * day at a particular time. It is also appropriate for for recurring activities where the total
	 * time to perform a fixed number of executions is important, such as a countdown timer that
	 * ticks once every second for ten seconds. Finally, fixed-rate execution is appropriate for
	 * scheduling multiple repeating timer tasks that must remain synchronized with respect to one
	 * another.
	 *
	 * Parameters: task - task to be scheduled. firstTime - First time at which task is to be
	 * executed. period - time in milliseconds between successive task executions.
	 *
	 * Throws: IllegalArgumentException - if time.getTime() is negative.IllegalStateException - if
	 * task was already scheduled or canceled, timer was canceled, or timer thread terminated.
	 */
	public void scheduleAtFixedRate(TimerTask task, Date firstTime, long period) {
		long absoluteTime = firstTime.getTime();
		if(absoluteTime < 0 || period <= 0) {
			throw new IllegalArgumentException();
		}

		schedule(task, absoluteTime, period, true);
	}

	protected void schedule(TimerTask task, long absoluteTime, long period, boolean fixedRate){
		//absoluteTime should be positive,
		//period should be positive (if period == 0 then alarm is not repeatable).
		//This method is thread safe

		//check if task has not already been scheduled and if this timer is running
		synchronized (list) {

			if (list.isCanceled) {
				throw new IllegalStateException();
			}

			//configure task and check if it has not already been scheduled
			task.setScheduled(list, absoluteTime, period, fixedRate);

			TimerTaskRef taskRef = list.newTimerTaskRef(task);
			list.addTaskToScheduledTasks(taskRef, absoluteTime);
		}
	}

	/**
	 * Terminates this timer, discarding any currently scheduled tasks. Does not interfere with a
	 * currently executing task (if it exists). Once a timer has been terminated, its execution
	 * thread terminates gracefully, and no more tasks may be scheduled on it.
	 *
	 * Note that calling this method from within the run method of a timer task that was invoked by
	 * this timer absolutely guarantees that the ongoing task execution is the last task execution
	 * that will ever be performed by this timer.
	 */
	public void cancel() {
		list.cancel();
	}


	/**
	 * <p>
	 * Executes {@link TimerTask} scheduling. This method must be called only for {@link Timer} that are
	 * not automatic (i.e. no thread is automatically started at {@link Timer} creation.
	 * </p>
	 * <p>
	 * This method can be called once on this {@link Timer} and returns only if {@link #cancel()} is called.
	 * </p>
	 * <p>
	 * This method does not catch the exceptions that may be thrown by the scheduled {@link TimerTask}.
	 * </p>
	 * @throws IllegalStateException
	 *             if this timer is already running or canceled.
	 */
	@Override
	public void run(){
		TimerTaskList list = this.list;

		//Checks if the timer has not already been started (can be started only once)
		synchronized (list) {
			if(started) {
				throw new IllegalStateException();
			}
			this.started = true;
		}

		while (!list.isCanceled) { // take a snapshot without synchronization
			long currentTime = CurrentTime.get(true);
			//Extract from timer linked list of tasks, a list of tasks to launch now

			TimerTaskRef taskToLaunchRef = null;// linked list of tasks to launch
			synchronized (list) {
				//find last task to launch now and first task that should not be launched now
				taskToLaunchRef = list.nextTaskRef;
				TimerTaskRef lastTaskToLaunchRef = null;
				TimerTaskRef firstTaskToNotLaunchRef = taskToLaunchRef;
				while (firstTaskToNotLaunchRef != null && firstTaskToNotLaunchRef.getAbsoluteTime() <= currentTime) {
					lastTaskToLaunchRef = firstTaskToNotLaunchRef;
					firstTaskToNotLaunchRef = firstTaskToNotLaunchRef.next;
				}
				//here if lastTaskToLaunch is not null, there are tasks to launch now
				if (lastTaskToLaunchRef != null) {
					//remove all tasks to launch from the scheduled tasks
					list.nextTaskRef = firstTaskToNotLaunchRef;
					lastTaskToLaunchRef.next = null;
					//mark all tasks to launch as launched
					TimerTaskRef task = taskToLaunchRef;
					while(task != null){
						task = task.next;
					}

					list.tasksToLaunch = taskToLaunchRef;
				}
				else
				{	//no task to launch now
					//firstTaskToLaunch is the next task to launch

					//Check again for cancellation in the synchronize.
					//Previous check was done outside of the synchronize so 'isCanceled' value
					//may have change.
					if (!list.isCanceled) {
						try
						{
							if (taskToLaunchRef != null) {
								//there is a task scheduled
								//sleep to wake up when next task should be launched
								long sleepTime = taskToLaunchRef.getAbsoluteTime() - CurrentTime.get(true);
								taskToLaunchRef = null;
								if(sleepTime > 0){
									list.wait(sleepTime);
								}
								//else next task should be launched now (continue)
							}
							else {
								//there is no task scheduled
								list.wait();// wait infinite
							}
						}
						catch(InterruptedException e){

						}
					}
					//here we can have be woke up because
					//- task has been added to timer
					//- task should be launched
					//- timer is stopped
					//- current thread has been interrupted
					//
					//continue execution, a new loop will be done unless this pump has been stopped
				}
			}

			//run all the launched tasks
			list.runLaunchedTasks(this);
		}
	}

	public static void setDefaultUncaughtExceptionHandler(@Nullable UncaughtExceptionHandler h) {
		if (Constants.getBoolean(Constants.CONSTANT_USE_SECURITYMANAGER)) {
			SecurityManager sm = System.getSecurityManager();
			if (sm != null) {
				sm.checkPermission(new RuntimePermission("setDefaultUncaughtExceptionHandler"));
			}
		}
		defaultUncaughtExceptionHandler = h;
	}

	@Nullable
	public static UncaughtExceptionHandler getDefaultUncaughtExceptionHandler() {
		return defaultUncaughtExceptionHandler;
	}

	public void setUncaughtExceptionHandler(Thread.UncaughtExceptionHandler h) {
		this.uncaughtExceptionHandler = h;
	}

	@Nullable
	public Thread.UncaughtExceptionHandler getUncaughtExceptionHandler() {
		return this.uncaughtExceptionHandler;
	}

	public void dump(PrintStream s){
		try{
			s.print("Timer ");
			s.println(started ? "started:" : "not started:");

			list.dump(s);
		}
		catch(Throwable t){
			// Catch any kind of error.
			s.println(t.toString());
		}
	}
}
