/*
 * Java
 *
 * Copyright 2017-2023 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.bon;

import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;

import ej.sni.SNI;

public class ResourceBuffer implements Closeable {

	/** Encoded <code>true</code> */
	private static final int TRUE = 1;

	/** native data */
	private int dataSize; // native knows

	/** native data */
	private int data; // native knows

	/**
	 * External resource ID (positive value) or any negative values (internal
	 * resource)
	 */
	private int id; // native knows

	/**
	 * Current data pointer
	 */
	private int ptr; // native knows

	/** Start pointer (inclusive) */
	private int startAddr; // native knows

	/** End pointer (inclusive) */
	private int endAddr; // native knows

	static {
		// SOAR error 66: cannot only touch this class in ni file
		// (nativeRequirements). Force to embed this exception when this
		// class is used. Do not use a .types.list to not always embed this
		// exception when it is not required.
		new EOFException();
	}

	/**
	 * @throws IOException
	 *
	 */
	public ResourceBuffer(String resource) throws IOException {

		if (!resource.startsWith("/")) { // can throw a NPE
			throw new IllegalArgumentException();
		}

		int resourceNameLength = resource.length;
		byte[] path = new byte[resourceNameLength+1];
		SNI.toCString(resource, path);
		if (!open(path, resourceNameLength)) {
			throw new IOException();
		}
	}

	@Override
	public native void close() throws IOException;

	public native int readVarSInt() throws IOException;

	public native int readVarUInt() throws IOException;

	public native long readVarLong() throws IOException;

	public boolean readBoolean() throws IOException {
		int value = readUnsignedByte();
		return value == TRUE;
	}

	public byte readByte() throws IOException {
		return (byte) readSignedByte();
	}

	public short readShort() throws IOException {
		return (short) platformReadInt16();
	}

	public char readChar() throws IOException {
		return (char) platformReadUInt16();
	}

	public int readInt() throws IOException {
		return platformReadInt32();
	}

	// public native long readLong() throws IOException;
	//
	// public native float readFloat() throws IOException;
	//
	// public native double readDouble() throws IOException;

	public native String readString() throws IOException;

	public ResourceArray readArray() throws IOException {
		ResourceArray array = new ResourceArray(this);
		return array;
	}

	public void seek(long offset) throws IOException {
		checkOpen();
		setPointer((int) (startAddr + offset), false);
	}

	public int available() throws IOException {
		checkOpen();
		return ptr <= endAddr ? endAddr - ptr + 1 : 0;
	}

	public void align(int nbBytes) throws IOException {
		try {
			seek(align(ptr, nbBytes) - startAddr);
		}
		catch(IndexOutOfBoundsException e) {
			throw new EOFException();
		}
	}

	/* package visibililty */
	native int readUnsignedByte() throws IOException;

	/* package visibililty */
	native int readSignedByte() throws IOException;

	/* package visibililty */
	native int platformReadUInt16() throws IOException;

	/* package visibililty */
	native int platformReadInt16() throws IOException;

	/* package visibililty */
	native int platformReadInt32() throws IOException;

	/* package visibililty */
	/**
	 * If <code>allowEOF</code> is true, the given address can be right after the last element of the buffer.
	 * This corresponds to an EOF.
	 * If <code>allowEOF</code> is false, an {@link IndexOutOfBoundsException} is thrown if the given address
	 * is right after the last element of the buffer.
	 */
	void setPointer(int address, boolean allowEOF) {
		int limitAddr = this.endAddr; // last allowed address
		if(allowEOF) {
			++limitAddr;
		}
		if(address < this.startAddr || address > limitAddr) {
			throw new IndexOutOfBoundsException();
		}
		this.ptr = address;
	}

	/* package visibility */
	int getCurrentPtr() {
		return ptr;
	}

	/**
	 * Opens the resource buffer and initializes this ResourceBuffer.
	 *
	 * @param path a NULL-Terminated string that starts with '/'
	 * @param length length of path without the final '\0'
	 * @return true on success, false otherwise.
	 */
	private native boolean open(byte[] path, int length);

	/**
	 * Align a value on given alignment
	 *
	 * @param value
	 *            the value to align
	 * @param alignment
	 *            the alignment to perform
	 * @return the aligned value
	 */
	/* package visibility */
	static int align(int value, int alignment) {
		--alignment;
		return alignment >= 1 ? (value + alignment) & ~alignment : value;
	}


	/**
	 * Ensures resource is opened. Same test in native side when reading a data.
	 *
	 * @throws IOException
	 *             when resource is closed.
	 */
	private void checkOpen() throws IOException {
		// when closed, endAddr is set to startAddr-2 (see native side)
		// - endAddr == startAddr means there is 1 element in the buffer
		// - endAddr == startAddr-1 means there is 0 element in the buffer
		// - endAddr == startAddr-2 means the buffer is closed
		if (endAddr == startAddr-2) {
			throw new IOException();
		}
	}
}
