/*
 * Java
 *
 * Copyright 2017-2023 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.bon;

import java.io.EOFException;
import java.io.IOException;

public class ResourceArray {

	/** type of elements */
	private static final byte ARRAY_BITS_MASK = (byte) (0x7 << 5);
	private static final byte ARRAY_BITS_U8 = (byte) (0x0 << 5);
	private static final byte ARRAY_BITS_U16 = (byte) (0x1 << 5);
	private static final byte ARRAY_BITS_U32 = (byte) (0x2 << 5);
	private static final byte ARRAY_BITS_8 = (byte) (0x3 << 5);
	private static final byte ARRAY_BITS_16 = (byte) (0x4 << 5);
	private static final byte ARRAY_BITS_32 = (byte) (0x5 << 5);

	/**
	 * Flag in the header set when the length of the array can be encoded in the
	 * header (less than {@link #ARRAY_MAX_TINY_LENGTH}).
	 */
	private static final int ARRAY_FLAG_TINY_LENGTH = 1 << 4;

	/** Maximum length that can be encoded in the header. */
	private static final int ARRAY_MAX_TINY_LENGTH = 0xf;// 4 bits

	/**
	 * Resource buffer where the array is
	 */
	private final ResourceBuffer buf;

	/**
	 * First element address in {@link ResourceBuffer}
	 */
	private final int startAddress;

	/**
	 * Number of elements (not number of bytes)
	 */
	private final int size; // 0 init

	/**
	 * Value in #ARRAY_BITS_MASK
	 */
	private final byte elementFormat;

	/**
	 * Pointer increment (4, 2 or1)
	 */
	private final byte elementFactor;

	/**
	 * Creates a new {@link ResourceArray} and updates the given {@link ResourceBuffer}
	 * positions.
	 */
	/* package visibility */
	ResourceArray(ResourceBuffer buf) throws IOException {

		int header = buf.readUnsignedByte();

		// Read the length
		int lengthInHeader = header & ARRAY_MAX_TINY_LENGTH;
		int fullLength;
		if ((header & ARRAY_FLAG_TINY_LENGTH) != 0) {
			fullLength = lengthInHeader;
		} else {
			int lengthLowestBits = buf.readUnsignedByte();
			fullLength = (lengthInHeader << 8) | lengthLowestBits;
		}

		// mark first element address
		this.size = fullLength;
		this.elementFormat = (byte) (header & ARRAY_BITS_MASK);
		this.buf = buf;

		switch (elementFormat) {
			case ARRAY_BITS_U8:
			case ARRAY_BITS_8:
				this.elementFactor = 1;
				break;
			case ARRAY_BITS_U16:
			case ARRAY_BITS_16:
				this.elementFactor = 2;
				break;
			case ARRAY_BITS_U32:
			case ARRAY_BITS_32:
				this.elementFactor = 4;
				break;
			default:
				// this is not a resource array!
				throw new IOException();
		}

		// adjust first element address adding potential header padding
		// (first element is aligned on its size)
		int bufferPtr = buf.getCurrentPtr();
		this.startAddress = ResourceBuffer.align(bufferPtr, elementFactor);
		try {
			// Point right after the end of the array
			buf.setPointer(startAddress + size * elementFactor, true);
		}
		catch(IndexOutOfBoundsException e) {
			throw new EOFException();
		}
	}

	public ResourceBuffer getBuffer() {
		return buf;
	}

	public int length() {
		return size;
	}

	public int elementAt(int index) throws IOException {
		if (index < 0 || index >= size) {
			throw new IndexOutOfBoundsException();
		}
		buf.setPointer(startAddress + index * elementFactor, false);
		return elementAtCurrentIndex(elementFormat);
	}

	public int[] elements() throws IOException {
		buf.setPointer(startAddress, false);
		int elementFormat = this.elementFormat;
		int l = size;
		int[] ret = new int[l];
		for (int i = -1; ++i < l;) {
			ret[i] = elementAtCurrentIndex(elementFormat);
		}
		return ret;
	}

	public ResourceBuffer seekToElementPointer(int index) throws IOException {
		buf.seek(elementAt(index));
		return buf;
	}

	private int elementAtCurrentIndex(int elementFormat) throws IOException {
		switch (elementFormat) {
			case ARRAY_BITS_U8:
				return buf.readUnsignedByte();
			case ARRAY_BITS_U16:
				return buf.platformReadUInt16();
			case ARRAY_BITS_U32:
				return buf.platformReadInt32();
			case ARRAY_BITS_8:
				return buf.readSignedByte();
			case ARRAY_BITS_16:
				return buf.platformReadInt16();
			case ARRAY_BITS_32:
				return buf.platformReadInt32();
			default:
				// this is not a resource array!
				throw new IOException();
		}
	}
}
