/**
 * Java
 *
 * Copyright 2008-2014 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.bon;

import java.util.Hashtable;
import java.util.NoSuchElementException;

import com.is2t.bon.ImmutablesNatives;

/**
	 * This class gives access to the global immutable objects pool.<br>
	 * <br>
	 * The method <code>at(String id)</code> allows an object to be retrieved from the pool
	 * using an ID string.<br>
	 * The pool is made of several erasable blocks, potentially 0. The number of
	 * these blocks is system dependent.
	 */
public class Immutables {

	/**
	 * <p>
	 * Retrieves the object that match the the given ID in the immutable objects pool.
	 * </p>
	 * <p>
	 * If no object can be found with such ID, a {@link NoSuchElementException} is thrown.<br>
	 * </p>
	 *
	 * @param ID
	 *            the ID of the immutable object to get
	 * @return the immutable object matching the ID
	 * @throws NoSuchElementException
	 *            if the ID is not found
	 * @throws ImmutablesError
	 * 			  if an internal error occurred during immutable access
	 */
	// IMPLEMENTATION NOTE: VM knows the number of calls from this API to native method (2 calls)
    // Changing the number of calls need to be reported to NativePool implementation of com_is2t_bon_ImmutablesNatives_get
	public static Object get(String id)
	{
		try{ return ImmutablesNatives.get(id); }
		catch(IndexOutOfBoundsException e){
			throw new NoSuchElementException(id);
		}
	}

	/**
	 * @param ID
	 *            of immutable object to be put
	 * @param o
	 *            the object to set immutable
	 * @throws NullPointerException
	 *             if given object is null
	 * @throws OutOfMemoryError
	 *             if there is no more space in the immutable memory
	 */
	public static void put(String ID, Object o) {
		throw new OutOfMemoryError();	// TODO
	}

	/**
	 * TODO
	 *
	 * @param table
	 *            the table that contains the objects to set immutables
	 * @throws NullPointerException
	 *             if given table is null
	 * @throws ClassCastException
	 *             if a key is not a {@link String}
	 * @throws OutOfMemoryError
	 *             if there is no more space in the immutable memory
	 * @see #freeMemory()
	 */
	public static void putAll(Hashtable table) {
		throw new OutOfMemoryError();	// TODO
	}
	
	/**
	 * Return an array with all the IDs of the immutable objects.
	 * 
	 * @see #at(String)
	 * @return an String array.
	 */
	// IMPLEMENTATION NOTE: VM knows the number of calls from this API to native method (2 calls)
    // Changing the number of calls need to be reported to NativePool implementation of com_is2t_bon_ImmutablesNatives_getNbImmutableKeys/com_is2t_bon_ImmutablesNatives_fillImmutableKeys
	public static String[] allIDs()
	{
		String[] allIDs = new String[ImmutablesNatives.getNbImmutableKeys()];
		ImmutablesNatives.fillImmutableKeys(allIDs);
		return allIDs;
	}

	public static boolean isImmutable(Object o){
		return ImmutablesNatives.isImmutable(o);
	}


	public static long freeMemory(){
		return 0;//TODO
	}

	/**
	 * Returns the amount of immutable memory used by the system.
	 */
	public static long totalMemory(){
		return ImmutablesNatives.initialImmutableMemory();//TODO
	}

}