/*
 * Java
 *
 * Copyright 2008-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.bon;

import com.is2t.cldc.support.ObjectGraphBrowser;

/**
 * An immortal object has two major properties:<br>
 * (1) it is not managed by the garbage collector<br>
 * (2) it does not move around in memory, i.e. its physical memory location remains the same forever.
 */
public class Immortals {

	/**
	 * Gets whether an object is immortal or not. An object is immortal:
	 * <ul>
	 * <li>if it has been declared as immortal calling {@link #setImmortal(Object)},</li>
	 * <li>if it is immutable (see {@link Immutables}).</li>
	 * </ul>
	 *
	 * @param o
	 *            the queried object
	 * @return <code>true</code> if the queried object is immortal, or immutable. <code>false</code> otherwise.
	 * @throws NullPointerException
	 *             if given object is null
	 */
	native public static boolean isImmortal(Object o);

	/**
	 * Returns the amount of immortal memory used by the system.
	 */
	public native static long totalMemory();

	/**
	 * Returns the amount of free immortal memory still available.
	 */
	public native static long freeMemory();

	/**
	 * Returns the received object and turns it into an immortal object.<br>
	 * If the received object was an immutable, it remains immutable.<br>
	 * If the object was already an immortal, it remains immortal.<br>
	 * If the object <code>o</code> was a reclaimable object, it
	 * turns into an immortal object. Upon success, the returned object is immortal, otherwise
	 * an OutOfMemoryError is thrown<br>
	 * <br>
	 * <code> Object obj1 = new Object();</code><br>
	 * <code> Object obj2 = Immortal.immortal(obj1);</code><br>
	 * <code> obj1 == obj2 </code> is always true.<br>
	 * <br>
	 * @return the received object.
	 */
	public native static <T> T setImmortal(T o);


	public static <T> T deepImmortal(T root)
	{
		//implementation is quite costly... but simple :)
		ObjectGraphBrowser graphBrowser = new ObjectGraphBrowser() {

			@Override
			public void browse(Object o) {
				setImmortal(o);
			}
		};

		graphBrowser.browseGraph(root);

		return root;
	}


	public static void run(Runnable runnable)
	{
		try
		{
			switchAllocationIntoImmortals(true);
			runnable.run();
		}
		finally{ switchAllocationIntoImmortals(false); }
	}


	private static native void switchAllocationIntoImmortals(boolean on);

}