/*
 * Java
 *
 * Copyright 2018-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.bon.timer;

import java.io.PrintStream;

import ej.bon.Timer;
import ej.bon.TimerTask;
import ej.kf.Feature;
import ej.kf.Feature.State;
import ej.kf.FeatureStateListener;
import ej.kf.Kernel;
import ej.kf.Module;

/**
 * Timer behavior when KF is enabled. A Timer instance owned by the Kernel:
 * <li>can manage TimerTask owned by apps. TimerTaskRef instances are allocated
 * by the Kernel, even if the {@link TimerTask} is added by a
 * {@link Feature}.</li>
 * <li>Timer#cancel() called by a Feature only disconnects {@link TimerTask}
 * owned by this {@link Feature}.</li>
 */
public class TimerTaskListKernel extends TimerTaskList implements FeatureStateListener {

	public TimerTaskListKernel() {
		Kernel.addFeatureStateListener(this);
	}

	/**
	 * This method can be called by both Kernel or Feature contexts.
	 */
	@Override
	protected boolean isAccessible(TimerTaskRef taskRef) {
		Module caller = Kernel.getContextOwner();
		Kernel.enter();
		Kernel kernel = Kernel.getInstance();
		if (Kernel.getOwner(this) == kernel && caller != kernel) {
			// List(K) & Caller(F) => TimerTask(F)
			return Kernel.getOwner(taskRef.task) == caller;
		} else {
			// List(K) & Caller(K) => true
			// List(F) & Caller(F) => true
			// List(F) & Caller(K) => true
			return true;
		}
	}

	@Override
	public boolean stopTimerOnCancel() {
		// Timer is stopped if caller is the Kernel or if the Timer is owned by the
		// Feature. Otherwise, caller is a Feature and Timer is a Kernel timer: cancel
		// does not stop the Timer.
		Kernel kernel = Kernel.getInstance();
		Module caller = Kernel.getContextOwner();
		boolean fullCancel = caller == kernel || caller == Kernel.getOwner(this);
		if (fullCancel) {
			Kernel.removeFeatureStateListener(this);
		}
		return fullCancel;
	}

	@Override
	public TimerTaskRef newTimerTaskRef(TimerTask task) {
		// Whatever the caller context, the TimerTaskRef instance is owned by the
		// Kernel.
		Kernel.enter();
		return super.newTimerTaskRef(task);
	}

	@Override
	public void addTaskToScheduledTasks(TimerTaskRef taskRef, long absoluteTime) {
		// Modification of the list must be done under Kernel context.
		Kernel.enter();
		super.addTaskToScheduledTasks(taskRef, absoluteTime);
	}

	@Override
	public void removeReferenceTo(TimerTask task) {
		// Modification of the list must be done under Kernel context.
		Kernel.enter();
		super.removeReferenceTo(task);
	}

	@Override
	public void stateChanged(Feature feature, State previousState) {
		if (feature.getState() == State.STOPPED) {
			synchronized (this) {
				// remove all TimerTask references owned by this Feature
				TimerTaskRef taskRef = this.nextTaskRef;
				TimerTaskRef previousTaskRef = null;
				while (taskRef != null) {
					TimerTaskRef nextTask = taskRef.next;
					if (Kernel.getOwner(taskRef.task) == feature) {
						if (previousTaskRef == null) {
							this.nextTaskRef = nextTask;
						} else {
							previousTaskRef.next = nextTask;
						}
					}
					previousTaskRef = taskRef;
					taskRef = nextTask;
				}
			}
		}
	}

	@Override
	public void uncaughtException(Timer timer, TimerTask timerTask, Throwable e) {
		// Timer owned by the Kernel => uncaught exception handlers are called in Kernel
		// context
		Kernel.enter();
		super.uncaughtException(timer, timerTask, e);
	}

	@Override
	protected void dump(PrintStream s, TimerTaskRef taskRef, long platformTimeMillis, boolean isLaunched, boolean isRunning) {
		super.dump(s, taskRef, platformTimeMillis, isLaunched, isRunning);

		s.print("   - Owner: ");
		Module owner = Kernel.getOwner(taskRef.task);
		String ownerName;
		if(owner == Kernel.getInstance()){
			ownerName = "Kernel";
		}
		else {
			ownerName = owner.getName();
		}
		s.println(ownerName);
	}
}
