/*
 * Java
 *
 * Copyright 2018-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.bon.timer;

import java.io.PrintStream;

import ej.bon.CurrentTime;
import ej.bon.Timer;
import ej.bon.TimerTask;
import ej.bon.Util;

/**
 * Implementation of the linked list of {@link TimerTask} within a {@link Timer}
 * Synchronization is done on this object instance, outside this class (in the
 * {@link Timer} code).
 */
public class TimerTaskList {

	/**
	 * set to true when cancel() method is called
	 */
	public boolean isCanceled;

	/**
	 * ordered linked list of TimerTaskRef instances
	 */
	public TimerTaskRef nextTaskRef;

	/**
	 * Linked list of {@link TimerTaskRef} to launch.<p>
	 * This list is modified only by the {@link Timer} thread but
	 * can be read by any thread in {@link Timer#dump(PrintStream)}.
	 */
	public TimerTaskRef tasksToLaunch;

	/**
	 * @return true if the Timer must be stopped when {@link Timer#cancel()} is
	 *         called, false otherwise.
	 */
	public boolean stopTimerOnCancel() {
		return true;
	}

	/**
	 * Clear the list (remove all references to TimerTask and links between tasks).
	 *
	 * @return true if the Timer must be stopped, false otherwise.
	 */
	public synchronized void cancel() {
		// stop the pump and remove all links to tasks and other objects to allow GC
		notify();

		TimerTaskRef taskRef = this.nextTaskRef;
		TimerTaskRef previousTaskRef = null;
		while (taskRef != null) {
			TimerTaskRef nextTaskRef = taskRef.next;
			if (isAccessible(taskRef)) {
				if (previousTaskRef == null) {
					this.nextTaskRef = nextTaskRef;
				} else {
					previousTaskRef.next = nextTaskRef;
				}
			}
			previousTaskRef = taskRef;
			taskRef = nextTaskRef;
		}

		this.isCanceled = stopTimerOnCancel();
	}

	/**
	 * Tells whether the given task can be accessed or not. The default behavior of this
	 * implementation returns <code>true</code>.
	 *
	 * @return true if this {@link TimerTaskRef} can be accessed, false
	 *         otherwise.
	 */
	protected boolean isAccessible(TimerTaskRef taskRef) {
		return true;
	}

	/**
	 * Create a new {@link TimerTaskRef} wrapper for a {@link TimerTask}.
	 *
	 * @param task
	 *            the task to wrap
	 * @return the {@link TimerTaskRef} that wraps the given
	 */
	public TimerTaskRef newTimerTaskRef(TimerTask task) {
		return new TimerTaskRef(task);
	}

	/**
	 * Add the task in nextTask linked list. This method is not thread safe. Caller
	 * should be synchronized on this before calling this method.
	 *
	 * @param taskRef
	 * @param absoluteTime
	 *            should be equals to task.absoluteTime.
	 */
	public void addTaskToScheduledTasks(TimerTaskRef taskRef, long absoluteTime) {

		TimerTaskRef previousTask = null;
		TimerTaskRef nextTask = this.nextTaskRef;
		// find previous and next task
		while (nextTask != null && nextTask.task.absoluteTime < absoluteTime) {
			previousTask = nextTask;
			nextTask = nextTask.next;
		}

		// add it to the list
		taskRef.next = nextTask;

		if (previousTask != null) {
			previousTask.next = taskRef;
		} else {
			// timer next task is updated -> wake up pump
			this.nextTaskRef = taskRef;
			this.notify();
		}
	}

	/**
	 * Remove the given task from the tasks list if it is canceled.<br>
	 * The timer keeps a linked list of the {@link TimerTask}, when a
	 * {@link TimerTask} is canceled, we need to remove the reference to the task.If
	 * there are no external references to the task, it becomes eligible for garbage
	 * collection.
	 *
	 * @param task
	 *            the task to remove
	 */
	public void removeReferenceTo(final TimerTask task) {
		synchronized (this) {
			TimerTaskRef previousTaskRef = null;
			TimerTaskRef currentTaskRef = this.nextTaskRef;

			search: {
				while (currentTaskRef != null) {
					if (task == currentTaskRef.task) {
						break search;
					}
					previousTaskRef = currentTaskRef;
					currentTaskRef = currentTaskRef.next;
				}

				// task not found
				return;
			}

			// task found
			if (task.isCanceled) {
				if (previousTaskRef != null) {
					previousTaskRef.next = currentTaskRef.next;
				} else {
					// this is the first task in the list
					this.nextTaskRef = currentTaskRef.next;
				}
				currentTaskRef.next = null;
			}
		}
	}

	/**
	 * Schedule the task if it is repeatable. Otherwise, remove it from all linked
	 * list. This method is thread safe.
	 * <p>
	 * This method updates {@link #tasksToLaunch} list.
	 *
	 *
	 * @param taskRef
	 *            the task the re-schedule
	 * @param previousExecutionTime
	 *            value depends on fixedRate attribute of the task:
	 *            <li>if the task HAS NOT a fixed rate : it is the computed
	 *            scheduled execution time of the previous launch</li>
	 *            <li>if the task HAS a fixed rate : it is the actual scheduled
	 *            execution time of the previous launch</li>
	 *
	 */
	public void reschedule(TimerTaskRef taskRef, long previousExecutionTime) {
		TimerTask task = taskRef.task;
		long period = task.period;

		// Synchronize here to be sure to always synchronize when rescheduling a task.
		// Rescheduling a task consists in removing the given task from the launched
		// tasks list and to add it to the scheduled tasks list if it is a repeatable
		// task.
		// Timer.dump() is synchronized on "this" and may be browsing the launched tasks
		// list.
		synchronized (this) {

			this.tasksToLaunch = taskRef.next;

			if (period != 0) {
				// task is repeatable
				// check if timer is not canceled
				if (!this.isCanceled) {
					// here the task may have been canceled,
					// it will be scheduled anyway but never launch. It is not a problem.

					// compute its next absolutetime
					long absoluteTime = previousExecutionTime + period;
					task.absoluteTime = absoluteTime;// update task absolute time
					addTaskToScheduledTasks(taskRef, absoluteTime);
				}
			} else {
				// task is not repeatable, remove all its links to other task or timer
				taskRef.next = null;
				taskRef.task = null; // unlink before returning to main loop to ensure that the TimerTask is no more
				// touched from the stack.
			}
		}
	}


	/**
	 * Launch all the tasks that have been added to the {@link #tasksToLaunch} list.
	 */
	public void runLaunchedTasks(Timer timer) {

		//launch all tasks to launch and reschedule it
		//no need to be synchronized here, tasks to launch cannot be added to
		//a timer (i.e. their "next" field and "tasksToLaunch" cannot be modified by another thread).
		TimerTaskRef taskToLaunchRef;
		while ((taskToLaunchRef=this.tasksToLaunch) != null && !isCanceled) {

			TimerTask taskToLaunch = taskToLaunchRef.task;
			if (!taskToLaunch.isCanceled) { // take a snapshot - no need of synchronization
				taskToLaunch.hasRun = true;
				long executionTime = taskToLaunch.fixedRate ? taskToLaunch.absoluteTime : CurrentTime.get(true);
				try {
					taskToLaunch.run();
				} catch (Throwable e) {
					try {
						taskToLaunch.uncaughtException(timer, e);
					} catch (Throwable e2) {
						// nothing to do : ignore all errors
					}
				}

				// schedule the task if it is repeatable (check is done in the timer)
				// and update tasksToLaunch list
				reschedule(taskToLaunchRef, executionTime);
			} else {
				// Task has been canceled: remove it from the lists and update tasksToLaunch list.
				// Synchronized to prevent modification of tasksToLaunch list while the dumper is
				// browsing the list.
				synchronized (this) {
					this.tasksToLaunch = taskToLaunchRef.next;
					taskToLaunchRef.next = null;
				}
			}
		}
	}

	public void uncaughtException(Timer timer, TimerTask timerTask, Throwable e) {
		Thread.UncaughtExceptionHandler h = timer.uncaughtExceptionHandler;
		if (h == null) {
			h = Timer.defaultUncaughtExceptionHandler;
			if (h == null) {
				// see javadoc step '3' (default behavior)
				timerTask.cancel();
				e.printStackTrace();
			} else {
				// see javadoc step '2'
				h.uncaughtException(Thread.currentThread(), e);
			}
		} else {
			// see javadoc step '1'
			h.uncaughtException(Thread.currentThread(), e);
		}
	}

	public synchronized void dump(PrintStream s){

		if(isCanceled){
			s.print(" - Timer canceled");
			return;
		}

		long platformTimeMillis = Util.platformTimeMillis();

		// Dump launched tasks
		TimerTaskRef taskRef = this.tasksToLaunch;
		boolean firstTask = true;
		while(taskRef != null){
			if (isAccessible(taskRef)) {
				dump(s, taskRef, platformTimeMillis, true, firstTask);
			}
			firstTask = false;
			taskRef = taskRef.next;
		}

		// Dump scheduled tasks
		taskRef = this.nextTaskRef;
		while(taskRef != null){
			if (isAccessible(taskRef)) {
				dump(s, taskRef, platformTimeMillis, false, false);
			}
			taskRef = taskRef.next;
		}

	}

	/**
	 * Prints a status of the given task to the specified print stream. This method is
	 * used only for debugging.
	 * 
	 * @param s
	 *        {@code PrintStream} to use for output.
	 * @param taskRef
	 *        the task to dump.
	 * @param platformTimeMillis
	 *        current platform time in milliseconds.
	 * @param isLaunched
	 * 		  true if the task is launched, false if it is waiting.
	 * 
	 * @see Timer#dump(PrintStream)
	 */
	protected void dump(PrintStream s, TimerTaskRef taskRef, long platformTimeMillis, boolean isLaunched, boolean isRunning){
		TimerTask task = taskRef.task;
		s.print(" - Task ");
		s.println(task.toString());

		s.print("   - Status: ");
		String status;
		if(isRunning){
			status = "running";
		}
		else {
			status = isLaunched ? "launched" : "waiting";
		}
		s.println(status);

		if(isLaunched){
			s.print("   - Launched ");
			s.print(platformTimeMillis - task.absoluteTime);
			s.println(" ms ago");
		}
		else {
			s.print("   - Next schedule in ");
			s.print(task.absoluteTime - platformTimeMillis);
			s.println(" ms");
		}

		if(task.period != 0){
			s.print("   - Period: ");
			s.print(task.period);
			s.print(" ms at fixed-");
			s.println(task.fixedRate ? "rate" : "delay");
		}
		else {
			s.println("   - One shot");
		}
	}
}
