/*
 * Java
 *
 * Copyright 2013-2020 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.util;

import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;

import ej.annotation.NonNull;
import ej.annotation.NonNullByDefault;
import ej.annotation.Nullable;

/*
 * In WeakHashMap, a key may be null. To handle this case, keys are wrapped
 * into a special object: WeakKey (also used to ensure that objects no longer
 * in use are garbaged). For the key "null", use NullKey class.
 *
 */
@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
public class WeakHashMap<K, V> extends AbstractHashMap<K, V> {
	
	private static class WeakKey<T> extends WeakReference<T> {

		int hashCode;

		@SuppressWarnings("null")// When null analysis is not enabled, implementation allows passing a null reference
		public WeakKey(@NonNull T key, ReferenceQueue<T> q) {
			super(key, q);
			this.hashCode = (key == null ? 0 : key.hashCode());
		}

		@Override
		public int hashCode() {
			return this.hashCode;
		}

		@Override
		public boolean equals(Object obj) {
			Object originalKey = this.get();
			if (originalKey == null) {
				// original reference is garbage collected
				return obj == this;
			} else {
				// call the equals method on the given object because it can
				// be another weak key which will compare its referenced
				// object with originalKey
				return obj.equals(originalKey);
			}
		}

	}

	private static class NullKey<T> extends WeakKey<T> {

		@SuppressWarnings("null") // When null analysis is not enabled, implementation of WeakKey allows passing a null reference
		public NullKey() {
			super(null, null);
		}

		// No need to redefine hashCode() since it is the same
		// behavior as WeakKey.hashCode()
		@Override
		public boolean equals(Object obj) { //NOSONAR
			return obj == this;
		}
		
		@Override
		public int hashCode() {
			return 0;
		}

	}

	private static class WeakHashMapEntry<K, V> extends AbstractHashMapEntry<K, V> {

		WeakKey<K> weakKey;

		public WeakHashMapEntry(WeakKey<K> key, V value, AbstractHashMapEntry<K, V> next) {
			super(value, next);
			this.weakKey = key;
		}

		@Override
		public K getKey() {
			return this.weakKey.get();
		}

		@Override
		public Object getInternKey() {
			return this.weakKey;
		}

	}

	ReferenceQueue<K> queue;

	public WeakHashMap() {
		super();
	}

	public WeakHashMap(int initialCapacity) {
		super(initialCapacity);
	}

	public WeakHashMap(int initialCapacity, float loadFactor) {
		super(initialCapacity, loadFactor);
	}

	public WeakHashMap(Map<? extends K, ? extends V> m) {
		super(m);
	}
	
	@Override
	void init(){
		super.init();
		this.queue = new ReferenceQueue<>();
	}

	@Override
	public void clear() {
		// Clear the garbaged references
		this.clean();

		// empty the table
		super.clear();
	}

	@Override
	public boolean containsKey(Object key) {
		this.clean();
		return super.containsKey(key);
	}

	@Override
	public boolean containsValue(Object value) {
		this.clean();
		return super.containsValue(value);
	}

	@Override
	@Nullable 
	public V get(Object key) {
		this.clean();
		return super.get(key);
	}

	private void clean() {
		Reference<? extends K> ref = queue.poll();
		while(ref != null) {
			this.remove(ref);
			ref = queue.poll();
		}
	}

	@Override
	protected AbstractHashMapEntry<K, V> newHashEntry(K key, V value, AbstractHashMapEntry<K, V> entryPoint) {
		WeakKey<K> weakKey;
		if(key == null) {
			weakKey = new NullKey<K>();
		}
		else {
			weakKey = new WeakKey<K>(key, this.queue);
		}
		return new WeakHashMapEntry<K, V>(weakKey, value, entryPoint);
	}
}
