/*
 * Java
 *
 * Copyright 2013-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.util;

import ej.annotation.NonNullByDefault;
import ej.annotation.Nullable;

@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
public class Vector<E> extends AbstractArrayList<E> {
	
	private class VectorEnumeration implements Enumeration<E> {
		
		int position;
		
		public VectorEnumeration() {
			this.position = 0;
		}

		public boolean hasMoreElements() {
			return this.position < Vector.this.elementCount;
		}

		public E nextElement() {
			if(!this.hasMoreElements()) {
				throw new NoSuchElementException();
			}
			
			@SuppressWarnings("unchecked")
			E result = (E) Vector.this.elementData[this.position++];
			return result;
		}
		
	}

	public Vector() {
		super();
	}

	public Vector(Collection<? extends E> c) {
		super(c);
	}

	public Vector(int initialCapacity) {
		super(initialCapacity);
	}

	public Vector(int initialCapacity, int capacityIncrement) {
		super(initialCapacity, capacityIncrement);
	}

	@Override
	public synchronized boolean add(E e) {
		return super.add(e);
	}

	@Override
	public synchronized boolean addAll(Collection<? extends E> c) {
		return super.addAll(c);
	}

	@Override
	public synchronized boolean addAll(int index, Collection<? extends E> c) {
		return super.addAll(index, c);
	}

	public void addElement(E obj) {
		//No need to synchronize this method because add(E) is synchronized
		this.add(obj);
	}

	public int capacity() {
		return this.elementData.length;
	}

	// AbstractArrayList implements Cloneable
	// Sonar does not seem to handle this fact
	// And Vector.clone() is synchronized
	@Override
	public synchronized Object clone() { //NOSONAR
		return super.clone();
	}

	@Override
	public synchronized boolean containsAll(Collection<?> c) {
		return super.containsAll(c);
	}

	public synchronized void copyInto(Object[] anArray) {
		if(anArray.length < this.elementCount) {
			throw new IndexOutOfBoundsException();
		}
		super.toArray(anArray);
	}

	public E elementAt(int index) {
		//No need to synchronize this method because get(int) is synchronized
		return this.get(index);
	}

	public Enumeration<E> elements() {
		return new VectorEnumeration();
	}

	@Override
	public synchronized void ensureCapacity(int minCapacity) {
		super.ensureCapacity(minCapacity);
	}

	@Override
	public synchronized boolean equals(@Nullable Object o) {
		return super.equals(o);
	}

	public synchronized E firstElement() {
		if(this.elementCount == 0) {
			throw new NoSuchElementException();
		}
		@SuppressWarnings("unchecked")
		E result = (E) this.elementData[0];
		return result;
	}

	@Override
	public synchronized E get(int index) {
		try {
			return super.get(index);
		}
		catch(IndexOutOfBoundsException e) {
			throw new ArrayIndexOutOfBoundsException(index);
		}
	}

	@Override
	public synchronized int hashCode() {
		return super.hashCode();
	}

	@Override
	public synchronized int indexOf(Object o, int index) {
		return super.indexOf(o, index);
	}

	public synchronized void insertElementAt(E obj, int index) {
		try {
			this.add(index, obj);
		}
		catch(IndexOutOfBoundsException e) {
			throw new ArrayIndexOutOfBoundsException(index);
		}
	}

	public synchronized E lastElement() {
		final int elementCount = this.elementCount;
		if(elementCount == 0) {
			throw new NoSuchElementException();
		}
		
		@SuppressWarnings("unchecked")
		E result = (E) this.elementData[elementCount-1];
		return result;
	}

	@Override
	public synchronized int lastIndexOf(Object o) {
		return super.lastIndexOf(o);
	}

	@Override
	public synchronized int lastIndexOf(Object o, int index) {
		return super.lastIndexOf(o, index);
	}

	@Override
	public ListIterator<E> listIterator() {
		return this.listIterator(0);
	}

	@Override
	public synchronized ListIterator<E> listIterator(int index) {
		return super.listIterator(index);
	}

	@Override
	public synchronized E remove(int index) {
		try {
			return super.remove(index);
		}
		catch(IndexOutOfBoundsException e) {
			throw new ArrayIndexOutOfBoundsException(index);
		}
	}

	@Override
	public synchronized boolean removeAll(Collection<?> c) {
		return super.removeAll(c);
	}

	public synchronized void removeAllElements() {
		this.clear();
	}

	public synchronized boolean removeElement(Object obj) {
		return this.remove(obj);
	}

	public void removeElementAt(int index) {
		this.remove(index);
	}

	@Override
	protected synchronized void removeRange(int fromIndex, int toIndex) {
		super.removeRange(fromIndex, toIndex);
	}

	@Override
	public synchronized boolean retainAll(Collection<?> c) {
		return super.retainAll(c);
	}

	@Override
	public synchronized E set(int index, E element) {
		try {
			return super.set(index, element);
		}
		catch(IndexOutOfBoundsException e) {
			throw new ArrayIndexOutOfBoundsException(index);
		}
	}

	public void setElementAt(E obj, int index) {
		this.set(index, obj);
	}

	public synchronized void setSize(int newSize) {
		if(newSize < 0) {
			throw new ArrayIndexOutOfBoundsException(newSize);
		}
		
		super.ensureCapacity(newSize);//call super to avoid synchronized wrapper
		final int elementCount = this.elementCount;
		int theMin = Math.min(newSize, elementCount);
		int theMax = Math.max(newSize, elementCount);
		
		final Object[] elementData = this.elementData;
		for(int i = theMin; i < theMax; i++) {
			elementData[i] = null;
		}
		
		this.elementCount = newSize;
	}

	@Override
	public synchronized List<E> subList(int fromIndex, int toIndex) {
		return super.subList(fromIndex, toIndex);
	}

	@Override
	public synchronized Object[] toArray() {
		return super.toArray();
	}

	@Override
	public synchronized <T> T[] toArray(T[] a) {
		return super.toArray(a);
	}

	@Override
	public synchronized String toString() {
		return super.toString();
	}

	@Override
	public synchronized void trimToSize() {
		super.trimToSize();
	}
}