/*
 * Java
 *
 * Copyright 2013-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.util;

import java.util.AbstractHashMap.AbstractHashMapEntry;
import java.util.HashMap.HashMapEntry;

import ej.annotation.NonNullByDefault;
import ej.annotation.Nullable;

@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
public class Hashtable<K, V> extends Dictionary<K, V> implements Map<K, V>, Cloneable, java.io.Serializable {

	private static final int DEFAULT_INITIAL_CAPACITY = 11;
	private static final float DEFAULT_LOAD_FACTOR = 0.75f;
	

	/**
	 * {@link HashMapEntry} that does not accept null values.
	 */
	@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
	private static class HashtableEntry<K, V> extends HashMapEntry<K, V> implements Cloneable {
		
		/**
		 * @param key must be different from null
		 * @param value must be different from null
		 */
		public HashtableEntry(K key, V value, AbstractHashMapEntry<K, V> next) {
			super(key, value, next);
		}
		
		@Override
		@Nullable 
		public V setValue(V value) {
			if(value == null) {
				throw new NullPointerException(); // see Hashtable javadoc
			}
			return super.setValue(value);
		}
		
	}
	
	private static class HashMapForHashtable<K,V> extends HashMap<K, V>{
		
		public HashMapForHashtable(int initialCapacity, float loadFactor) {
			super(initialCapacity, loadFactor);
		}

		public HashMapForHashtable(Map<? extends K, ? extends V> m) {
			super(m);
		}
		
		@Override
		protected AbstractHashMapEntry<K, V> newHashEntry(K key, V value, AbstractHashMapEntry<K, V> next) {
			return new HashtableEntry<K, V>(key, value, next);
		}
	}
	
	private HashMapForHashtable<K, V> map;
	
	public Hashtable() {
		this(DEFAULT_INITIAL_CAPACITY, DEFAULT_LOAD_FACTOR);
	}

	public Hashtable(int initialCapacity) {
		this(initialCapacity, DEFAULT_LOAD_FACTOR);
	}

	public Hashtable(int initialCapacity, float loadFactor) {
		this.map = new HashMapForHashtable<>(initialCapacity, loadFactor);
	}

	public Hashtable(Map<? extends K, ? extends V> t) {
		this.map = new HashMapForHashtable<>(t);
		this.putAll(t);
	}

	public synchronized void clear() {
		this.map.clear();
	}

	@SuppressWarnings("unchecked")
	@Override
	public synchronized Object clone() {
		try {
			Hashtable<K, V> clone = (Hashtable<K, V>) super.clone();
			clone.map = (HashMapForHashtable<K, V>) clone.map.clone();
			
			return clone;
		
		} catch(Throwable e){
			// Should not happen since Hashtable implements Cloneable
			throw new InternalError(); //NOSONAR
		}
	}

	public synchronized boolean contains(Object value) {
		return containsValue(value);
	}

	public synchronized boolean containsKey(Object key) {
		if(key == null){
			throw new NullPointerException();
		}
		return map.containsKey(key);
	}

	public boolean containsValue(Object value) {
		if (value == null) {
			throw new NullPointerException();
		}
		return map.containsValue(value);
	}

	@Override
	public synchronized Enumeration<V> elements() {
		return this.map.new AbstractHashMapEnumeration<V>(HashMap.VALUES);
	}

	public Set<Map.Entry<K, V>> entrySet() {
		return this.map.entrySet();
	}

	@Override
	public synchronized boolean equals(@Nullable Object o) {
		if(o == this){
			return true;
		}
		return this.map.equals(o);
	}

	@Override
	@Nullable 
	public synchronized V get(Object key) {
		if(key == null){
			throw new NullPointerException();
		}
		return map.get(key);
	}

	@Override
	public synchronized int hashCode() {
		return map.hashCode();
	}

	@Override
	public synchronized boolean isEmpty() {
		return map.isEmpty();
	}

	@Override
	public synchronized Enumeration<K> keys() {
		return this.map.new AbstractHashMapEnumeration<K>(HashMap.KEYS);
	}

	public Set<K> keySet() {
		return map.keySet();
	}

	@Override
	@Nullable 
	public synchronized V put(K key, V value) {
		if(value == null || key == null) {
			throw new NullPointerException();
		}
		return map.put(key, value);
	}

	public synchronized void putAll(Map<? extends K, ? extends V> t) {
		//Don't use map.putAll because it will not check for null keys or values.
		for(Map.Entry<? extends K, ? extends V> entry : t.entrySet()) {
			this.put(entry.getKey(), entry.getValue());
		}
	}

	protected void rehash() {
		map.rehash();
	}

	@Override
	@Nullable 
	public synchronized V remove(Object key) {
		if(key == null){
			throw new NullPointerException();
		}
		return map.remove(key);
	}

	@Override
	public synchronized int size() {
		return map.size();
	}

	@Override
	public synchronized String toString() {
		return map.toString();
	}

	public Collection<V> values() {
		return map.values();
	}
}