/*
 * Java
 *
 * Copyright 2008-2020 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.util;

import java.io.Serializable;

import com.is2t.vm.support.util.GregorianCalendar;

import ej.annotation.Nullable;

public abstract class Calendar implements Serializable, Cloneable, Comparable<Calendar> {

	public static final int ALL_STYLES = 0;
	public static final int SHORT = 1;
	public static final int LONG = 2;

	public static final int AM = 0;
	public static final int PM = 1;
	public static final int AM_PM = 9;

	public static final int ERA = 0;
	public static final int YEAR = 1;
	public static final int MONTH = 2;
	public static final int WEEK_OF_YEAR = 3;
	public static final int WEEK_OF_MONTH = 4;
	public static final int DATE = 5;
	public static final int DAY_OF_MONTH = 5;
	public static final int DAY_OF_YEAR = 6;
	public static final int DAY_OF_WEEK = 7;
	public static final int DAY_OF_WEEK_IN_MONTH = 8;
	public static final int HOUR = 10;
	public static final int HOUR_OF_DAY = 11;
	public static final int MINUTE = 12;
	public static final int SECOND = 13;
	public static final int MILLISECOND = 14;

	public static final int ZONE_OFFSET = 15;
	public static final int DST_OFFSET = 16;
	public static final int FIELD_COUNT = 17;

	public static final int JANUARY = 0;
	public static final int FEBRUARY = 1;
	public static final int MARCH = 2;
	public static final int APRIL = 3;
	public static final int MAY = 4;
	public static final int JUNE = 5;
	public static final int JULY = 6;
	public static final int AUGUST = 7;
	public static final int SEPTEMBER = 8;
	public static final int OCTOBER = 9;
	public static final int NOVEMBER = 10;
	public static final int DECEMBER = 11;
	public static final int UNDECIMBER = 12;

	public static final int SUNDAY = 1;
	public static final int MONDAY = 2;
	public static final int TUESDAY = 3;
	public static final int WEDNESDAY = 4;
	public static final int THURSDAY = 5;
	public static final int FRIDAY = 6;
	public static final int SATURDAY = 7;

	public static Calendar getInstance() {
		return getInstance(TimeZone.getDefault());
	}

	public static Calendar getInstance(TimeZone zone) {
		Calendar instance = new GregorianCalendar();
    	instance.setTimeZone(zone);
    	return instance ;
	}

	private static final int UNDEFINED_VALUE = 0;

	protected static final int STATE_RESET_TIME = 0;
	protected static final int STATE_UNVALIDATED_FIELDS = 1;
	protected static final int STATE_SYNCHRONIZED = 2;

	protected static final int FIELD_UNSET = 0;
	protected static final int FIELD_INTERNALLY_SET = 1;
	protected static final int FIELD_EXTERNALLY_SET = 2;
	protected static final int FIRST_FIELD_EXTERNALLY_SET = 3;

	protected static final int RESOLUTION_UNDEFINED = -1;
	protected static final int RESOLUTION_YEAR_MONTH_DAY_OF_MONTH = 0;
	protected static final int RESOLUTION_YEAR_MONTH_WEEK_OF_MONTH_DAY_OF_WEEK = 1;
	protected static final int RESOLUTION_YEAR_MONTH_DAY_OF_WEEK_IN_MONTH_DAY_OF_WEEK = 2;
	protected static final int RESOLUTION_YEAR_DAY_OF_YEAR = 3;
	protected static final int RESOLUTION_YEAR_DAY_OF_WEEK_WEEK_OF_YEAR = 4;

	protected static final int RESOLUTION_HOUR_OF_DAY = 0;
	protected static final int RESOLUTION_AM_PM_HOUR = 1;

	public int[] fields;
	protected int[] setFields;
	@Nullable
	private int[] orderedSetFields;
	protected int setFieldsCounter;

	protected long time;
	@Nullable
	protected TimeZone timeZone;
	protected int firstDayOfTheWeek; // subclasses dependent
	protected int minimalDayInFirstWeek;
	private int state;
	protected boolean areFieldsSet; // API field. Value is true if and only if state==STATE_SYNCHRONIZED


	protected Calendar() {
		this.state = STATE_RESET_TIME;
		this.areFieldsSet = false;
		/* Values related to default locale */
		this.firstDayOfTheWeek = SUNDAY;
		this.minimalDayInFirstWeek = 1;

		this.fields = new int[FIELD_COUNT];
		this.setFields = new int[FIELD_COUNT];
		this.setFieldsCounter = FIRST_FIELD_EXTERNALLY_SET;
        this.setTimeInMillis(System.currentTimeMillis());
	}

	public boolean after(Object when) {
		// If you change this impl., change also before(Object) impl.

		// From javadoc : is equivalent to
		// - when is a Calendar
		// - this.compareTo(when) > 0
		if(!(when instanceof Calendar)) {
			return false;
		}

		// When is a Calendar for sure, safe cast
		return this.compareTo((Calendar)when) > 0;
	}

	public boolean before(Object when) {
		// If you change this impl., change also after(Object) impl.

		// From javadoc : is equivalent to
		// - when is a Calendar
		// - this.compareTo(when) < 0
		if(!(when instanceof Calendar)) {
			return false;
		}

		// When is a Calendar for sure, safe cast
		return this.compareTo((Calendar)when) < 0;
	}

	public final void clear() {
		// Reset all fields, and time to UNDEFINED_VALUE
		for(int i = 0; i < FIELD_COUNT; i++) {
			this.fields[i] = UNDEFINED_VALUE;
			this.setFields[i] = FIELD_UNSET;
			this.orderedSetFields = null;
		}
		this.state = STATE_UNVALIDATED_FIELDS;
		this.areFieldsSet = false;
	}

	public final void clear(int field) {
		try {
			this.fields[field] = UNDEFINED_VALUE;
			this.setFields[field] = FIELD_UNSET;
			this.state = STATE_UNVALIDATED_FIELDS;
			this.areFieldsSet = false;
			this.orderedSetFields = null;
		}
		catch(ArrayIndexOutOfBoundsException e) {
			// field is not valid (< 0 or > FIELD_COUNT)
			// Nothing is mentioned in Javadoc about this case
			// Fail silently
		}
	}

	@Override
	public Object clone() {
		try {
			Calendar clone = (Calendar) super.clone(); // safe cast
			clone.fields = this.fields.clone();
			clone.setFields = this.setFields.clone();
			clone.orderedSetFields = null;

			return clone;
		}
		catch(Throwable e) {
			// Will not happen since Calendar implements Cloneable
			throw new InternalError();
		}
	}

	@Override
	public int compareTo(Calendar anotherCalendar) {
		// Must force time update for this and anotherCalendar (due to lazy init.)
		// Comparing long and return int value, so we cannot substract long value
		// because of invalid value due to int cast
		long thisTime = this.getTimeInMillis();
		long anotherTime = anotherCalendar.getTimeInMillis(); // will throw NullPointerException as specified in Javadoc
		if(thisTime == anotherTime) {
			return 0;
		}
		else if(thisTime > anotherTime) {
			return 1;
		}
		else {
			return -1;
		}
	}

	protected void complete() {
		if(this.state == STATE_RESET_TIME) {
			this.computeTime();
		}
		this.computeFields();
	}

	/**
	 * Get an ordered list of field initialization (from newer to older)
	 * @return ordered list of field indexes
	 */
	private int[] orderedSetFields() {
		int[] orderedSetFields = this.orderedSetFields;
		if (orderedSetFields != null) {
			return orderedSetFields;
		}

		final int[] setFields = this.setFields;
		int setFieldsLength = setFields.length;
		int resultLength = 0;

		/* Count number of set elements */
		for (int i = setFieldsLength; --i >= 0;) {
			if (setFields[i] >= FIRST_FIELD_EXTERNALLY_SET) {
				resultLength++;
			}
		}

		if (resultLength == 0) {
			return new int[0];
		}

		/* Make a copy of set elements order and prepare result array */
		int setFieldsClone[] = new int[resultLength];
		int setFieldsIndex[] = new int[resultLength];
		int resultIndex = 0;
		for (int i = 0; i < setFieldsLength; i++) {
			if (setFields[i] >= FIRST_FIELD_EXTERNALLY_SET) {
				setFieldsClone[resultIndex] = setFields[i];
				setFieldsIndex[resultIndex] = i;
				resultIndex++;
			}
		}

		/* Sort both setFieldsClone reverse, and sort setFieldsIndex in the same order */
		// Insertion sort
		for (int i =  1; i < resultLength; i++) {
	          for (int j = i; j > 0 && setFieldsClone[j - 1] < setFieldsClone[j]; j--) {
	        	  // swap setFieldsClone
	        	  int c = setFieldsClone[j];
	        	  setFieldsClone[j] = setFieldsClone[j-1];
	        	  setFieldsClone[j-1] = c;
	        	  // swap setFieldsIndexes
	        	  c = setFieldsIndex[j];
	        	  setFieldsIndex[j] = setFieldsIndex[j-1];
	        	  setFieldsIndex[j-1] = c;
	          }
		}

		this.orderedSetFields = setFieldsIndex;

		return setFieldsIndex;
	}

	private int dateResolutionMethodIntern(boolean complete, int[] orderedSetFields) {
		/*
		 Resolution for the date fields:
		     YEAR + MONTH + DAY_OF_MONTH
		     YEAR + MONTH + WEEK_OF_MONTH + DAY_OF_WEEK
		     YEAR + MONTH + DAY_OF_WEEK_IN_MONTH + DAY_OF_WEEK
		     YEAR + DAY_OF_YEAR
		     YEAR + DAY_OF_WEEK + WEEK_OF_YEAR
	     */
		int orderedSetFieldsLength = orderedSetFields.length;
		
		final int[] setFields = this.setFields;
		boolean yearSet = setFields[YEAR] != FIELD_UNSET;
		boolean monthSet = setFields[MONTH] != FIELD_UNSET;
		boolean dayOfWeekSet = setFields[DAY_OF_WEEK] != FIELD_UNSET;
		
		for (int i = 0; i < orderedSetFieldsLength; i++) {
			int field = orderedSetFields[i];
			switch(field) {
			case DAY_OF_MONTH:
				if ((yearSet && monthSet) || !complete) {
					return RESOLUTION_YEAR_MONTH_DAY_OF_MONTH;
				}
				break;
			case WEEK_OF_MONTH:
				if ((yearSet && monthSet && dayOfWeekSet) || !complete) {
					return RESOLUTION_YEAR_MONTH_WEEK_OF_MONTH_DAY_OF_WEEK;
				}
				break;
			case DAY_OF_YEAR:
				if (yearSet || !complete) {
					return RESOLUTION_YEAR_DAY_OF_YEAR;
				}
				break;
			case WEEK_OF_YEAR:
				if ((yearSet && dayOfWeekSet) || !complete) {
					return RESOLUTION_YEAR_DAY_OF_WEEK_WEEK_OF_YEAR;
				}
				break;
			case DAY_OF_WEEK_IN_MONTH:
				if ((yearSet && monthSet && dayOfWeekSet) || !complete) {
					return RESOLUTION_YEAR_MONTH_DAY_OF_WEEK_IN_MONTH_DAY_OF_WEEK;
				}
				break;
			case DAY_OF_WEEK:
				/* Search which one of WEEK_OF_MONTH, DAY_OF_WEEK_IN_MONTH or WEEK_OF_YEAR was set first */
				for (int j = 1; j < orderedSetFieldsLength; j++) {
					int field2 = orderedSetFields[j];
					switch(field2) {
					case WEEK_OF_MONTH:
						if ((yearSet && monthSet) || !complete) {
							return RESOLUTION_YEAR_MONTH_WEEK_OF_MONTH_DAY_OF_WEEK;
						}
						break;
					case DAY_OF_WEEK_IN_MONTH:
						if ((yearSet && monthSet) || !complete) {
							return RESOLUTION_YEAR_MONTH_DAY_OF_WEEK_IN_MONTH_DAY_OF_WEEK;
						}
						break;
					case WEEK_OF_YEAR:
						if (yearSet || !complete) {
							return RESOLUTION_YEAR_DAY_OF_WEEK_WEEK_OF_YEAR;
						}
						break;
					}
				}

				/* No method found, if incomplete return a default one matching as much set fields as possible */
				if (!complete) {
					if (monthSet) {
						return RESOLUTION_YEAR_MONTH_WEEK_OF_MONTH_DAY_OF_WEEK;
					} else {
						return RESOLUTION_YEAR_DAY_OF_WEEK_WEEK_OF_YEAR;
					}
				}
				break;
			default:
				break;
			}
		}

		if (!complete) {
			/* Still no method found, pick a default one */
			if (monthSet) {
				return RESOLUTION_YEAR_MONTH_DAY_OF_MONTH;
			} else {
				return RESOLUTION_YEAR_DAY_OF_YEAR;
			}
		}

		return RESOLUTION_UNDEFINED;
	}

	protected int dateResolutionMethod() {
		int[] orderedSetFields = orderedSetFields();

		/* Search for a complete resolution method */
		int resolution = dateResolutionMethodIntern(true, orderedSetFields);

		if (resolution != RESOLUTION_UNDEFINED) {
			return resolution;
		}

		/* Then search for an incomplete resolution method */
		return dateResolutionMethodIntern(false, orderedSetFields);
	}

	private int timeResolutionMethodIntern(boolean complete) {
		int[] orderedSetFields = orderedSetFields();

		/*
		 Resolution for the time of day fields:
		     HOUR_OF_DAY
		     AM_PM + HOUR

		 */
		for (int field : orderedSetFields) {
			switch(field) {
			case HOUR_OF_DAY:
				return RESOLUTION_HOUR_OF_DAY;
			case AM_PM:
				if (isSet(HOUR) || !complete) {
					return RESOLUTION_AM_PM_HOUR;
				}
				break;
			case HOUR:
				if (isSet(AM_PM) || !complete) {
					return RESOLUTION_AM_PM_HOUR;
				}
				break;
			}
		}

		return RESOLUTION_UNDEFINED;
	}

	protected int timeResolutionMethod() {
		/* Search for a complete resolution method */
		int resolution = timeResolutionMethodIntern(true);

		if (resolution != RESOLUTION_UNDEFINED) {
			return resolution;
		}

		/* Then search for an incomplete resolution method */
		resolution = timeResolutionMethodIntern(false);

		if (resolution != RESOLUTION_UNDEFINED) {
			return resolution;
		}

		/* Default resolution method */
		return RESOLUTION_HOUR_OF_DAY;
	}

	protected abstract void computeFields();

	protected abstract void computeTime();

	@Override
	public boolean equals(@Nullable Object obj) {
		try {
			Calendar objCalendar = (Calendar) obj;
			TimeZone timeZone;
			return this.time == objCalendar.time
				&& this.getFirstDayOfWeek() == objCalendar.getFirstDayOfWeek()
			    && this.getMinimalDaysInFirstWeek() == objCalendar.getMinimalDaysInFirstWeek()
			    && ((timeZone=this.timeZone) == null ? objCalendar.timeZone == null : timeZone.equals(objCalendar.timeZone));
		}
		catch(NullPointerException | ClassCastException e) {
			return false;
		}
	}

	public int get(int field) {
		validateBeforeGet();

		// here state == STATE_SYNCHRONIZED for sure!
        return this.fields[field]; // will throw ArrayIndexOutOfBoundsException if necessary
	}

	public int getActualMaximum(int field) {
		// Steps :
		// - compare least maximum and real maximum for this field
		//     - if they are equal, return one of them
		//     - else, try to guess the maximum value of the field by setting
		//       it into a cloned calendar, and stop when maximum is reached
		//       (return maximum) or when the value is stabilized (return
		//       stabilized value)

		int leastMaximum = this.getLeastMaximum(field);
		int maximum = this.getMaximum(field);

		// step 1
		if(leastMaximum == maximum) {
			return maximum;
		}

		// handle specific case that return always the same result and take a while to compute
		switch (field) {
		case YEAR:
		case ZONE_OFFSET:
		case DST_OFFSET:
			return maximum;
		}

		// step 2

		// do not forgot to work on a cloned calendar to not destroy
		// the current one...

		Calendar clone = (Calendar) this.clone(); // safe cast
		clone.setLenient(true); // Set clone as lenient to allow permissive date interpretation
								 // Useless for now as the lenient field is not supported, so calendars are always lenient
		clone.get(Calendar.YEAR); // Get a random field to trigger a field computation on clone

		int stabilizedValue = leastMaximum;

		while(stabilizedValue <= maximum) {
			clone.set(field, stabilizedValue);
			if(clone.get(field) != stabilizedValue) {
				// we passed the maximum
				break;
			}
			else {
				// maximum not reached yet, continue!
				stabilizedValue++; // try a greater value
			}
		}

		return stabilizedValue - 1;
	}

	public int getActualMinimum(int field) {
		// Steps :
		// - same as getActualMaximum(int) but looking for the minimum

		int greatestMinimum = this.getGreatestMinimum(field);
		int minimum = this.getMinimum(field);

		// step 1
		if(greatestMinimum == minimum) {
			return minimum;
		}

		// step 2

		// do not forgot to work on a cloned calendar to not destroy
		// the current one...

		Calendar clone = (Calendar) this.clone(); // safe cast
		clone.setLenient(true);  // Set clone as lenient to allow permissive date interpretation
		 						  // Useless for now as the lenient field is not supported, so calendars are always lenient

		int stabilizedValue = greatestMinimum;

		while(stabilizedValue >= minimum) {
			clone.set(field, stabilizedValue);
			if(clone.get(field) != stabilizedValue) {
				// we passed the minimum
				break;
			}
			else {
				// minimum not reached yet, continue!
				stabilizedValue--; // try a lesser value
			}
		}

		return stabilizedValue + 1;
	}

	public int getFirstDayOfWeek() {
		return this.firstDayOfTheWeek;
	}

	public abstract int getGreatestMinimum(int field);

	public abstract int getLeastMaximum(int field);

	public abstract int getMaximum(int field);

	public int getMinimalDaysInFirstWeek() {
		return this.minimalDayInFirstWeek;
	}

	public abstract int getMinimum(int field);

	public final Date getTime() {
		return new Date(this.getTimeInMillis()) ;
	}

	public long getTimeInMillis() {
		validateBeforeGet();

    	// else if(state == STATE_SYNCHRONIZED || state == STATE_RESET_TIME) => NOP
    	return this.time;
	}

	public TimeZone getTimeZone() {
		TimeZone result = this.timeZone;
		assert result != null; // see #getInstance()
		return result;
	}

	@Override
	public int hashCode() {
		TimeZone timeZone;
		int otheritems = (this.firstDayOfTheWeek << 1) | (this.minimalDayInFirstWeek << 4) | ((timeZone = this.timeZone) != null ? timeZone.hashCode() << 7 : 0);
		long t = this.getTimeInMillis();
		return (int) t ^ (int) (t >> 32) ^ otheritems;
	}

	protected final int internalGet(int field) {
		return this.fields[field];
	}

	final void internalSet(int field, int value) {
		this.fields[field] = value;
	}

	public boolean isLenient() {
		return true;
	}

	public final boolean isSet(int field) {
		return this.setFields[field] != FIELD_UNSET;
	}

	protected boolean isExternallySet(int field) {
		return this.setFields[field] >= FIELD_EXTERNALLY_SET;
	}

	public void set(int field, int value) {
		if(state == STATE_RESET_TIME) {
    		// This is the first time a field is modified
			// => compute all fields before (lazy initialization)
    		computeFields();
        	setFields();
    	}

    	this.fields[field] = value;
    	int[] setFields = this.setFields;
    	if (setFields[field] < FIRST_FIELD_EXTERNALLY_SET ||
    			setFields[field] != setFieldsCounter - 1) {
	    	setFields[field] = setFieldsCounter;
	    	this.setFieldsCounter++;
	    	this.orderedSetFields = null;
    	}
    	this.state = STATE_UNVALIDATED_FIELDS;
		this.areFieldsSet = false;
	}

	public final void set(int year, int month, int date) {
		this.set(YEAR, year);
		this.set(MONTH, month);
		this.set(DATE, date);
	}

	public final void set(int year, int month, int date, int hourOfDay, int minute) {
		this.set(YEAR, year);
		this.set(MONTH, month);
		this.set(DATE, date);
		this.set(HOUR_OF_DAY, hourOfDay);
		this.set(MINUTE, minute);
	}

	public final void set(int year, int month, int date, int hourOfDay, int minute, int second) {
		this.set(YEAR, year);
		this.set(MONTH, month);
		this.set(DATE, date);
		this.set(HOUR_OF_DAY, hourOfDay);
		this.set(MINUTE, minute);
		this.set(SECOND, second);
	}

	public void setFirstDayOfWeek(int value) {
		this.firstDayOfTheWeek = value;
	}

	public void setLenient(boolean lenient) {
		// non-lenient mode not supported: check with an assert
		assert lenient == true;
	}

	public void setMinimalDaysInFirstWeek(int value) {
		this.minimalDayInFirstWeek = value;
	}

	public final void setTime(Date date) {
		this.setTimeInMillis(date.getTime());
	}

	public void setTimeInMillis(long millis) {
		this.time = millis;
		this.state = STATE_RESET_TIME;
		this.areFieldsSet = false;
	}

	public void setTimeZone(TimeZone value) {
    	this.timeZone = value;

    	final int state = this.state;
//    	if(state == STATE_RESET_TIME) {
    		// nothing has been computed - just change timezone
//    	}
    	/*else*/ if(state == STATE_UNVALIDATED_FIELDS) {
    		// synchronize time with fields with the new timezone
    		validateFields();
    	}
    	else if(state == STATE_SYNCHRONIZED) {
    		// fields need to be recomputed
    		this.state = STATE_RESET_TIME;
    		this.areFieldsSet = false;
    	}
	}

	@SuppressWarnings("nls")
	@Override
	public String toString() {
		// Mainly for debug purpose
		StringBuilder b = new StringBuilder("Calendar [")
		 .append("time=").append(this.time).append(',')
		 .append("minimalDayInFirstWeek=").append(this.minimalDayInFirstWeek).append(',')
		 .append("firstDayOfTheWeek=").append(this.firstDayOfTheWeek).append(',')
		 .append("state=").append(this.state).append(',')
		 .append("lenient=true,timezone=").append(this.timeZone)
		 .append(']');

		return b.toString();
	}

	private void setFields() {
		// Set fields as internally set, except for ZONE_OFFSET and DST_OFFSET that are not overwritten
		int[] setFields = this.setFields;
    	int setFieldsLength = setFields.length;
		for (int i = setFieldsLength; --i >= 0;) {
			if (i != ZONE_OFFSET && i != DST_OFFSET) {
				setFields[i] = FIELD_INTERNALLY_SET;
			} else {
				if (setFields[i] < FIELD_EXTERNALLY_SET) {
					setFields[i] = FIELD_INTERNALLY_SET;
				} else {
					setFields[i] = FIELD_EXTERNALLY_SET;
				}
			}
		}
		orderedSetFields = null;
		setFieldsCounter = FIRST_FIELD_EXTERNALLY_SET;
	}

	private void validateFields(){
    	computeTime();
    	computeFields();
		state = STATE_SYNCHRONIZED;
		this.areFieldsSet = true;

    	setFields();
    }

	/** INTERNAL-API
     * Compute and validate fields.
     * Fields may be accessed directly using underlying {@link #fields} array
     */
    public void validateBeforeGet(){
    	if(state == STATE_RESET_TIME){
    		computeFields();
    		state = STATE_SYNCHRONIZED;
    		this.areFieldsSet = true;

        	setFields();
    	}
    	else if(state == STATE_UNVALIDATED_FIELDS){
    		validateFields();
    	}
    }

}
