/*
 * Java
 *
 * Copyright 2013-2020 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.util;

import java.lang.CloneNotSupportedException;
import ej.annotation.NonNullByDefault;
import ej.annotation.Nullable;

@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
public abstract class AbstractMap<K, V> implements Map<K, V> {

	//Don't use enum for performance issues
	static final int KEYS = 0;
	static final int VALUES = 1;
	static final int ENTRIES = 2;
	@Nullable
	Set<K> keys;
	Collection<V> values;

	public static class SimpleEntry<K, V> extends AbstractMapEntry<K, V>{
		
		K key;
		
		public SimpleEntry(Entry<? extends K, ? extends V> entry) {
			this.key = entry.getKey();
			this.value = entry.getValue();
		}
		
		public SimpleEntry(K key, V value) {
			this.key = key;
			this.value = value;
		}
		
		public K getKey() {
			return this.key;
		}
	}

	/*
	 * Intenal common superclass for all Map implementation entries
	 * The key field is managed by sub classes (it can be a direct or a weak reference). 
	 */	
	@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
	protected static abstract class AbstractMapEntry<K, V> implements Entry<K, V>, java.io.Serializable {
		
		@Nullable
		V value;

		/**
	     * Called when this entry is accessed via {@link #put(Object, Object)}.
	     * This version does nothing, but in LinkedHashMap, it must do some
	     * bookkeeping for access-traversal mode.
	     */
		void access() {

		}

		/**
	     * Called when this entry is removed from the map. This version simply
	     * returns the value, but in LinkedHashMap, it must also do bookkeeping.
	     *
	     * @return the value of this key as it is removed
	     */
		@Nullable
		V cleanup() {
	      return this.value;
	    }

		@Override
		public boolean equals(@Nullable Object o) {
			if(!(o instanceof Map.Entry)) {
				return false;
			}

			@SuppressWarnings("unchecked")
			Map.Entry<K, V> oAsEntry = (Map.Entry<K, V>) o;
			K key = getKey();
			V value;
			return (key == null ? oAsEntry.getKey() == null : key.equals(oAsEntry.getKey())) && ((value=this.value) == null ? oAsEntry.getValue() == null : value.equals(oAsEntry.getValue()));
		}

		/**
		 * Get the key (from user point of view)
		 */
		public abstract K getKey();

		/**
		 * Get the key (from implementation point of view)
		 */
		@Nullable 
		public Object getInternKey(){
			return getKey();
		}
		
		@Nullable 
		public V getValue() {
			return this.value;
		}

		@Override
		public int hashCode() {
			K key = getKey();
			V value;
			return  (key == null ? 0 : key.hashCode()) ^ ((value=this.value) == null ? 0 : value.hashCode());
		}
		
		@Nullable 
		public V setValue(V value) {
			V oldValue = this.value;
			this.value = value;
			return oldValue;
		}

		@Override
		public String toString() {
			StringBuilder b = new StringBuilder();
			b.append(this.getKey() == this ? "this" : this.getKey()); //$NON-NLS-1$
			b.append('=');
			b.append(this.value == this ? "this" : this.value); //$NON-NLS-1$
			return b.toString();
		}

	}

	public static class SimpleImmutableEntry<K, V> extends SimpleEntry<K, V> {


		public SimpleImmutableEntry(Entry<? extends K, ? extends V> entry) {
			super(entry);
		}

		public SimpleImmutableEntry(K key, V value) {
			super(key, value);
		}


		@Override
		@Nullable 
		public V setValue(V value) {
			throw new UnsupportedOperationException();
		}
	}

	protected AbstractMap() {
	}

	public void clear() {
		// Should be override in subclasses for speed reasons
		this.entrySet().clear();
	}

	// clone() method should be implemented only if implementing Cloneable interface
	// In API, AbstractMap does not implement Cloneable and has clone() method defined
	@Override
	protected Object clone() throws CloneNotSupportedException { //NOSONAR
		// There is no fields in this class, nothing to do
		@SuppressWarnings("unchecked")
		AbstractMap<K, V> result = (AbstractMap<K, V>) super.clone();
		return result;
	}

	public boolean containsKey(Object key) {
		// O(speed) prior to O(size): loop is duplicated
		Iterator<Map.Entry<K, V>> iterator = this.entrySet().iterator();
		if(key != null) {
			while(iterator.hasNext()) {
				Map.Entry<K, V> entry = iterator.next();
				if(key.equals(entry.getKey())) {
					return true;
				}
			}			
		}
		else {
			while(iterator.hasNext()) {
				Map.Entry<K, V> entry = iterator.next();
				if(entry.getKey() == null) {
					return true;
				}
			}
		}
		return false;
	}

	public boolean containsValue(Object value) {
		// O(speed) prior to O(size): loop is duplicated
		Iterator<Map.Entry<K, V>> iterator = this.entrySet().iterator();
		if(value != null) {
			while(iterator.hasNext()) {
				Map.Entry<K, V> entry = iterator.next();
				if(value.equals(entry.getValue())) {
					return true;
				}
			}			
		}
		else {
			while(iterator.hasNext()) {
				Map.Entry<K, V> entry = iterator.next();
				if(entry.getValue() == null) {
					return true;
				}
			}
		}

		return false;
	}

	public abstract Set<Entry<K, V>> entrySet();

	@Override
	public boolean equals(@Nullable Object o) {
		// Steps:
		// - check if the arg is this (return true)
		// - then, check if the arg is a map of the same size as this
		// - then, check if all the entries in this are in the arg, and vice-versa

		// step 1
		if(o == this) {
			return true;
		}

		// step 2
		if(!(o instanceof Map)) {
			return false;
		}

		@SuppressWarnings("unchecked")
		Map<K, V> oAsMap = (Map<K, V>) o; // cast is sure
		if(oAsMap.size() != this.size()) {
			return false;
		}

		// step 3
		return this.entrySet().equals(oAsMap.entrySet());
	}
	
	@Nullable 
	public V get(Object key) {
		// O(speed) prior to O(size): loop is duplicated
		Iterator<Map.Entry<K, V>> iterator = this.entrySet().iterator();
		if(key != null) {
			while(iterator.hasNext()) {
				Map.Entry<K, V> entry = iterator.next();
				if(key.equals(entry.getKey())) {
					return entry.getValue();
				}
			}			
		}
		else {
			while(iterator.hasNext()) {
				Map.Entry<K, V> entry = iterator.next();
				if(entry.getKey() == null) {
					return entry.getValue();
				}
			}
		}
		return null;
	}

	@Override
	public int hashCode() {
		int hashCode = 0;

		Iterator<Map.Entry<K, V>> iterator = this.entrySet().iterator();
		while(iterator.hasNext()) {
			Entry<K, V> next = iterator.next();
			assert next != null;
			hashCode += next.hashCode();
		}

		return hashCode;
	}

	public boolean isEmpty() {
		return this.size() == 0;
	}

	private class AbstractKeysSet extends AbstractSet<K> {

		@Override
		public boolean add(K e) {
			throw new UnsupportedOperationException();
		}

		@Override
		public boolean addAll(Collection<? extends K> c) {
			throw new UnsupportedOperationException();
		}

		@Override
		public void clear() {
			AbstractMap.this.clear();
		}

		@Override
		public boolean contains(Object k) {
			return AbstractMap.this.containsKey(k);
		}

		@Override
		public boolean isEmpty() {
			return AbstractMap.this.isEmpty();
		}

		@Override
		public Iterator<K> iterator() {
			return new Iterator<K>() {

				private Iterator<Map.Entry<K, V>> iterator = AbstractMap.this.entrySet().iterator();

				public boolean hasNext() {
					return iterator.hasNext();
				}

				public K next() {
					Entry<K, V> next = iterator.next();
					assert next != null; // entrySet never null
					return next.getKey();
				}

				public void remove() {
					iterator.remove();
				}
			};
		}

		@Override
		public int size() {
			return AbstractMap.this.size();
		}
	}
	
	public Set<K> keySet() {
		if(keys == null){
			// Return a wrapper over this map
			keys = new AbstractKeysSet();

		}
		return keys;
	}
	
	@Nullable 
	public V put(K key, V value) {
		// as specified in Javadoc
		throw new UnsupportedOperationException();
	}

	public void putAll(Map<? extends K, ? extends V> m) {
		for(Map.Entry<? extends K, ? extends V> entry : m.entrySet()) {
			this.put(entry.getKey(), entry.getValue());
		}
	}
	
	@Nullable 
	public V remove(Object key) {
		// O(speed) prior to O(size): loop is duplicated
		Iterator<Map.Entry<K, V>> iterator = this.entrySet().iterator();
		Entry<K, V> entry = null;
		if(key != null) {
			while(iterator.hasNext()) {
				Map.Entry<K, V> e = iterator.next();
				if(key.equals(e.getKey())) {
					entry = e;
				}
			}			
		}
		else {
			while(iterator.hasNext()) {
				Map.Entry<K, V> e = iterator.next();
				if(e.getKey() == null) {
					entry = e;
				}
			}
		}
		
		V oldValue = null;
		if (entry != null) {
			oldValue = entry.getValue();
			iterator.remove();
		}
		return oldValue;
	}

	public int size() {
		// must be override in subclasses for speed reasons
		return this.entrySet().size();
	}

	@Override
	public String toString() {
		StringBuilder b = new StringBuilder().append('{');

		Iterator<Entry<K, V>> iterator = this.entrySet().iterator();
		while(iterator.hasNext()) {
			Entry<K, V> entry = iterator.next();
			K key = entry.getKey();
			V value = entry.getValue();
			b.append(key == this ? "this" : key) //$NON-NLS-1$
			 .append('=')
			 .append(value == this ? "this" : value); //$NON-NLS-1$
			if(iterator.hasNext()) {
				b.append(", "); //$NON-NLS-1$
			}
		}

		b.append('}');
		return b.toString();
	}

	public Collection<V> values() {
		if(values == null){
			// return a wrapper over this map
			values =  new AbstractCollection<V>() {

				@Override
				public boolean add(V e) {
					throw new UnsupportedOperationException();
				}

				@Override
				public boolean addAll(Collection<? extends V> c) {
					throw new UnsupportedOperationException();
				}

				@Override
				public void clear() {
					AbstractMap.this.clear();
				}

				@Override
				public boolean contains(Object v) {
					return AbstractMap.this.containsValue(v);
				}

				@Override
				public boolean isEmpty() {
					return AbstractMap.this.isEmpty();
				}

				@Override
				public Iterator<V> iterator() {
					return new Iterator<V>() {
						private Iterator<Entry<K, V>> iterator = AbstractMap.this.entrySet().iterator();

						public boolean hasNext() {
							return this.iterator.hasNext();
						}

						public V next() {
							return this.iterator.next().getValue();
						}

						public void remove() {
							this.iterator.remove();
						}
					};
				}

				@Override
				public int size() {
					return AbstractMap.this.size();
				}
			};

		}

		return values;
	}

}
