/*
 * Java
 *
 * Copyright 2013-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.util;

import ej.annotation.NonNullByDefault;
import ej.annotation.Nullable;

@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
public abstract class AbstractList<E> extends AbstractCollection<E> implements List<E> {

	protected class InternalIterator implements Iterator<E> {
		
		int lastElemIndex;
		int position;
		int modCountNeeded; // needed modCount to execute operations.
							// If invalid, throws ConcurrentModificationException
		
		public InternalIterator() {
			this.lastElemIndex = -1;
			this.position = 0;
			this.modCountNeeded = AbstractList.this.modCount;
		}

		public boolean hasNext() {
			return this.position != AbstractList.this.size();
		}

		public E next() {
			if (this.position == AbstractList.this.size()) {
				throw new NoSuchElementException();
			}
			
			if(modCountNeeded != modCount) {
				throw new ConcurrentModificationException();
			}
			
			try {
				E nextElem = AbstractList.this.get(this.position);
				this.lastElemIndex = this.position++;
				return nextElem;
			}
			catch(IndexOutOfBoundsException e) {
				// AbstractList.get throws IndexOutOfBoundsException because
				// the index is invalid.
				throw new NoSuchElementException();
			}
		}

		public void remove() {
			if(this.modCountNeeded != AbstractList.this.modCount) {
				throw new ConcurrentModificationException();
			}
			
			final int lastElemIndex = this.lastElemIndex;
			if(lastElemIndex == -1) {
				// Already done once, or 0 call to next()
				throw new IllegalStateException();
			}
			
			try {
				final AbstractList<E> thisList = AbstractList.this;
				thisList.remove(lastElemIndex);
				final int position = this.position;
				if(lastElemIndex < position) {
					// Removed elem was before current position
					// Need to decrement current pos to stay in List bounds
					this.position = position - 1;
				}
				this.modCountNeeded = thisList.modCount; // Not a concurrent modif.
				this.lastElemIndex = -1;
			}
			catch(IndexOutOfBoundsException e) {
				// AbstractList.get throws IndexOutOfBoundsException because
				// the index is invalid. So some elements have been removed
				// from the list, but this iterator was valid!
				// There were concurrent modifications.
				throw new ConcurrentModificationException();
			}
		}
	}
	
	protected class InternalListIterator extends InternalIterator implements ListIterator<E> {

		public InternalListIterator(int index) {
			super();
			
			if (index < 0 || index > AbstractList.this.size()) {
				throw new IndexOutOfBoundsException(String.valueOf(index));
			}
			
			this.position = index;
		}
		
		public void add(E e) {
			final AbstractList<E> thisList = AbstractList.this;
			if(this.modCountNeeded != thisList.modCount) {
				throw new ConcurrentModificationException();
			}
			
			try {
				AbstractList.this.add(this.position, e);
				this.lastElemIndex = -1;
				this.modCountNeeded = thisList.modCount; // Not a concurrent modif.
				this.position++;
			}
			catch(IndexOutOfBoundsException ex) {
				throw new ConcurrentModificationException();
			}
		}

		public boolean hasPrevious() {
			return this.position > 0;
		}

		public int nextIndex() {
			return this.position;
		}

		public E previous() {
			if(modCountNeeded != modCount) {
				throw new ConcurrentModificationException();
			}
			
			try {
				E previousElem = AbstractList.this.get(this.position - 1);
				this.lastElemIndex = --this.position;
				return previousElem;
			}
			catch(IndexOutOfBoundsException e) {
				// AbstractList.get throws IndexOutOfBoundsException because
				// the index is invalid.
				throw new NoSuchElementException();
			}
		}

		public int previousIndex() {
			return this.position - 1;
		}

		public void set(E e) {
			if(modCountNeeded != modCount) {
				throw new ConcurrentModificationException();
			}
			
			if(this.lastElemIndex == -1) {
				throw new IllegalStateException();
			}
			
			try {
				final AbstractList<E> thisList = AbstractList.this;
				thisList.set(lastElemIndex, e);
				this.modCountNeeded = thisList.modCount; // Not a concurrent modif.
			}
			catch(IndexOutOfBoundsException ex) {
				// AbstractList.set throws IndexOutOfBoundsException because
				// the index is invalid. So some elements have been removed
				// from the list, but this iterator was valid!
				// There were concurrent modifications.
				throw new ConcurrentModificationException();
			}
		}
	}

	protected transient int modCount = 0;

	protected AbstractList() {
	}

	@Override
	public boolean add(E e) {
		add(this.size(), e);
		return true;
	}

	public void add(int index, E element) {
		// See javadoc. This impl. always throws UnsupportedOperationException
		throw new UnsupportedOperationException();
	}

	public boolean addAll(int index, Collection<? extends E> c) {
		if (index < 0 || index > this.size()) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}
		
		boolean hasBeenModified = false;
		Iterator<? extends E> iterator = c.iterator();
		while(iterator.hasNext()) {
			// Only set hasBeenModified in the loop, in case collection c
			// has been cleared between hasBeenModified declaration and
			// iterator declaration
			hasBeenModified = true;
			// Increment index to insert elements in c in the same order they
			// appear in c
			this.add(index++, iterator.next());
		}
		return hasBeenModified;
	}

	@Override
	public void clear() {
		this.removeRange(0, this.size());
	}

	@Override
	@SuppressWarnings("unchecked")
	public boolean equals(@Nullable Object o) {
		if(this == o) {
			return true;
		}
		
		if(o == null) {
			return false;
		}
		
		List<E> otherList = null;
		try {
			otherList = (List<E>) o;
		}
		catch(ClassCastException e) {
			// o is not a list or not a list of the same type.
			return false;
		}
		
		// otherList is not null for sure.
		if(this.size() != otherList.size()) {
			return false;
		}
		
		// Iterate over the two lists (they have the same size) and compare
		// pairs of elements
		Iterator<E> iteratorThis = this.iterator();
		Iterator<E> iteratorOther = otherList.iterator();
		while(iteratorThis.hasNext() && iteratorOther.hasNext()) {
			E obj1 = iteratorThis.next();
			E obj2 = iteratorOther.next();
			if(!(obj1 == null ? (obj2 == null) : obj1.equals(obj2))) {
				return false;
			}
		}
		
		return true;
	}

	public abstract E get(int index);

	@Override
	public int hashCode() {
		int hashCode = 1;
		 for(E e : this) {
		 	hashCode = 31 * hashCode + (e == null ? 0 : e.hashCode());
		 }
		 return hashCode;
	}

	public int indexOf(Object o) {
		return this.indexOf(o, 0);
	}
	
	protected int indexOf(Object o, int index) {
		if(index < 0) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}
		
		boolean oIsNull = (o == null);
		ListIterator<E> listIterator = this.listIterator(index);
		while(listIterator.hasNext()) {
			E elem = listIterator.next();
			boolean found;
			if(oIsNull) {
				found = elem == null;
			}
			else {
				assert o != null;
				found = o.equals(elem);	
			}
			if(found) {
				// return nextIndex() - 1, so previousIndex()
				return listIterator.previousIndex();
			}
		}
		return -1;
	}

	@Override
	public Iterator<E> iterator() {
		return new InternalIterator();
	}

	public int lastIndexOf(Object o) {
		return this.lastIndexOf(o, this.size());
	}
	
	protected int lastIndexOf(Object o, int index) {
		boolean oIsNull = (o == null);
		ListIterator<E> listIterator = this.listIterator(index);
		while(listIterator.hasPrevious()) {
			E elem = listIterator.previous();
			boolean found;
			if(oIsNull) {
				found = elem == null;
			}
			else {
				assert o != null;
				found = o.equals(elem);	
			}
			if(found) {
				// return previousIndex() + 1, so nextIndex()
				return listIterator.nextIndex();
			}
		}
		return -1;
	}

	public ListIterator<E> listIterator() {
		return this.listIterator(0);
	}

	public ListIterator<E> listIterator(final int index) {
		return new InternalListIterator(index);
	}

	public E remove(int index) {
		// see javadoc, always throws UnsupportedOperationException
		throw new UnsupportedOperationException();
	}

	protected void removeRange(int fromIndex, int toIndex) {
		ListIterator<E> listIterator = this.listIterator(fromIndex);
		// Ensure NoSuchElement exception raised by iterator
		for(int i = fromIndex; i < toIndex; i++) {
			listIterator.next();
			listIterator.remove();
		}
	}

	public E set(int index, E element) {
		// see javadoc, always throws UnsupportedOperationException
		throw new UnsupportedOperationException();
	}

	public List<E> subList(int fromIndex, int toIndex) {
		return (this instanceof RandomAccess ? new RandomAccessList<E>(this, fromIndex, toIndex) : new SubList<E>(this, fromIndex, toIndex));
	}

}

@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
class SubList<E> extends AbstractList<E> {
	
	private AbstractList<E> originalList;
	private int offset;
	private int size;
	
	public SubList(AbstractList<E> list, int fromIndex, int toIndex) {
		if (fromIndex < 0|| fromIndex > toIndex) {
			throw new IndexOutOfBoundsException(String.valueOf(fromIndex));
		}
		if (toIndex > list.size()) {
			throw new IndexOutOfBoundsException(String.valueOf(toIndex));
		}
		
		this.originalList = list;
		this.modCount = this.originalList.modCount;
		this.offset = fromIndex;
		this.size = toIndex - fromIndex;
	}
	
	@Override
	public void add(int index, E element) {
		if(index < 0 || index > size) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}
		
		final AbstractList<E> originalList = this.originalList;
		if (this.modCount != originalList.modCount) {
			throw new ConcurrentModificationException();
		}
		
		originalList.add(index + this.offset, element);
		this.size++;
		this.modCount = originalList.modCount; // Not a concurrent modif
	}
	
	@Override
	public E set(int index, E element) {
		if(index < 0 || index > size) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}
		final AbstractList<E> originalList = this.originalList;
		if (this.modCount != originalList.modCount) {
			throw new ConcurrentModificationException();
		}
		
		E result = originalList.set(index + this.offset, element);
		this.modCount = originalList.modCount; // Not a concurrent modif
		return result;
	}
	
	@Override
	public E remove(int index) {
		if(index < 0 || index > size) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}
		final AbstractList<E> originalList = this.originalList;
		if (this.modCount != originalList.modCount) {
			throw new ConcurrentModificationException();
		}
		
		E result = originalList.remove(index + this.offset);
		this.size--;
		this.modCount = originalList.modCount; // Not a concurrent modif
		return result;
	}
	
	@Override
	public boolean addAll(int index, Collection<? extends E> c) {
		if(index < 0 || index > size) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}
		final AbstractList<E> originalList = this.originalList;
		if (this.modCount != originalList.modCount) {
			throw new ConcurrentModificationException();
		}
		
		boolean result = originalList.addAll(index, c);
		this.size += c.size();
		this.modCount = originalList.modCount; // Not a concurrent modif
		return result;
	}
	
	@Override
	protected void removeRange(int fromIndex, int toIndex) {
		final AbstractList<E> originalList = this.originalList;
		if (this.modCount != originalList.modCount) {
			throw new ConcurrentModificationException();
		}
		
		final int offset = this.offset;
		originalList.removeRange(fromIndex + offset, toIndex + offset);
		this.size -= (toIndex - fromIndex);
		this.modCount = originalList.modCount; // Not a concurrent modif
	}

	@Override
	public E get(int index) {
		if (index < 0 || index >= this.size) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}
		final AbstractList<E> originalList = this.originalList;
		if (this.modCount != originalList.modCount) {
			throw new ConcurrentModificationException();
		}
		return originalList.get(index + this.offset);
	}

	@Override
	public int size() {
		if (this.modCount != this.originalList.modCount) {
			throw new ConcurrentModificationException();
		}
		
		return this.size;
	}
	
	@Override
	public List<E> subList(int fromIndex, int toIndex) {
		return new SubList<E>(this, fromIndex, toIndex);
	}
	
	@Override
	public Iterator<E> iterator() {
		return listIterator(0); // see javadoc
	}
	
	@Override
	public ListIterator<E> listIterator(final int index) {
		if (index < 0 || index > this.size) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}
		if (this.modCount != this.originalList.modCount) {
			throw new ConcurrentModificationException();
		}
		
		return new ListIterator<E>() {
			private ListIterator<E> internalIterator = SubList.this.originalList.listIterator(index + SubList.this.offset);

			public void add(E e) {
				this.internalIterator.add(e);
				SubList.this.size++;
				SubList.this.modCount = SubList.this.originalList.modCount;
			}

			public boolean hasNext() {
				return this.nextIndex() < SubList.this.size;
			}

			public boolean hasPrevious() {
				return this.previousIndex() >= 0;
			}

			public E next() {
				if (! hasNext()) {
					throw new NoSuchElementException();
				}
				
				return this.internalIterator.next();
			}

			public int nextIndex() {
				return this.internalIterator.nextIndex() - SubList.this.offset;
			}

			@Nullable
			public E previous() {
				return this.internalIterator.previous();
			}

			public int previousIndex() {
				return this.internalIterator.previousIndex() - SubList.this.offset;
			}

			public void remove() {
				this.internalIterator.remove();
				SubList.this.size--;
				SubList.this.modCount = SubList.this.originalList.modCount;
			}

			public void set(E e) {
				this.internalIterator.set(e);
			}
		};
	}
	
}

@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
class RandomAccessList<E> extends SubList<E>  implements RandomAccess {
	
	public RandomAccessList(AbstractList<E> list, int fromIndex, int toIndex) {
		super(list, fromIndex, toIndex);
	}
	
	@Override
	public List<E> subList(int fromIndex, int toIndex) {
		return new RandomAccessList<E>(this, fromIndex, toIndex);
	}
}