/*
 * Java
 *
 * Copyright 2013-2019 IS2T. All rights reserved
 * IS2T PROPRIETARY/CONFIDENTIAL Use is subject to license terms.
 */
package java.util;

import com.is2t.tools.ArrayTools;

import ej.annotation.NonNull;
import ej.annotation.NonNullByDefault;
import ej.annotation.Nullable;

@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
public abstract class AbstractArrayList<E> extends AbstractList<E> implements List<E>, RandomAccess, Cloneable, java.io.Serializable {

	private static final int DEFAULT_CAPACITY = 10;

	@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
	private class AbstractArrayListIterator extends AbstractList<E>.InternalIterator {

		public AbstractArrayListIterator() {
			super();
		}

		@Override
		@Nullable
		public E next() {
			// Specific impl: use AbstractArrayList.this.elements instead of
			// calling AbstractArrayList.this.get()
			if(modCountNeeded != modCount) {
				throw new ConcurrentModificationException();
			}

			try {
				if (this.position == AbstractArrayList.this.elementCount) {
					throw new NoSuchElementException();
				}

				this.lastElemIndex = this.position;
				@SuppressWarnings("unchecked")
				E result = (E) AbstractArrayList.this.elementData[this.position++];
				return result;
			}
			catch(ArrayIndexOutOfBoundsException e) {
				// No need exception cause because it implementation specific
				throw new NoSuchElementException(); // NOSONAR
			}
		}
	}

	@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
	private class AbstractArrayListListIterator extends AbstractList<E>.InternalListIterator  {

		public AbstractArrayListListIterator(int index) {
			super(index);
		}

		@Override
		public E previous() {
			// Specific impl: use AbstractArrayList.this.elements instead of
			// calling AbstractArrayList.this.get()

			if(modCountNeeded != modCount) {
				throw new ConcurrentModificationException();
			}

			try {
				if (this.position == 0) {
					throw new NoSuchElementException();
				}

				this.lastElemIndex = this.position;
				this.position--;
				@SuppressWarnings("unchecked")
				E result = (E) AbstractArrayList.this.elementData[this.position];
				return result;
			}
			catch(IndexOutOfBoundsException e) {
				// AbstractList.get throws IndexOutOfBoundsException because
				// the index is invalid.
				// No need exception cause because it implementation specific
				throw new NoSuchElementException(); //NOSONAR
			}
		}
	}

	protected int capacityIncrement;
	protected int elementCount;
	protected Object[] elementData; // Use Object[] since it is not possible to create
									 // a new array from a generic type

	public AbstractArrayList() {
		this(DEFAULT_CAPACITY);
	}

	public AbstractArrayList(Collection<? extends E> c) {
		// copy the collection elements in a new array
		this.elementData = c.toArray(new Object[c.size()]);
		this.elementCount = this.elementData.length;
	}

	public AbstractArrayList(int initialCapacity) {
		if(initialCapacity < 0) {
			throw new IllegalArgumentException(String.valueOf(initialCapacity));
		}
		this.elementData = new Object[initialCapacity];
		this.elementCount = 0;
	}

	public AbstractArrayList(int initialCapacity, int capacityIncrement) {
		this(initialCapacity);
		this.capacityIncrement = capacityIncrement;
	}

	@Override
	public boolean add(E e) {
		this.ensureFreeSpace(1);
		this.elementData[this.elementCount] = e;
		this.elementCount++;

		return true;
	}

	@Override
	public void add(int index, E element) {
		if(index < 0 || index > this.elementCount) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}

		this.ensureFreeSpace(1);
		@SuppressWarnings("null")
		@NonNull Object[] elementData = this.elementData;
		System.arraycopy(elementData, index, elementData, index + 1, this.elementCount - index);
		this.elementData[index] = element;
		this.elementCount++;
	}

	@Override
	public boolean addAll(Collection<? extends E> c) {
		final int elementCount = this.elementCount;
		this.ensureCapacity(elementCount + c.size());
		return this.addAll(elementCount, c);
	}

	@Override
	public boolean addAll(int index, Collection<? extends E> c) {
		final int elementCount = this.elementCount;
		if(index < 0 || index > elementCount) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}

		Object[] collectionElements = c.toArray();
		int collectionElementsLength = collectionElements.length;
		this.ensureFreeSpace(collectionElementsLength);

		int movedCount = elementCount - index;
		@SuppressWarnings("null")
		@NonNull Object[] elementData = this.elementData;
		if(movedCount > 0) {
			// need to move some elements in this list
			System.arraycopy(elementData, index, elementData, index + collectionElementsLength, movedCount);
		}

		System.arraycopy(collectionElements, 0, elementData, index, collectionElementsLength);
		this.elementCount = elementCount + collectionElementsLength;

		return collectionElementsLength != 0; // true if this collection has changed
	}

	@Override
	public void clear() {
		final int elementCount = this.elementCount;
		final Object[] elementData = this.elementData;
		for(int i = 0; i < elementCount; i++) {
			elementData[i] = null;
		}
		this.elementCount = 0;
		this.modCount++;
	}

	@Override
	public Object clone() {
		// Must return a shallow copy (so not a deep copy, elements themselves
		// are not cloned). Use System.arraycopy

		try {
			@SuppressWarnings("unchecked")
			AbstractArrayList<E> cloned = (AbstractArrayList<E>) super.clone();
			final int elementCount = this.elementCount;
			final Object[] clonedElementData = new Object[elementCount];
			cloned.elementData = clonedElementData;
			@SuppressWarnings("null")
			@NonNull Object[] elementData = this.elementData;
			System.arraycopy(elementData, 0, clonedElementData, 0, elementCount);
			cloned.elementCount = elementCount;
			cloned.modCount = 0;
			return cloned;
		}
		catch(CloneNotSupportedException e) {
			// If clone operation fails (see javadoc of Object.clone)
			throw new InternalError(); //NOSONAR
		}
	}

	@Override
	public boolean contains(Object o) {
		// indexOf(Object) has the same spec. that contains(Object)
		return this.indexOf(o) >= 0;
	}

	public void ensureCapacity(int minCapacity) {
		int neededFreeSpace = minCapacity - this.elementCount;
		if(minCapacity < 0 || neededFreeSpace < 0) {
			return;
		}

		this.ensureFreeSpace(neededFreeSpace);
	}

	@Override
	public E get(int index) {
		if(index < 0 || index >= this.elementCount) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}

		@SuppressWarnings("unchecked")
		E result = (E) this.elementData[index];
		return (E) result;
	}

	@Override
	public int indexOf(Object o) {
		// O(speed) prior to O(size): loop is duplicated
		final int elementCount = this.elementCount;
		final Object[] elementData = this.elementData;
		if(o != null) {
			for(int i = 0; i < elementCount; i++) {
				if(o.equals(elementData[i])) {
					return i;
				}
			}
		}
		else {
			for(int i = 0; i < elementCount; i++) {
				if(elementData[i] == null) {
					return i;
				}
			}
		}
		return -1;
	}

	@Override
	public boolean isEmpty() {
		return this.elementCount == 0;
	}

	@Override
	public Iterator<E> iterator() {
		return new AbstractArrayListIterator();
	}

	@Override
	public int lastIndexOf(Object o) {
		// O(speed) prior to O(size): loop is duplicated
		final int elementCount = this.elementCount;
		final Object[] elementData = this.elementData;
		if(o != null) {
			for(int i = elementCount - 1; i >= 0; i--) {
				if(o.equals(elementData[i])) {
					return i;
				}
			}			
		}
		else {
			for(int i = elementCount - 1; i >= 0; i--) {
				if(elementData[i] == null) {
					return i;
				}
			}
		}

		return -1;
	}

	@Override
	public ListIterator<E> listIterator() {
		return this.listIterator(0);
	}

	@Override
	public ListIterator<E> listIterator(int index) {
		return new AbstractArrayListListIterator(index);
	}

	@Override
	public E remove(int index) {
		int elementCount = this.elementCount;
		if(index < 0 || index >= elementCount) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}

		final Object[] elementData = this.elementData;
		@SuppressWarnings("unchecked")
		E removedElem = (E) elementData[index];
		int rankDeleted = elementCount - index - 1;
		if(rankDeleted > 0) {
			// if this is not the last element to remove
			System.arraycopy(elementData, index + 1, elementData, index, rankDeleted);
		}
		// else {
		//		Last element is removed, nothing to do!
		// }
		this.elementCount = --elementCount;
		elementData[elementCount] = null;
		this.modCount++;

		return removedElem;
	}

	@Override
	public boolean remove(Object o) {
		boolean oIsNull = (o == null);
		int thisSize = size();

		for(int i = 0; i < thisSize; i++) {
			final Object[] elementData = this.elementData;
			final Object element = elementData[i];
			boolean found;
			if(oIsNull) {
				found = element == null;
			}
			else {
				assert o != null;
				found = o.equals(element);
			}
			if(found) {
				int moveCount = --thisSize - i;
				System.arraycopy(elementData, i + 1, elementData, i, moveCount);
				elementData[--this.elementCount] = null;
				this.modCount++;
				return true;
			}
		}

		return false;
	}

	@Override
	public boolean removeAll(Collection<?> c) {
		boolean hasBeenModified = false;

		// To remove elements, use compact algo on elements array
		int j = 0; // filling index
		final int elementCount = this.elementCount;
		final Object[] elementData = this.elementData;
		for(int i = 0; i < elementCount; i++) {
			Object elem = elementData[i];
			assert elem != null;
			if(!c.contains(elem)) {
				elementData[j] = elementData[i];
				j++;
			}
		}

		// if some elements have been removed
		if(j != elementCount) {
			for(int k = j; k < elementCount; k++) {
				elementData[k] = null;
			}
			this.elementCount = j;
			this.modCount++; // some elements removed, but considered
							 // as one structural modification
			hasBeenModified = true;
		}
		return hasBeenModified;
	}

	@Override
	protected void removeRange(int fromIndex, int toIndex) {
		if(toIndex == fromIndex) {
			return;
		}

		if(fromIndex < 0 || fromIndex >= size() || toIndex > size() || toIndex < fromIndex) {
			throw new IndexOutOfBoundsException();
		}

		int removeCount = toIndex - fromIndex;
		final int elementCount = this.elementCount;
		@SuppressWarnings("null")
		@NonNull Object[] elementData = this.elementData;
		int movedCount = elementCount - toIndex;
		System.arraycopy(elementData, toIndex, elementData, fromIndex, movedCount);
		for(int i= elementCount - removeCount; i < elementCount; i++) {
			elementData[i] = null;
		}
		this.elementCount = elementCount - removeCount;
		this.modCount++;
	}

	@Override
	public boolean retainAll(Collection<?> c) {
		boolean hasBeenModified = false;

		final int elementCount = this.elementCount;
		final Object[] elementData = this.elementData;
		// To remove elements, use compact algo on elements array
		int j = 0; // filling index
		for(int i = 0; i < elementCount; i++) {
			Object result = elementData[i];
			assert result != null;
			if(c.contains(result)) {
				elementData[j] = elementData[i];
				j++;
			}
		}

		// if some elements have been removed
		if(j != elementCount) {
			for(int k = j; k < elementCount; k++) {
				this.elementData[k] = null;
			}
			this.elementCount = j;
			this.modCount++; // some elements removed, but considered
							 // as one structural modification
			hasBeenModified = true;
		}
		return hasBeenModified;
	}

	@Override
	public E set(int index, E element) {
		if(index < 0 || index >= this.elementCount) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}

		final Object[] elementData = this.elementData;
		@SuppressWarnings("unchecked")
		E oldElement = (E) elementData[index];
		assert oldElement != null;
		elementData[index] = element;
		return oldElement;
	}

	@Override
	public int size() {
		return this.elementCount;
	}

	@Override
	public Object[] toArray() {
		final int elementCount = this.elementCount;
		Object[] toArray = new Object[elementCount];
		@SuppressWarnings("null")
		@NonNull Object[] elementData = this.elementData;
		System.arraycopy(elementData, 0, toArray, 0, elementCount);
		return toArray;
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T> T[] toArray(T[] a) {
		int thisSize = this.size();
		int destSize = a.length;

		if(thisSize > destSize) {
			a = (T[])ArrayTools.createNewArrayFromType(a.getClass(), thisSize);

		}else if(destSize > thisSize){
			a[thisSize] = null; //null-terminate
		}

		@SuppressWarnings("null")
		@NonNull Object[] elementData = this.elementData;
		System.arraycopy(elementData, 0, a, 0, thisSize);
		return a;
	}

	public void trimToSize() {
		final int elementCount = this.elementCount;
		@SuppressWarnings("null")
		@NonNull Object[] elementData = this.elementData;
		System.arraycopy(elementData, 0, this.elementData = new Object[elementCount], 0, elementCount);
		this.modCount++;
	}

	private void ensureFreeSpace(int freeSpace) {
		final int elementCount = this.elementCount;
		final Object[] elementData = this.elementData;
		final int elementDataLength = elementData.length;
		if(elementCount + freeSpace > elementDataLength) {
			int newCapacity;
			if(this.capacityIncrement <= 0) {
				newCapacity = Math.max(elementDataLength * 2, freeSpace);
			}
			else {
				int nextCapacity = elementDataLength + this.capacityIncrement;
				newCapacity = (nextCapacity-elementCount > freeSpace ? nextCapacity : elementCount+freeSpace);
			}
			System.arraycopy(elementData, 0, this.elementData = new Object[newCapacity], 0, elementCount);
		}
		// Always increment modification count since when you call this method,
		// you want to add something to the list
		this.modCount++;
	}
}
