/*
 * Java
 *
 * Copyright 2013-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang.ref;

import ej.annotation.Nullable;

public class ReferenceQueue<T> {

	//linked list of Reference that have been set to null or enqueued (next in the list is accessible with Reference.nextInQueue)
	@Nullable
	private Reference<T> head;//VM known (use by poll)

	//we have to use a special object to synchronize on the queue
	//since we do not want to interact with the application synchro
	private Object monitor;

	//thread that is waiting for an element.
	//VM will wake up this thread
	@Nullable
	private Thread waitingThread;//VM Known


	public ReferenceQueue()
	{
		monitor = new Object();
		//head = null; VM done
	}


	/**
	 * Queries the queue and returns and removes the first element of the queue.
	 * If the queue is empty, returns <code>null</code>.
	 * @return Reference or <code>null</code>
	 */
	@Nullable 
	public Reference<T> poll() {
		return poll(this);
	}

	/**
	 * Queries the queue, returns and removes the first element of the queue.
	 * If the queue is empty, blocks the current thread until the queue gets
	 * at least one element (automatically added by the system).
	 * @return Reference
	 * @throws InterruptedException if the thread is interrupted
	 */
	public Reference<T> remove() throws InterruptedException {
		Reference<T> result = remove(0);
		assert result != null; // see specification
		return result;
	}
	
	@Nullable 
	public Reference<T> remove(long timeout) throws InterruptedException
	{
		Reference<T> result;

		synchronized(monitor){
			while(waitingThread != null){
				monitor.wait();
			}
			waitingThread = Thread.currentThread();
		}

		//Only 1 thread can wait for an element to be add in this queue.
		//The others are blocked on wait until current thread
		//get an element. So the others thread can be interrupted.
		//poll() can be done by another thread because poll() is not synchronized on monitor
		try{
			if(timeout == 0){
				//wait until data is available
				while( (result=poll())==null ){
					waitForReference(0);
					//We have been waken up because an element has been added in the queue but
					//somebody may catch it before us (with a call to poll()) so we need to
					//loop until some data is available.
				}
			}
			else {
				//wait until data is available or timeout
				result=poll();
				if(result==null ){
					waitForReference(timeout);
					result=poll();
				}
			}
		}
		finally{
			//InterruptedException may occur
			synchronized(monitor){
				waitingThread = null;
				monitor.notify();//no need to notify all: only one can wait for a reference
			}

		}

		return result;
	}


	//Wait for an element to be add in this ReferenceQueue.
	static native public void waitForReference(long timeout) throws InterruptedException;

	static native public <T> Reference<T> poll(ReferenceQueue<T> queue);
	
	// return false if the reference is already enqueued or not registered to a queue, true otherwise
	static native <T> boolean enqueue(Reference<T> reference);

}
