/*
 * Java
 *
 * Copyright 2008-2024 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;

import java.io.PrintStream;
import java.io.Serializable;

import com.is2t.tools.ArrayTools;
import com.is2t.vm.support.err.EDCErrorMessages;
import ej.annotation.Nullable;

import ej.error.Message;

public class Throwable implements Serializable {

	//WARNING: don't add remove fields before or between "VM known fields"
	// VM known fields
	@Nullable
 	private String message ;
	@Nullable
 	protected int[] ips ;
	@Nullable
 	protected Throwable cause; // default is null
	@Nullable
 	private Throwable[] suppressed;
	// End VM known fields

	/**
	 * Shared array for all the call to {@link #getSuppressed()} when there is no suppressed exception.
	 */
	private static final Throwable[] EMPTY_SUPPRESSED_EXCEPTIOON = new Throwable[0];
	
    public Throwable() {
    	fillInStackTrace() ;
    }

    public Throwable(@Nullable String message) {
    	this.message = message ;
    	fillInStackTrace() ;
    }

    public Throwable(String message, @Nullable Throwable cause) {
    	this.message = message ;
    	this.cause = cause;
    	fillInStackTrace() ;
	}

    public Throwable(@Nullable Throwable cause) {
    	this.cause = cause;
    	if(cause != null) {
    		this.message = cause.toString();
    	}
    	fillInStackTrace() ;
	}

    @Nullable 
    public String getMessage() {
    	return this.message ;
    }

    public final synchronized void addSuppressed(@Nullable Throwable exception){
    	if(exception == this){
			throw new IllegalArgumentException(Message.at(new EDCErrorMessages(), EDCErrorMessages.ThrowableCouldNotSuppressItself));
    	}
    	if(exception == null){
			throw new NullPointerException(Message.at(new EDCErrorMessages(), EDCErrorMessages.ThrowableCouldNotSuppressNull));
    	}
    	
    	if(this instanceof OutOfMemoryError) {
    		// Suppressed exceptions not recorded for OutOfMemoryError because it is an immutable object.
    		// This is an acceptable limitation.
    		// Actually, JavaSE has the same behavior and JDK API allows to create an exception with supression disabled.
    		return;
    	}
    	
    	Throwable[] localSuppressed = this.suppressed;
    	if(localSuppressed != null) {
    		this.suppressed = (Throwable[])ArrayTools.add(localSuppressed, exception);
    	}
    	else {
    		this.suppressed = new Throwable[] {exception};
    	}
    }

    public final synchronized Throwable[] getSuppressed() {
    	Throwable[] localSuppressed = this.suppressed;
    	
    	if(localSuppressed != null) {
    		return localSuppressed.clone();
    	}
    	else {
    		return EMPTY_SUPPRESSED_EXCEPTIOON;
    	}
    }

    public void printStackTrace() {
    	this.printStackTrace(System.err, 0, false);
    }

    //This method is called when an exception occurred and an error occurred
    //while printing the stack trace of the stack
    public static native void internalError(Throwable t) ;

    public synchronized Throwable fillInStackTrace() {
    	this.ips = System.getStackTrace(Thread.currentThread());
    	return this;
    }

    @Nullable
    public synchronized Throwable getCause() {
    	return this.cause;
	}

    @Nullable
    public String getLocalizedMessage() {
    	return this.getMessage();
	}

	/**
	 * Provides programmatic access to the stack trace information printed by
	 * {@link #printStackTrace()}. Returns an array of stack trace elements, each representing one
	 * stack frame. The zeroth element of the array (assuming the array's length is non-zero)
	 * represents the top of the stack, which is the last method invocation in the sequence.
	 * Typically, this is the point at which this throwable was created and thrown. The last element
	 * of the array (assuming the array's length is non-zero) represents the bottom of the stack,
	 * which is the first method invocation in the sequence.
	 *
	 * <p>
	 * Some virtual machines may, under some circumstances, omit one or more stack frames from the
	 * stack trace. In the extreme case, a virtual machine that has no stack trace information
	 * concerning this throwable is permitted to return a zero-length array from this method.
	 * Generally speaking, the array returned by this method will contain one element for every
	 * frame that would be printed by {@code printStackTrace}. Writes to the returned array do not
	 * affect future calls to this method.
	 *
	 * @return an array of stack trace elements representing the stack trace pertaining to this
	 *         throwable.
	 */
	public StackTraceElement[] getStackTrace() {
		int[] ips = this.ips;
		if(ips == null) {
			if(this instanceof OutOfMemoryError) {
				// OutOfMemoryError may be an immutable object so its
				// stacktrace cannot be set
				ips = System.stackIpsForOutOfMemoryError();
			}
			else {
				return new StackTraceElement[0];
			}
		}
		return System.getStackTraceElements(ips);
	}

    public synchronized Throwable initCause(@Nullable Throwable cause) {
    	if(cause == this) {
    		throw new IllegalArgumentException();
    	}
    	if(this.cause != null) {
    		throw new IllegalStateException();
    	}
    	this.cause = cause;
    	return this;
	}

    public void printStackTrace(PrintStream s) {
    	printStackTrace(s, 0, false);
    }
    
    public void printStackTrace(PrintStream s, int indentation, boolean printThreadName) {
    	synchronized (Throwable.class) {
    		try
    		{
    			// Print thread name here to keep an atomic print in synchronized statement 
    			// and redirect any errors to internalError() native.
    			if(printThreadName) {
    				s.print("Exception in thread \""); //$NON-NLS-1$
    				s.print(Thread.currentThread().getName());
    				s.print("\" "); //$NON-NLS-1$
    			}
    			
    			if(this instanceof OutOfMemoryError)
    			{
    				s.print("java.lang.OutOfMemoryError"); //$NON-NLS-1$
    			}
    			else
    			{
    				printClassName(s, this);
    			}
    			printMessageStackTrace(s, indentation);
    			
    			Throwable[] suppressedExceptions = getSuppressed();
    			for(Throwable suppressedException : suppressedExceptions) {
    				printIndentation(s, indentation);
    				s.print("\tSuppressed: "); //$NON-NLS-1$
    				suppressedException.printStackTrace(s, indentation + 1, false);
    			}

    			Throwable cause = this.cause;
    			while(cause != null){
    				printIndentation(s, indentation);
    				s.print("Caused by: "); //$NON-NLS-1$
    				printClassName(s, cause);
    				cause.printMessageStackTrace(s, indentation);
    				cause = cause.cause;
    			}
    		}
    		catch(Throwable t)
    		{
    			internalError(this);
    		}
    	}
    }

    private void printClassName(PrintStream s, Throwable t){
    	//may allocate data
		try
		{
			s.print(t.getClass().getName());
		}
		// fail silently
		catch(OutOfMemoryError e) { //NOSONAR

		}
    }

    private void printMessage(PrintStream s){
		String message = getMessage();
		if(message != null)
		{
			s.print(": "); //$NON-NLS-1$
			s.print(message);
		}
    }

    private void printMessageStackTrace(PrintStream s, int indentation){
    	int[] ips = this.ips;
    	if(this instanceof OutOfMemoryError && ips == null) {
    		//OutOfMemoryError may be an immutable object so its
    		//stacktrace cannot be set
			ips = System.stackIpsForOutOfMemoryError();
		}
    	printMessage(s);
		s.println();
		System.printStackTrace(s, ips, indentation);
    }
    
    /*default*/ static void printIndentation(PrintStream s, int indentation) {
    	while(--indentation >= 0) {
    		s.print('\t');
    	}
    }

    @Override
	public String toString() {

    	StringBuilder sb = new StringBuilder(getClass().getName());

    	String message = getLocalizedMessage();
    	if(message != null) {
    		sb.append(": ").append(message); //$NON-NLS-1$
    	}
    	return sb.toString();
    }

}
