/*
 * Java
 *
 * Copyright 2004-2024 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;

import java.io.PrintStream;
import java.util.HashMap;
import java.util.Map;

import com.is2t.vm.support.CalibrationConstants;
import com.is2t.vm.support.InternalLimitsError;

import ej.annotation.Nullable;
import ej.bon.Constants;

public class Thread extends Object implements Runnable {

	/**
	 * A thread state. A thread can be in one of the following states:
	 * <ul>
	 * <li>{@link #NEW}<br>
	 * A thread that has not yet started is in this state.</li>
	 * <li>{@link #RUNNABLE}<br>
	 * A thread executing in the Java virtual machine is in this state.</li>
	 * <li>{@link #BLOCKED}<br>
	 * A thread that is blocked waiting for a monitor lock is in this state.</li>
	 * <li>{@link #WAITING}<br>
	 * A thread that is waiting indefinitely for another thread to perform a particular action is in
	 * this state.</li>
	 * <li>{@link #TIMED_WAITING}<br>
	 * A thread that is waiting for another thread to perform an action for up to a specified
	 * waiting time is in this state.</li>
	 * <li>{@link #TERMINATED}<br>
	 * A thread that has exited is in this state.</li>
	 * </ul>
	 * 
	 * <p>
	 * A thread can be in only one state at a given point in time. These states are virtual machine
	 * states which do not reflect any operating system thread states.
	 * 
	 * @see #getState
	 */
	public enum State {
		/**
		 * Thread state for a thread blocked waiting for a monitor lock. A thread in the blocked
		 * state is waiting for a monitor lock to enter a synchronized block/method or reenter a
		 * synchronized block/method after calling {@link Object#wait() Object.wait}.
		 */
		BLOCKED,

		/**
		 * Thread state for a thread which has not yet started.
		 */
		NEW,

		/**
		 * Thread state for a runnable thread. A thread in the runnable state is executing in the
		 * Java virtual machine but it may be waiting for other resources from the operating system
		 * such as processor.
		 */
		RUNNABLE,

		/**
		 * Thread state for a terminated thread. The thread has completed execution.
		 */
		TERMINATED,

		/**
		 * Thread state for a waiting thread with a specified waiting time. A thread is in the timed
		 * waiting state due to calling one of the following methods with a specified positive
		 * waiting time:
		 * <ul>
		 * <li>{@link #sleep Thread.sleep}</li>
		 * <li>{@link Object#wait(long) Object.wait} with timeout</li>
		 * <li>{@link #join(long) Thread.join} with timeout</li>
		 * </ul>
		 */
		TIMED_WAITING,

		/**
		 * Thread state for a waiting thread. A thread is in the waiting state due to calling one of
		 * the following methods:
		 * <ul>
		 * <li>{@link Object#wait() Object.wait} with no timeout</li>
		 * <li>{@link #join() Thread.join} with no timeout</li>
		 * </ul>
		 * 
		 * <p>
		 * A thread in the waiting state is waiting for another thread to perform a particular
		 * action.
		 * 
		 * For example, a thread that has called <tt>Object.wait()</tt> on an object is waiting for
		 * another thread to call <tt>Object.notify()</tt> or <tt>Object.notifyAll()</tt> on that
		 * object. A thread that has called <tt>Thread.join()</tt> is waiting for a specified thread
		 * to terminate.
		 */
		WAITING;
	}

	public static final int MIN_PRIORITY  = 1 ;
	public static final int NORM_PRIORITY = 5 ;
	public static final int MAX_PRIORITY  = 10 ;
	private static int SomeNameIndex = 0;

	public static final int 	THREAD_NOT_STARTED_ID = -1; //VM known
	public static final int 	THREAD_ENDED_ID = -2; //VM known
	
	private static final String THREAD_STR = "Thread"; //$NON-NLS-1$

	//Use by ej.reflect and console
	private static int PeakThreadCount = 1;//at least main thread //VM Known
	private static int TotalStartedThreadCount = 1;//at least main thread //VM Known

	@Nullable
	private static UncaughtExceptionHandler defaultUncaughtExceptionHandler;

	/* INSTANCE VARIABLES */
	private int id;     //VM known : direct access to native thread
	private int priority ;  //VM known - write access is done in native side in order to let this class transaction safe
	/** True if the thread is a daemon thread, false otherwise. */
	private boolean daemon;  //VM known

	protected String   name ; //VM known
	@Nullable
	private Runnable target ;//VM known
	@Nullable
	private UncaughtExceptionHandler uncaughtExceptionHandler;

	/** Set to true when the thread is terminated. This field is used by join */
	private boolean terminated;

	private static String someName(){
		return THREAD_STR+(++SomeNameIndex);
	}

	public Thread() {
		this(null, someName());
	}

	public Thread(@Nullable Runnable target) {
		this(target, someName());
	}

	public Thread(@Nullable Runnable target, String name) {
		id = THREAD_NOT_STARTED_ID; //not attached to some vmthreading mechanism
		this.priority = currentThread().priority;
		this.name = name;
		this.target = target ;
	}

	public Thread(String name) {
		this(null, name) ;
	}

	public static native int activeCount()
	//returns all alive threads, even those blocked for some reason
	;

	public static native Thread currentThread()
	//this method never returns null (while there is at least one thread that
	//executes the method that has called this native)
	;

	public final String getName() {
		return this.name ;
	}

	/**
	 * Changes the name of this thread to be equal to the argument <code>name</code>.
	 * <p>
	 * First the <code>checkAccess</code> method of this thread is called with no arguments. This
	 * may result in throwing a <code>SecurityException</code>.
	 * 
	 * @param name
	 *        the new name for this thread.
	 * @exception SecurityException
	 *            if the current thread cannot modify this thread.
	 * @see #getName
	 */
	public final void setName(String name) {
		if (Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
			checkAccess();
		}
		this.name = name;
	}


	public final int getPriority() {
		return this.priority ;
	}

	private native void interruptNative() ;
	public void interrupt() {
		if (Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
	        if (this != Thread.currentThread()){
	            checkAccess();
	        }
		}
        
		if(isAlive()) {
			interruptNative();
		}
	}

	public static native boolean interrupted(); //NOSONAR

	public native boolean isInterrupted();

	public final boolean isAlive(){
		return id>=0  && !terminated;
	}


	public final void join() throws InterruptedException{
		join(0);
	}

	/**
	 * Waits at most {@code millis} milliseconds for this thread to die. A timeout of {@code 0}
	 * means to wait forever.
	 * 
	 * <p>
	 * This implementation uses a loop of {@code this.wait} calls conditioned on
	 * {@code this.isAlive}. As a thread terminates the {@code this.notifyAll} method is invoked. It
	 * is recommended that applications not use {@code wait}, {@code notify}, or {@code notifyAll}
	 * on {@code Thread} instances.
	 * 
	 * @param millis
	 *        the time to wait in milliseconds
	 * 
	 * @throws IllegalArgumentException
	 *         if the value of {@code millis} is negative
	 * 
	 * @throws InterruptedException
	 *         if any thread has interrupted the current thread. The <i>interrupted status</i> of
	 *         the current thread is cleared when this exception is thrown.
	 */
	public final synchronized void join(long millis) throws InterruptedException {
		if(millis == 0){
			//wait infinitely
			while(isAlive()){
				this.wait();
			}
		}
		else {
			//wait for at most millis ms
			long t0 = System.currentTimeMillis();
			long delay = millis;
			while(isAlive()){
				this.wait(delay);

				//check if timeout has been reached
				long exhausted = System.currentTimeMillis() - t0;
				delay = millis - exhausted;
				if(delay <= 0){
					return;
				}
			}
		}
	}

	/**
	 * Waits at most {@code millis} milliseconds plus {@code nanos} nanoseconds for this thread to
	 * die.
	 * 
	 * <p>
	 * This implementation uses a loop of {@code this.wait} calls conditioned on
	 * {@code this.isAlive}. As a thread terminates the {@code this.notifyAll} method is invoked. It
	 * is recommended that applications not use {@code wait}, {@code notify}, or {@code notifyAll}
	 * on {@code Thread} instances.
	 * 
	 * @param millis
	 *        the time to wait in milliseconds
	 * 
	 * @param nanos
	 *        {@code 0-999999} additional nanoseconds to wait
	 * 
	 * @throws IllegalArgumentException
	 *         if the value of {@code millis} is negative, or the value of {@code nanos} is not in
	 *         the range {@code 0-999999}
	 * 
	 * @throws InterruptedException
	 *         if any thread has interrupted the current thread. The <i>interrupted status</i> of
	 *         the current thread is cleared when this exception is thrown.
	 */
	public final synchronized void join(long millis, int nanos) throws InterruptedException {

		//SEE comment in sleep(long,int)

		if (nanos >= 0 && nanos <= 999999 && millis>=0) {
			if (nanos < 500000) {
				if (millis == 0 && nanos != 0){
					join(1) ;
				}else{
					join(millis) ;
				}
			} else{
				join(millis+1) ;
			}
		} else {
			throw new IllegalArgumentException();
		}
	}

	@Override
	public void run() {
		Runnable target = this.target;
		if (target!=null) {
			target.run();
		}
	}

	// precond newPriority>= Thread.MIN_PRIORITY && newPriority <= Thread.MAX_PRIORITY
	private native void setPriorityNative(int newPriority);

	public final void setPriority(int newPriority) {
		if (Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
			checkAccess();
		}
		if(newPriority < Thread.MIN_PRIORITY || newPriority > Thread.MAX_PRIORITY) {
			throw new IllegalArgumentException();
		}
		setPriorityNative(newPriority);
	}

	public static void sleep(long millis) throws InterruptedException{
		if(millis < 0) {
			throw new IllegalArgumentException();
		}
		opSleep(millis);
	}

	/**
	 * Causes the currently executing thread to sleep (temporarily cease execution) for the
	 * specified number of milliseconds plus the specified number of nanoseconds, subject to the
	 * precision and accuracy of system timers and schedulers. The thread does not lose ownership of
	 * any monitors.
	 * 
	 * @param millis
	 *        the length of time to sleep in milliseconds
	 * 
	 * @param nanos
	 *        {@code 0-999999} additional nanoseconds to sleep
	 * 
	 * @throws IllegalArgumentException
	 *         if the value of {@code millis} is negative, or the value of {@code nanos} is not in
	 *         the range {@code 0-999999}
	 * 
	 * @throws InterruptedException
	 *         if any thread has interrupted the current thread. The <i>interrupted status</i> of
	 *         the current thread is cleared when this exception is thrown.
	 */
	public static void sleep(long millis, int nanos) throws InterruptedException {
		//
		// Note : 0 < nanos < 999.999
		//
		// If the amount of nanoseconds is less than 500.000, so we simply
		// call the wait(timeout) unless timeout is equal to zero.
		//
		// If the amout is greater than 500000 (ie. 0,5ms) we increase the timeout
		// by 1 and then call the wait(timeout) method.

		if (nanos >= 0 && nanos <= 999999 && millis>=0) {
			if (nanos < 500000) {
				if (millis == 0 && nanos != 0){
					sleep(1) ;
				}else{
					sleep(millis) ;
				}
			} else{
				sleep(millis+1) ;
			}
		} else {
			throw new IllegalArgumentException();
		}
	}

	//VM/Soar Known : converted in opcode
	private static native void opSleep(long millis);

	public synchronized void start(){
		if(id != THREAD_NOT_STARTED_ID) {
			throw new IllegalThreadStateException();//thread already started
		}

		int errorCode = startNative();
		if(errorCode != 0){
			if(errorCode == -1) {
				throw new InternalLimitsError("Too many alive java threads ("+Thread.activeCount()+')'); //$NON-NLS-1$
			}
			else if(errorCode == -2) {
				throw new OutOfMemoryError("Stacks space"); //$NON-NLS-1$
			}
			else {
				throw new InternalLimitsError(String.valueOf(errorCode));
			}
		}

		updateThreadCounters();
	}

	private native int startNative();

	//call this method every time a thread is started
	private static synchronized void updateThreadCounters(){

		++TotalStartedThreadCount;
		int activeCount = activeCount();
		if(activeCount > PeakThreadCount) {
			PeakThreadCount = activeCount;
		}
	}

	public static synchronized void resetPeakThreadCount(){
		Thread.PeakThreadCount = Thread.activeCount();
	}

	@Override
	public String toString() {
		return new StringBuilder(THREAD_STR)
		.append('[')
		.append(this.name)
		.append(',')
		.append(this.priority)
		.append(']')
		.toString() ;
	}

	public static void yield(){
		//this method is replaced by an opcode
		opYield();
	}
	//VM/Soar Known : converted in opcode
	private static native void opYield();


	//-----------------jvm specific methods and data-----------------//

	//VM known
	private static void callWrapper() { //NOSONAR
		//Native prepareCallMethod() pushes a frame of this method,
		//in order to catch all thrown exception and get out of this engine (via breakrun opcode)

		//Its objectif is to catch uncaught exception and to store back the exception into the native world
		//When this method has finished, we go back to the native world
		try{
			breakRun();
			breakRun(); //ip is set here by the vm (the previous breakRun is just to simulate an invoke opcode
			//in order to be in the exception catch handler, see MowanaVMEngine.prepareCallMethod and MowanaVMEngine.uncaughtException),
			//when returning from the method, the breakrun opcode is executed.
			return;
		}catch(Throwable t){
			storeException(t);
		}
		breakRun();
		return;
	}

	//VM Known: entry point for all java threads
	private void runWrapper(){
		try{
			run();
		}catch(Throwable t){
			handleUncaughtException(t);
		}
		finally {
			//notify all the threads that have call join
			synchronized (this) {
				this.terminated = true;
				this.notifyAll();
			}
		}
	}

	//VM Known : this method is called by the VM to execute the clinits.
	//It is not the first frame of the stack (a runWrapper is called before this method)
	public static void clinitWrapper(int nbClinits){
		try
		{
			for(int clinitIndex = 0; clinitIndex < nbClinits; ++clinitIndex) {
				execClinit(clinitIndex);
			}
		}
		catch(RuntimeException t) {
			// all runtime exceptions that occurred in clinit are converted in
			// ExceptionInInitializerError
			// See jvms7.pdf - Section 5.5 - Initialization
			throw new ExceptionInInitializerError(t);
		}
		//error thrown during clinit will be caught by runWrapper method
	}

	/**
	 * Execute the clinit method at the given index
	 * @param clinitMethodPtr the 0 based index in the clinit table of the clinit method to execute
	 */
	private static native void execClinit(int clinitIndex);


	//store the exception back into the native world
	protected static native void storeException(Throwable t);

	//this must be replaced by a 'breakrun' opcode
	static native public void breakRun(); //NOSONAR //special native opcode

	// VM Known
	/*default*/ static void main(String[] args)
	{
		//call to this method are replaced by a call to main method
	}

	public interface UncaughtExceptionHandler {
		void uncaughtException(Thread t, Throwable e);
	}

	@Nullable 
	public static UncaughtExceptionHandler getDefaultUncaughtExceptionHandler() {
		return defaultUncaughtExceptionHandler;
	}

	public static void setDefaultUncaughtExceptionHandler(@Nullable UncaughtExceptionHandler eh) {
		if (Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
			SecurityManager sm = System.getSecurityManager();
			if(sm != null){
				sm.checkPermission(new RuntimePermission("setDefaultUncaughtExceptionHandler")); //$NON-NLS-1$
			}
		}
		defaultUncaughtExceptionHandler= eh;
	}

	@Nullable 
	public UncaughtExceptionHandler getUncaughtExceptionHandler() {
		return uncaughtExceptionHandler;
	}

	public void setUncaughtExceptionHandler(@Nullable UncaughtExceptionHandler eh) {
		if (Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
			checkAccess();
		}
		uncaughtExceptionHandler = eh;
	}

	protected static void handleUncaughtException(Throwable t){
		Thread thread = Thread.currentThread();
		Thread.UncaughtExceptionHandler uncaughtExceptionHandler = thread.uncaughtExceptionHandler;
		try{
			if(uncaughtExceptionHandler != null){
				uncaughtExceptionHandler.uncaughtException(thread, t);
			}
			else {
				UncaughtExceptionHandler h = defaultUncaughtExceptionHandler;
				if(h != null){
					h.uncaughtException(thread, t);
				}else{
					t.printStackTrace(System.err, 0, true);
				}
			}
		}catch(Throwable t2){
			// here exception thrown in uncaughtException handler - just print it
			t2.printStackTrace();
		}
	}

	/**
	 * Prints a stack trace of the current thread to the standard error stream. This method is used
	 * only for debugging.
	 * 
	 * @see Throwable#printStackTrace()
	 */
	public static void dumpStack() {
		new Throwable().printStackTrace();
	}

	/**
	 * Copies into the specified array every active thread.
	 * 
	 * <p>
	 * An application might use the {@linkplain #activeCount activeCount} method to get an estimate
	 * of how big the array should be, however <i>if the array is too short to hold all the threads,
	 * the extra threads are silently ignored.</i> If it is critical to obtain every active thread
	 * in the current thread's thread group and its subgroups, the invoker should verify that the
	 * returned int value is strictly less than the length of {@code tarray}.
	 * 
	 * <p>
	 * Due to the inherent race condition in this method, it is recommended that the method only be
	 * used for debugging and monitoring purposes.
	 * 
	 * @param tarray
	 *        an array into which to put the list of threads
	 * 
	 * @return the number of threads put into the array
	 * 
	 * @throws SecurityException
	 *         if the thread cannot access all the threads
	 */
	native public static int enumerate(Thread tarray[]);

	/**
	 * Returns a map of stack traces for all live threads. The map keys are threads and each map
	 * value is an array of <tt>StackTraceElement</tt> that represents the stack dump of the
	 * corresponding <tt>Thread</tt>. The returned stack traces are in the format specified for the
	 * {@link #getStackTrace getStackTrace} method.
	 * 
	 * <p>
	 * The threads may be executing while this method is called. The stack trace of each thread only
	 * represents a snapshot and each stack trace may be obtained at different time. A zero-length
	 * array will be returned in the map value if the virtual machine has no stack trace information
	 * about a thread.
	 * 
	 * <p>
	 * If there is a security manager, then the security manager's <tt>checkPermission</tt> method
	 * is called with a <tt>RuntimePermission("getStackTrace")</tt> permission to see if it is ok to
	 * get the stack trace of all threads.
	 * 
	 * @return a <tt>Map</tt> from <tt>Thread</tt> to an array of <tt>StackTraceElement</tt> that
	 *         represents the stack trace of the corresponding thread.
	 * 
	 * @throws SecurityException
	 *         if a security manager exists and its <tt>checkPermission</tt> method doesn't allow
	 *         getting the stack trace of thread.
	 * @see #getStackTrace
	 * @see SecurityManager#checkPermission
	 * @see RuntimePermission
	 * @see Throwable#getStackTrace
	 * 
	 */
	public static Map<Thread, StackTraceElement[]> getAllStackTraces() {
		if (Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
			checkPermissionGetStackTrace();
		}

		Thread[] allThreads = new Thread[activeCount()];
		int nbThreads = enumerate(allThreads);

		Map<Thread, StackTraceElement[]> map = new HashMap<>();
		for(int i=0 ; i<nbThreads ; ++i){
			map.put(allThreads[i], allThreads[i].getStackTrace());
		}

		return map;
	}

	/** 
	 * Any call to this method must be surrounded by a check
	 * of {@link CalibrationConstants#CONSTANT_USE_SECURITYMANAGER}.
	 */
	private static void checkPermissionGetStackTrace(){
		SecurityManager sm = System.getSecurityManager();
		if(sm != null){
			sm.checkPermission(new RuntimePermission("getStackTrace")); //$NON-NLS-1$
		}
	}

	/**
	 * Returns an array of stack trace elements representing the stack dump of this thread. This
	 * method will return a zero-length array if this thread has not started, has started but has
	 * not yet been scheduled to run by the system, or has terminated. If the returned array is of
	 * non-zero length then the first element of the array represents the top of the stack, which is
	 * the most recent method invocation in the sequence. The last element of the array represents
	 * the bottom of the stack, which is the least recent method invocation in the sequence.
	 * 
	 * <p>
	 * If there is a security manager, and this thread is not the current thread, then the security
	 * manager's <tt>checkPermission</tt> method is called with a
	 * <tt>RuntimePermission("getStackTrace")</tt> permission to see if it's ok to get the stack
	 * trace.
	 * 
	 * <p>
	 * Some virtual machines may, under some circumstances, omit one or more stack frames from the
	 * stack trace. In the extreme case, a virtual machine that has no stack trace information
	 * concerning this thread is permitted to return a zero-length array from this method.
	 * 
	 * @return an array of <tt>StackTraceElement</tt>, each represents one stack frame.
	 * 
	 * @throws SecurityException
	 *         if a security manager exists and its <tt>checkPermission</tt> method doesn't allow
	 *         getting the stack trace of thread.
	 * @see SecurityManager#checkPermission
	 * @see RuntimePermission
	 * @see Throwable#getStackTrace
	 * 
	 */
	public StackTraceElement[] getStackTrace() {
		if (Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
			if(this != Thread.currentThread()){
				checkPermissionGetStackTrace();
			}
		}
		return System.getStackTraceElements(this);
	}

	/**
	 * Returns the state of this thread. This method is designed for use in monitoring of the system
	 * state, not for synchronization control.
	 * 
	 * @return this thread's state.
	 */
	public State getState(){
		switch(getStateNative()){
			case STATE_NEW:
				return State.NEW;
			case STATE_READY:
			case STATE_RUNNING:
				return State.RUNNABLE;
			case STATE_WAITING:
				return State.WAITING;
			case STATE_MONITOR_QUEUED:
				return State.BLOCKED;
			case STATE_TERMINATED:
			default: //should not occur
				return State.TERMINATED;
		}
	}

	/** Thread state for a thread which has not yet started. */
	private static final int STATE_NEW            = 0;//VM known
	/** Thread state for a runnable thread. A thread in the runnable state is executing
	 * in the Java virtual machine but is waiting from the operating system the
	 * processor resource. */
	private static final int STATE_READY          = 1;//VM known
	/** Thread state for a runnable thread. A thread in the runnable state is executing
	 * in the Java virtual machine and the processor is assigned to it. */
	private static final int STATE_RUNNING        = 2;//VM known
	/**
	 * Thread state for a waiting thread. A thread is in the waiting state due to calling one of the following methods:
	 * <ul>
	 *   <li><code>Object.wait</code></li>
	 *   <li><code>Thread.join</code></li>
	 *   <li><code>Thread.sleep</code></li>
	 * </ul>
	 */
	private static final int STATE_WAITING        = 3;//VM known
	/**Thread state for a thread blocked waiting for a monitor lock.
	 * A thread in the monitor queued state is waiting for a monitor lock to enter
	 * a synchronized block/method or reenter a synchronized block/method after
	 * calling <code>Object.wait</code>.
	 */
	private static final int STATE_MONITOR_QUEUED = 4;//VM known
	/**
	 * Thread state for a terminated thread.
	 * The thread has completed execution.
	 */
	private static final int STATE_TERMINATED     = 5;//VM Known

	/** Returns one of the STATE_xxx */
	native private int getStateNative();

	/**
	 * Determines if the currently running thread has permission to
	 * modify this thread.
	 * <p>
	 * If there is a security manager, its <code>checkAccess</code> method
	 * is called with this thread as its argument. This may result in
	 * throwing a <code>SecurityException</code>.
	 *
	 * @exception  SecurityException  if the current thread is not allowed to
	 *               access this thread.
	 * @see        SecurityManager#checkAccess(Thread)
	 */
	public final void checkAccess() {
		if (Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
			SecurityManager securityManager = System.getSecurityManager();
			if (securityManager != null) {
				securityManager.checkAccess(this);
			}
		}
	}

	/**
	 * Returns <tt>true</tt> if and only if the current thread holds the monitor lock on the
	 * specified object.
	 * 
	 * <p>
	 * This method is designed to allow a program to assert that the current thread already holds a
	 * specified lock:
	 * 
	 * <pre>
	 * assert Thread.holdsLock(obj);
	 * </pre>
	 * 
	 * @param obj
	 *        the object on which to test lock ownership
	 * @throws NullPointerException
	 *         if obj is <tt>null</tt>
	 * @return <tt>true</tt> if the current thread holds the monitor lock on the specified object.
	 */
	native public static boolean holdsLock(Object obj) ;


	/**
	 * Returns the identifier of this Thread. The thread ID is a positive <tt>long</tt> number
	 * generated when this thread was created. The thread ID is unique and remains unchanged during
	 * its lifetime. When a thread is terminated, this thread ID may be reused.
	 * 
	 * @return this thread's ID.
	 */
	public long getId() {
		return id;
	}

	/**
	 * Marks this thread as either a {@linkplain #isDaemon daemon} thread
	 * or a user thread. The Java Virtual Machine exits when the only
	 * threads running are all daemon threads.
	 *
	 * <p> This method must be invoked before the thread is started.
	 *
	 * @param  on
	 *         if {@code true}, marks this thread as a daemon thread
	 *
	 * @throws  IllegalThreadStateException
	 *          if this thread is {@linkplain #isAlive alive}
	 *
	 * @throws  SecurityException
	 *          if {@link #checkAccess} determines that the current
	 *          thread cannot modify this thread
	 */
	public final void setDaemon(boolean on) {
		if (Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
			checkAccess();
		}
		if (isAlive()) {
			throw new IllegalThreadStateException();
		}
		daemon = on;
	}

	/**
	 * Tests if this thread is a daemon thread.
	 *
	 * @return  <code>true</code> if this thread is a daemon thread;
	 *          <code>false</code> otherwise.
	 * @see     #setDaemon(boolean)
	 */
	public final boolean isDaemon() {
		return daemon;
	}

}

class MainThread extends Thread{
	//Class of the main thread. Instance of this class is created by the VM.
	//VM should set the name of instances of this class. This name can be MainThreadName.

	// Known by the VM
	@Nullable
	private static String[] Args;

	@Override
	public void run(){
		String[] args = Args;
		assert args != null; // has been previously initialized by the VM
		Thread.main(args);
	}


}

