/*
 * Java
 *
 * Copyright 2004-2024 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;

import java.io.OutputStream;
import java.io.PrintStream;
import java.util.HashMap;
import java.util.PropertyPermission;

import com.is2t.vm.support.CalibrationConstants;

import ej.annotation.Nullable;
import ej.bon.Constants;

public final class System extends Object {


	private static final String SYSTEM_OUT_CLASS_PROPERTY = "com.is2t.system.out.class" ; //$NON-NLS-1$

	public static final PrintStream out;

	public static native OutputStream getOutputStream();

	// Put LineSep here to avoid cycle between System and PrintStream
	public static final String LineSep;
	private static final String StackFrameStart = "\tat "; //$NON-NLS-1$
	/*default*/ static final String StackFrameIPStart = ":0x"; //$NON-NLS-1$
	/*default*/ static final char StackFrameEnd = '@';
	private static final String RemainingFrameStart = "\t...\n"; //$NON-NLS-1$
	private static final String RemainingFrameEnd = " remaining frame(s).\n"; //$NON-NLS-1$

	/*
	 * System properties.
	 * The following properties are guaranteed to be defined:
	 * java.version           = Java version number
	 * java.vendor            = Java vendor specific string
	 * java.vendor.url        = Java vendor URL
	 * java.home              = Java installation directory
	 * java.class.version     = Java class version number
	 * java.class.path        = Java classpath
	 * os.name                = Operating System Name
	 * os.arch                = Operating System Architecture
	 * os.version             = Operating System Version
	 * file.separator         = File separator ("/" on Unix)
	 * path.separator         = Path separator (":" on Unix)
	 * line.separator         = Line separator ("\n" on Unix)
	 * user.name              = User account name
	 * user.home              = User home directory
	 * user.dir               = User's current working directory
	 * microedition.package   = package used by Connector to build connection
	 * microedition.commports = valid identifiers for a particular device and OS, a comma separated list of ports
	 * microedition.locale    = The current locale of the device, may be null
	 * microedition.profiles  = is a blank (Unicode U+0020) separated list of the J2ME profiles that this device supports.
	 */

	private static String[] PropertiesKeys;
	private static String[] PropertiesValues;

	static {
		String[][] wrapper = new String[2][];
		initializeProperties(wrapper);
		String[] pKeys = wrapper[0];
		assert pKeys != null;
		PropertiesKeys = pKeys;
		String[] pValues = wrapper[1];
		assert pValues != null;
		PropertiesValues = pValues;

		java.io.OutputStream tmp = buildOutputStream();
		out = new java.io.PrintStream(tmp);

		//        // DEBUG PURPOSE (uncomment to dump the list of system properties)
		//        for(int i=PropertiesKeys.length; --i>=0;){
		//        	out.println(PropertiesKeys[i]+"="+PropertiesValues[i]);
		//        }

		String lineSep = getPropertyNoCheck("line.separator", "\n"); //$NON-NLS-1$ //$NON-NLS-2$
		assert lineSep != null;
		LineSep = lineSep;
	}

	private System()
	{
		//CANNOT BE INSTANTIATED
	}

	public static final PrintStream err = out ;

	public static native void arraycopy(Object src, int srcOffset, Object dst,
			int dstOffset, int length) ;

	// public static native void arraycopy(int[] src, int srcOffset,
	//         int[] dst, int dstOffset, int length) ;

	// public static native void arraycopy(byte[] src, int srcOffset,
	//          byte[] dst, int dstOffset, int length) ;

	public static native long currentTimeMillis();

	public static void exit(int status) {
		//see important notes in Runtime.exit(int)
		Runtime.instance.exit(status);
	}

	public static void gc(){
		Runtime.getRuntime().gc();
	}

	private static native void initializeProperties(String[][] wrapper);

	/*
	 * Implementation notes:
	 * To avoid clinit cycles with permissions,
	 * EDC implementation shall directly call getPropertyNoCheck when property is a well known internal string literal.
	 */
	@Nullable 
	public static String getProperty(String key) {
		if (Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
			SecurityManager sm = getSecurityManager();
			if(sm != null){
				sm.checkPermission(new PropertyPermission(key, PropertyPermission.ACTION_READ));
			}
		}
		if(key.length==0) {
			throw new IllegalArgumentException();
		}

		return getPropertyNoCheck(key);
	}

	/*
	 * @see #getProperty comment
	 * Assume key is not null and not empty
	 * No permission check.
	 */
	@Nullable
	public static String getPropertyNoCheck(String key){
		String[] keys = PropertiesKeys;

		for(int i=keys.length ; --i>=0 ;)
		{	// search value at key
			if(keys[i].equals( key)) {
				return PropertiesValues[i];
			}
		}
		//key not found
		return null;
	}

	@Nullable
	public static String getPropertyNoCheck(String key, @Nullable String def) {
		String result = getPropertyNoCheck(key);
		if(result == null) {
			return def;
		}
		return result;
	}
	
	public static String getProperty(String key, String def) {
		String result = getProperty(key);
		if(result == null) {
			return def;
		}

		return result;
	}

	public static int identityHashCode(@Nullable Object x) {
		if(x==null) {
			return 0;
		}
		return hashCode(x);
	}
	public static native int hashCode(Object x);


	private static OutputStream buildOutputStream()
	{
		try
		{
			String sysOutClass = getPropertyNoCheck(SYSTEM_OUT_CLASS_PROPERTY);	// assume the properties have been initialized
			assert sysOutClass != null;
			sysOutClass = sysOutClass.trim();
			Class<?> sysOut = Class.forName(sysOutClass);
			return (OutputStream) sysOut.newInstance();
		}
		catch (Exception e)
		{
			// an error occurs
			// NullPointerException, ClassNotFoundException, ClassCastException, ...
			return getOutputStream();
		}
	}

	/*
	 * Non CLDC API methods, but intern implementation API.
	 */

	private static Object InternedStringsMonitor = new Object();
	@Nullable
	private static HashMap<String, String> InternedStrings;
	/*
	 * See specification in String.intern()
	 */
	public static String intern(String source){
		synchronized(InternedStringsMonitor){
			HashMap<String, String> internedStrings = InternedStrings;
			if(internedStrings == null){
				//create the hash table and initialize it
				internedStrings = new HashMap<String, String>(getNumberOfInternedStrings());
				for(String s=nextInternedString();s!=null;s=nextInternedString()){
					internedStrings.put(s,s);
				}
				InternedStrings = internedStrings;
			}
			String result = internedStrings.get(source);
			if(result==null){
				//need to create a new interned string
				result = source;
				internedStrings.put(result,result);
			}
			return result;
		}
	}

	// go through the area of internedStrings
	@Nullable
	private static native String nextInternedString();
	//return the number of literal string
	private static native int getNumberOfInternedStrings();

	/*
	 * Write the given stack trace in the given printstream.
	 * First element of stackTrace array is the number of stackFrame of the thread stack.
	 * Then this is a list of pairs of ips,methodAddr.
	 * If (stackTrace.length-1)/2 is lower than nbStackFrames, this means that stackTrace
	 * array does not contain all the stack frames. In this case, the number of remaining frames
	 * may be printed.
	 */
	public static final int MAX_LINE_CHARS = 256;
	public static void printStackTrace(PrintStream out, @Nullable int[] stackTrace, int indentation){

		if(stackTrace != null)
		{
			int nbStackFrames = stackTrace[0];
			int ipsLength = stackTrace.length;
			int nbStackFramesInStackTrace = (ipsLength-1) / 2;

			//compute number of frame that are not present in the stackTrace
			int remainingFrames = nbStackFrames - nbStackFramesInStackTrace;

			int end;//last stack frame offset+1 to print from the stackTrace array
			if(remainingFrames >= 0) {
				end = ipsLength;
			}
			else {
				end = (nbStackFrames*2)+1;
			}

			int i = 1;
			char[] bufferLine = new char[MAX_LINE_CHARS];
			while(i < end)
			{
				Throwable.printIndentation(out, indentation);
				int ip = stackTrace[i++];
				int methodAddr = stackTrace[i++];
				out.print(StackFrameStart);

				//print classname if class can be found
				Class<?> methodClass = getMethodClass(methodAddr);
				if(methodClass != null){
					out.print(methodClass.getName());
					out.print('.');
				}

				int nbChars = toStringStackLine(bufferLine, ip, methodAddr);
				for(int c=-1; ++c<nbChars;) {
					out.print(bufferLine[c]);
				}
				out.print('\n');
			}
			if(remainingFrames > 0) {
				Throwable.printIndentation(out, indentation);
				out.print(RemainingFrameStart);
				out.print(remainingFrames);
				out.print(RemainingFrameEnd);
			}
		}
	}

	/**
	 * Returns the current value of the running Java Virtual Machine's high-resolution time source,
	 * in nanoseconds.
	 *
	 * <p>
	 * This method can only be used to measure elapsed time and is not related to any other notion
	 * of system or wall-clock time. The value returned represents nanoseconds since some fixed but
	 * arbitrary <i>origin</i> time (perhaps in the future, so values may be negative). The same
	 * origin is used by all invocations of this method in an instance of a Java virtual machine;
	 * other virtual machine instances are likely to use a different origin.
	 *
	 * <p>
	 * This method provides nanosecond precision, but not necessarily nanosecond resolution (that
	 * is, how frequently the value changes) - no guarantees are made except that the resolution is
	 * at least as good as that of {@link #currentTimeMillis()}.
	 *
	 * <p>
	 * Differences in successive calls that span greater than approximately 292 years
	 * (2<sup>63</sup> nanoseconds) will not correctly compute elapsed time due to numerical
	 * overflow.
	 *
	 * <p>
	 * The values returned by this method become meaningful only when the difference between two
	 * such values, obtained within the same instance of a Java virtual machine, is computed.
	 *
	 * <p>
	 * For example, to measure how long some code takes to execute:
	 *
	 * <pre>
	 * {
	 * 	&#064;code
	 * 	long startTime = System.nanoTime();
	 * 	// ... the code being measured ...
	 * 	long estimatedTime = System.nanoTime() - startTime;
	 * }
	 * </pre>
	 *
	 * <p>
	 * To compare two nanoTime values
	 *
	 * <pre>
	 * {@code
	 * long t0 = System.nanoTime();
	 * ...
	 * long t1 = System.nanoTime();}
	 * </pre>
	 *
	 * one should use {@code t1 - t0 < 0}, not {@code t1 < t0}, because of the possibility of
	 * numerical overflow.
	 *
	 * @return the current value of the running Java Virtual Machine's high-resolution time source,
	 *         in nanoseconds
	 */
	native public static long nanoTime();

	/*
	 * Returns the String representation of the given stackTrace.
	 * stackTrace is a list of ips,methodAddr.
	 * First element of stackTrace array is the number of stackFrame of the thread stack.
	 * Then this is a list of pairs of ips,methodAddr.
	 * If (stackTrace.length-1)/2 is lower than nbStackFrames, this means that stackTrace
	 * array does not contain all the stack frames. In this case, the number of remaining frames
	 * may be printed.
	 */
	public static String toStringStackTrace(@Nullable int[] stackTrace){
		StringBuilder sb = new StringBuilder();

		if(stackTrace != null)
		{
			int nbStackFrames = stackTrace[0];
			int ipsLength = stackTrace.length;
			int nbStackFramesInStackTrace = (ipsLength-1) / 2;

			//compute number of frame that are not present in the stackTrace
			int remainingFrames = nbStackFrames - nbStackFramesInStackTrace;

			int end;//last stack frame offset+1 to print from the stackTrace array
			if(remainingFrames >= 0) {
				end = ipsLength;
			}
			else {
				end = (nbStackFrames*2)+1;
			}

			int i = 1;
			char[] bufferLine = new char[MAX_LINE_CHARS];
			while(i < end)
			{
				int ip = stackTrace[i++];
				int methodAddr = stackTrace[i++];
				sb.append(StackFrameStart);

				//append classname if class can be found
				Class<?> methodClass = getMethodClass(methodAddr);
				if(methodClass != null){
					sb.append(methodClass.getName())
					.append('.');
				}

				int nbChars = toStringStackLine(bufferLine, ip, methodAddr);
				sb.append(bufferLine, 0, nbChars)
				.append('\n');
			}
			if(remainingFrames > 0) {
				sb.append(RemainingFrameStart)
				.append(remainingFrames)
				.append(RemainingFrameEnd);
			}
		}

		return sb.toString();
	}

	/* Returns the stack trace of a thread.
	 * First element of stackTrace array is the number of stackFrame of the thread stack.
	 * Then this is a list of pairs of ips,methodAddr. The number of pairs may be lower
	 * than the number of stack frame.
	 *
	 * Returns null if the thread is terminated or not started
	 */
	@Nullable
	public static native int[] getStackTrace(Thread thread);

	//returns the class of the given method
	//May return null if the class cannot be found
	@Nullable
	public static native Class<?> getMethodClass(int methodAddr);

	/*
	 * Fill into buffer a string representation of the stack line described by its ip and method address
	 * @return the number of chars filled in buffer
	 */
	public static native int toStringStackLine(char[] buffer, int ip, int address);

	public static native final int[] stackIpsForOutOfMemoryError();

	/*default*/ static StackTraceElement[] getStackTraceElements(Thread thread){
		int[] ips = getStackTrace(thread);
		if(ips == null){
			return new StackTraceElement[0];
		}

		return getStackTraceElements(ips);
	}

	/*default*/ static StackTraceElement[] getStackTraceElements(int[] ips){
		int totalNbStackFrames = ips[0];
		int ipsLength = ips.length;
		int nbStackFramesInStackTrace = (ipsLength-1) / 2;
		
		int arrayLength = Math.min(totalNbStackFrames, nbStackFramesInStackTrace);
		int end = (arrayLength*2)+1;
		
		StackTraceElement[] stackTrace = new StackTraceElement[arrayLength];
		int i = 1;
		int frameIndex = 0;
		while(i < end){
			stackTrace[frameIndex++] = new StackTraceElement(ips[i+1], ips[i]);
			i += 2;
		}
		return stackTrace;
	}

	@Nullable 
	private static SecurityManager SecurityManager;
	public static void setSecurityManager(@Nullable SecurityManager s) {
		if (Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
			java.lang.SecurityManager currentSecurityManager = SecurityManager;
			if(currentSecurityManager != null){
				currentSecurityManager.checkPermission(new RuntimePermission("setSecurityManager")); //$NON-NLS-1$
			}
		}
		SecurityManager = s;
	}

	@Nullable 
	public static SecurityManager getSecurityManager() {
		return SecurityManager;
	}
}
