/*
 * Java
 * 
 * Copyright 2008-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;

import com.is2t.vm.support.err.EDCErrorMessages;
import com.is2t.vm.support.lang.SupportNumber;

import ej.annotation.Nullable;
import ej.error.Message;

public final class Long extends Number implements Comparable<Long> {

	public static final long MAX_VALUE = 0x7fffffffffffffffL;

	public static final long MIN_VALUE = 0x8000000000000000L;

	public static final int SIZE = 64;

//	public static int bitCount(long i) {
//	}
	
	/**
	 * @accelerable
	 */
	public static int compare(long x, long y) {
		if(x == y) {
			return 0;
		}
		else if(x < y) {
			return -1;
		}
		else {
			return 1;
		}
	}
	
	// Cyclomatic Complexity : Cyclomatic Complexity is 13 (max allowed is 10).
	// acceptable here
	public static Long decode(String nm) throws NumberFormatException { //NOSONAR
		// Check if arg string is a valid string
		if(nm.length == 0) {
			throw new NumberFormatException(Message.at(new EDCErrorMessages(), EDCErrorMessages.EmptyString));
		}
		
		boolean isPositive = true;
		int radix = 10;
		int startPos = 0; //Start pos. of the number in the string, after the sign char.
		
		// Check the sign of the number
		if(nm.startsWith("-")) { //$NON-NLS-1$
			startPos++;
			isPositive = false;
		}
		else if(nm.startsWith("+")) {//$NON-NLS-1$
			startPos++;
		}
		
		// Check radix
		if(nm.startsWith("0X", startPos) || nm.startsWith("0x", startPos)) {//$NON-NLS-1$ //$NON-NLS-2$
			// Hex --> 16
			startPos += 2;
			radix = 16;
		}
		else if(nm.startsWith("#", startPos)) {//$NON-NLS-1$
			// Hex too --> 16
			startPos++;
			radix = 16;
		}
		// Octal form if it starts with '0' and it is not the value zero
		// In other words, if it starts with 0 and there is at least one digit left.
		// Check octal after hexa because it starts with a zero
		else if(nm.startsWith("0", startPos) && (nm.length - (startPos + 1)) > 0) {//$NON-NLS-1$
			// Octal --> 8
			startPos++;
			radix = 8;
		}
		
		// Check if the remaining string starts with a '-' or a '+'. If true,
		// string is not well formed (bad place for the sign)
		if(nm.startsWith("+", startPos) || nm.startsWith("-", startPos)) {//$NON-NLS-1$ //$NON-NLS-2$
			throw new NumberFormatException(nm);
		}
		
		// Integer.valueOf will throw an error if the remaining string is not well formed
		return new Long(Long.valueOf(nm.substring(startPos), radix).longValue() * (isPositive ? 1L : -1L));
	}
	
	@Nullable 
	public static Long getLong(String nm) {
		return getLong(nm, null);
	}
	
	public static Long getLong(String nm, long val) {
		Long result = getLong(nm, new Long(val));
		assert result != null;
		return result;
	}
	
	public static Long getLong(String nm, Long val) {
		try {
			String resultAsString = System.getProperty(nm);
			if(resultAsString != null) {
				return decode(resultAsString);
			}
		}
		catch(NullPointerException e) {
			//nm is null, see System.getProperty
		}
		catch(NumberFormatException e) {
			// resultAsString is an invalid number
			// Return default value
		}
		catch(IllegalArgumentException e) {
			//nm is empty, see System.getProperty
		}
		
		// Return default value since resultAsString is null or an invalid number
		return val;
	}
	
//	public static long highestOneBit(long i) {
//	}

//	public static long lowestOneBit(long i) {
//	}

//	public static int numberOfLeadingZeros(long i) {
//	}

//	public static int numberOfTrailingZeros(long i) {
//	}

	public static long parseLong(@Nullable String s) throws NumberFormatException {
		return Long.parseLong(s, 10);
	}

	public static long parseLong(@Nullable String s, int radix) throws NumberFormatException {
		if (s == null || s.length() == 0) {
			throw new NumberFormatException();
		}

		if (radix < Character.MIN_RADIX || radix > Character.MAX_RADIX) {
			throw new NumberFormatException(Message.at(new EDCErrorMessages(), EDCErrorMessages.ArgumentInvalidRadix));
		}

		try {
			return SupportNumber.parseLong(s.chars, s.offset, s.length, radix);
		} catch (IllegalArgumentException e) {
			// cause is implementation dependant
			throw new NumberFormatException(s); //NOSONAR
		}
	}
	
//	public static long reverse(long i) {
//	}

//	public static long reverseBytes(long i) {
//	}
	
	/**
	 * @accelerable
	 */
	public static long rotateLeft(long i, int distance) {
		return (i << distance) | (i >>> (Long.SIZE - distance));
	}
	
	/**
	 * @accelerable
	 */
	public static long rotateRight(long i, int distance) {
		return (i >>> distance) | (i << (Long.SIZE - distance));
	}
	
	/**
	 * @accelerable
	 */
	public static int signum(long i) {
		if(i == 0) {
			return 0;
		}
		return (int) (i >> (Long.SIZE - 1)) | 1;
	}

	public static String toString(long i) {
		return Long.toString(i, 10);
	}

	public static String toString(long i, int radix) {
		return SupportNumber.toStringLong(i, radix);
	}
	
	/**
	 * Returns a string representation of the {@code long} argument as an unsigned integer in
	 * base&nbsp;2.
	 * 
	 * <p>
	 * The unsigned {@code long} value is the argument plus 2<sup>64</sup> if the argument is
	 * negative; otherwise, it is equal to the argument. This value is converted to a string of
	 * ASCII digits in binary (base&nbsp;2) with no extra leading {@code 0}s. If the unsigned
	 * magnitude is zero, it is represented by a single zero character {@code '0'} (
	 * <code>'&#92;u0030'</code>); otherwise, the first character of the representation of the
	 * unsigned magnitude will not be the zero character. The characters {@code '0'} (
	 * <code>'&#92;u0030'</code>) and {@code '1'} (<code>'&#92;u0031'</code>) are used as binary
	 * digits.
	 * 
	 * @param i
	 *        a {@code long} to be converted to a string.
	 * @return the string representation of the unsigned {@code long} value represented by the
	 *         argument in binary (base&nbsp;2).
	 */
	public static String toBinaryString(long i) {
		return SupportNumber.toUnsignedStringLong(i, 1);
	}

	/**
	 * Returns a string representation of the {@code long} argument as an unsigned integer in
	 * base&nbsp;16.
	 * 
	 * <p>
	 * The unsigned {@code long} value is the argument plus 2<sup>64</sup> if the argument is
	 * negative; otherwise, it is equal to the argument. This value is converted to a string of
	 * ASCII digits in hexadecimal (base&nbsp;16) with no extra leading {@code 0}s. If the unsigned
	 * magnitude is zero, it is represented by a single zero character {@code '0'} (
	 * <code>'&#92;u0030'</code>); otherwise, the first character of the representation of the
	 * unsigned magnitude will not be the zero character. The following characters are used as
	 * hexadecimal digits:
	 * 
	 * <blockquote> {@code 0123456789abcdef} </blockquote>
	 * 
	 * These are the characters <code>'&#92;u0030'</code> through <code>'&#92;u0039'</code> and
	 * <code>'&#92;u0061'</code> through <code>'&#92;u0066'</code>. If uppercase letters are
	 * desired, the {@link java.lang.String#toUpperCase()} method may be called on the result:
	 * 
	 * <blockquote> {@code Long.toHexString(n).toUpperCase()} </blockquote>
	 * 
	 * @param i
	 *        a {@code long} to be converted to a string.
	 * @return the string representation of the unsigned {@code long} value represented by the
	 *         argument in hexadecimal (base&nbsp;16).
	 */
	public static String toHexString(long i) {
		return SupportNumber.toUnsignedStringLong(i, 4);
	}

	public static Long valueOf(long value) {
		return new Long(value);
	}
	
	public static Long valueOf(String s) throws NumberFormatException {
		return new Long(Long.parseLong(s, 10));
	}
	
	public static Long valueOf(String s, int radix) throws NumberFormatException {
		return new Long(Long.parseLong(s, radix));
	}

	/* INSTANCE VARIABLES */
	private long value;

	public Long(long value) {
		this.value = value;
	}
	
	public Long(String s) throws NumberFormatException {
		this(valueOf(s).longValue());
	}

	public int compareTo(Long o) {
		return compare(this.value, o.value);
	}
	
	@Override
	public byte byteValue() {
		return (byte) this.value;
	}

	@Override
	public double doubleValue() {
		return (double) this.value;
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		try {
			return ((Long) obj).value == this.value;
		} catch (ClassCastException e) {
			// obj is not a Long
			return false;
		} catch (NullPointerException e) {
			// obj is null
			return false;
		}
	}

	@Override
	public float floatValue() {
		return (float) this.value;
	}

	@Override
	public int hashCode() {
		return (int) (this.value ^ (this.value >>> 32));
	}

	@Override
	public int intValue() {
		return (int) this.value;
	}

	@Override
	public long longValue() {
		return this.value;
	}
	
	@Override
	public short shortValue() {
		return (short) this.value;
	}

	@Override
	public String toString() {
		return Long.toString(this.value, 10);
	}

}
