/*
 * Java
 *
 * Copyright 2008-2019 IS2T. All rights reserved. 
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;

import com.is2t.vm.support.lang.SupportNumber;

import ej.annotation.Nullable;

public final class Double extends Number implements Comparable<Double> {


	public static final int MAX_EXPONENT = SupportNumber.DOUBLE_MAX_EXPONENT;

	public static final double MAX_VALUE = SupportNumber.DOUBLE_MAX_VALUE;

	public static final int MIN_EXPONENT = SupportNumber.DOUBLE_MIN_EXPONENT;

	public static final double MIN_VALUE = SupportNumber.DOUBLE_MIN_VALUE;

	public static final double NaN = SupportNumber.DOUBLE_NaN;

	public static final double NEGATIVE_INFINITY = SupportNumber.DOUBLE_NEGATIVE_INFINITY;

	public static final double POSITIVE_INFINITY = SupportNumber.DOUBLE_POSITIVE_INFINITY;

	public static final int SIZE = SupportNumber.DOUBLE_SIZE;

	public static final double MIN_NORMAL = SupportNumber.DOUBLE_MIN_NORMAL;

	private static long SIGN_MASK = 0b10000000_00000000_00000000_00000000_00000000_00000000_00000000_00000000L;

	public static native long doubleToLongBits(double value);

	public static native long doubleToRawLongBits(double value);

	public static boolean isInfinite(double v) {
		return v == Double.NEGATIVE_INFINITY || v == Double.POSITIVE_INFINITY;
	}

	public static boolean isNaN(double v) {
		return v != v;
	}

	public static native double longBitsToDouble(long bits);

	public static double parseDouble(String s) throws NumberFormatException {
		return SupportNumber.parseDouble(s);
	}

	// public static String toHexString(double d) {
	// }

	public static String toString(double f) {
		return SupportNumber.toStringDouble(f);
	}

	public static Double valueOf(double value) {
		return new Double(value);
	}

	public static Double valueOf(String s) throws NumberFormatException {
		return new Double(Double.parseDouble(s));
	}

	/* INSTANCE VARIABLES */
	private final double value;

	public Double(double value) {
		this.value = value;
	}

	public Double(String s) throws NumberFormatException {
		this(valueOf(s).doubleValue());
	}

	@Override
	public byte byteValue() {
		return (byte) value;
	}

	public static int compare(double d1, double d2) {
		if (Double.isNaN(d1) && Double.isNaN(d2)) {
			return 0;
		} else if (Double.isNaN(d1)) {
			return 1;
		} else if (Double.isNaN(d2)) {
			return -1;
		} else {
			boolean d1Positive = (doubleToLongBits(d1) & SIGN_MASK) == 0;
			boolean d2Positive = (doubleToLongBits(d2) & SIGN_MASK) == 0;

			if(d1Positive && d2Positive || !d1Positive && !d2Positive) {
				return d1 > d2 ? 1 : d1 < d2 ? -1 : 0;
			}else if(d1Positive) {
				return 1;
			}else {
				return -1;
			}
		}
	}

	@Override
	public int compareTo(Double o) {
			return compare(this.value, o.value);
	}

	@Override
	public double doubleValue() {
		return this.value;
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		try {
			return compare(value, ((Double) obj).value) == 0;
		} catch (NullPointerException | ClassCastException e) {
			return false;
		}
	}

	@Override
	public float floatValue() {
		return (float) this.value;
	}

	@Override
	public int hashCode() {
		long v = Double.doubleToLongBits(this.value);
		return (int) (v ^ (v >>> 32));
	}

	@Override
	public int intValue() {
		return (int) this.value;
	}

	public boolean isInfinite() {
		return Double.isInfinite(this.value);
	}

	public boolean isNaN() {
		return Double.isNaN(this.value);
	}

	@Override
	public long longValue() {
		return (long) this.value;
	}

	@Override
	public short shortValue() {
		return (short) this.value;
	}

	@Override
	public String toString() {
		return SupportNumber.toStringDouble(this.value);
	}

}