/*
 * Java
 *
 * Copyright 2004-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;


import java.io.InputStream;

import com.is2t.vm.support.CalibrationConstants;
import com.is2t.vm.support.InternalLimitsError;
import com.is2t.vm.support.ResourceLoader;
import com.is2t.vm.support.lang.Systools;

import ej.annotation.Nullable;
import ej.bon.Constants;

public final class Class<T> implements java.lang.reflect.AnnotatedElement, java.lang.reflect.GenericDeclaration, java.io.Serializable, java.lang.reflect.Type {
	
	private static final String RESOURCE_LOADER_CLASS_PROPERTY = "com.is2t.resource.loader.class" ; //$NON-NLS-1$
	
    //this MUST have NO field (vm assumes it)

    private Class()
    {
        //cannot be instanciated
    }
    
    @SuppressWarnings("unchecked")
	public <U> Class<? extends U> asSubclass(Class<U> clazz) {
		// Check if cast is possible instead of trying to cast and catch exception
    	// since isAssignableFrom is native
    	if(clazz.isAssignableFrom(this)) {
    		return (Class<? extends U>) this;
    	}
    	else {
    		throw new ClassCastException();
    	}
	}
    
    @SuppressWarnings("unchecked")
    @Nullable
	public T cast(@Nullable Object obj) {
    	// See javadoc :
    	// throws ClassCastException if the object is not null and is not
    	// assignable to the type T.
		if(obj != null && !this.isInstance(obj)) {
			throw new ClassCastException();
		}
		
		return (T) obj;
	}
    
    public boolean desiredAssertionStatus() {
    	// Depends on the value of "Enable assertions on Device"
        return Constants.getBoolean("com.microej.library.edc.assertions.enabled");  //$NON-NLS-1$
    }
    
    /**
     * Returns the {@code Class} representing the superclass of the entity
     * (class, interface, primitive type or void) represented by this
     * {@code Class}.  If this {@code Class} represents either the
     * {@code Object} class, an interface, a primitive type, or void, then
     * null is returned.  If this object represents an array class then the
     * {@code Class} object representing the {@code Object} class is
     * returned.
     *
     * @return the superclass of the class represented by this object.
     */
    @Nullable
    native public Class<? super T> getSuperclass();

    // IMPLEMENTATION NOTE: VM knows the number of calls from this API to native method (2 calls)
    // Changing the number of calls need to be reported to NativePool implementation of java_lang_Class_forNameNative
	public static Class<?> forName(String className) throws ClassNotFoundException {
		Class<?> c = forNameNative(className);
		if (c == null) {
			throw new InternalLimitsError("try to grow _jvm_sharedMemory_size"); //$NON-NLS-1$
		}
		return c;
	}
	
	@Nullable
	public static native Class<?> forNameNative(String className) throws ClassNotFoundException;
        
	public native String getName() ;
	

	/**
	 * Returns the simple name of the underlying class as given in the source code. Returns an empty
	 * string if the underlying class is anonymous.
	 * 
	 * <p>
	 * The simple name of an array is the simple name of the component type with "[]" appended. In
	 * particular the simple name of an array whose component type is anonymous is "[]".
	 * 
	 * @return the simple name of the underlying class
	 */
	public String getSimpleName() {
		return Systools.getSimpleName(getName());
	}
	
	/**
	 * Gets the package for this class. Null is returned if no package object was created by the
	 * class loader of this class.
	 * 
	 * @return the package of the class, or null if no package information is available from the
	 *         archive or codebase.
	 */
	@Nullable
	public Package getPackage() {
		return Package.getPackage(this);
	}

	// IMPLEMENTATION NOTE: VM knows the number of calls from this API to native method (2 calls)
    // Changing the number of calls need to be reported to NativePool implementation of java_lang_Class_getResourceAsStreamNative
	@Nullable
	public InputStream getResourceAsStream(String name){
	    if (name.length() == 0 ){
	    	return null;
	    }
		
	    // 1) Try to load internal resource
	    String path;
		if(name.charAt(0)=='/'){
			//absolute path
            path = name.substring(1);
		}
        else{
        	//relative path
        	String classPackage = getName().replace('.', '/');//get the package of the current class
        	int lastIndex = classPackage.lastIndexOf('/');
        	if(lastIndex != -1){
        		// package name can be null! (default package)
        		path = new StringBuilder().append(classPackage.chars,0,lastIndex+1).append(name).toString();
        	}
        	else{
        		path = name;
        	}
        }
		InputStream is = getResourceAsStreamNative(path);
		if(is != null){
			return is;
		}
		
		if(CalibrationConstants.ENABLE_RESOURCE_LOADER){
			// 2) Delegate load of resource to optional platform specific handler
			String resourceLoaderClassname = System.getPropertyNoCheck(RESOURCE_LOADER_CLASS_PROPERTY);
			if(resourceLoaderClassname == null) {
				return null;				
			}
			ResourceLoader loader;
			try{ loader = (ResourceLoader)(Class.forName(resourceLoaderClassname).newInstance()); }
			catch (InstantiationException | IllegalAccessException | ClassNotFoundException e) {
				return null;
			}
						
			try{ return loader.getResourceAsStream(path); }
			catch(Throwable e){
				return null;
			}
		}
		
		// no resource found
		return null;
    }

	@Nullable
	private native InputStream getResourceAsStreamNative(String name);

	public native boolean isArray() ;

	public native boolean isAssignableFrom(Class<?> cls) ;

	public native boolean isInstance(@Nullable Object obj) ;

	public native boolean isInterface() ;

	public native Object newInstance()
		throws InstantiationException, IllegalAccessException ;

	@Override
	public String toString() {
		//WARNING It's impossible to invoke toString() for Class objects which represents a primitive type
		//because the field TYPE does not exist inside wrapper classes in CLDC 1.1(Integer, Double ...)
		//i.e. you can't write : int.class.toString()
		if (isInterface()){
			return "interface " + getName(); //$NON-NLS-1$
		}else{
			return "class " + getName(); //$NON-NLS-1$
		}
	}

}
