/*
 * Java
 *
 * Copyright 2008-2019 IS2T. All rights reserved
 * IS2T PROPRIETARY/CONFIDENTIAL Use is subject to license terms.
 */
package java.io;

public class DataOutputStream extends FilterOutputStream implements DataOutput {

	protected int written;

    public DataOutputStream(OutputStream out) {
    	super(out);
    }
    
    @Override
	public void flush() throws IOException {
		OutputStream out = this.out;
		if (out != null) {// not yet closed
			out.flush();
		}
	}
    
    public final int size() {
		return this.written;
	}
    
    @Override
	public synchronized void write(byte[] b, int off, int len) throws IOException {
		OutputStream out = this.out;
		assert out != null;
    	out.write(b,off,len) ;
    	updateWritten(len);
	}
    
    @Override
	public synchronized void write(int b) throws IOException {
		OutputStream out = this.out;
		assert out != null;
		out.write(b);
		updateWritten(1);
    }
    
    public final void writeBoolean(boolean v) throws IOException {
		OutputStream out = this.out;
		assert out != null;
		out.write(v ?  1 :  0);
		updateWritten(1);
	}

    public final void writeByte(int v) throws IOException {
		OutputStream out = this.out;
		assert out != null;
 		out.write(v);
 		updateWritten(1);
   }
    
    public final void writeBytes(String s) throws IOException {
		OutputStream out = this.out;
		assert out != null;
		int len = s.length();
		char[] chars = s.chars;
		for(int i = 0; i < len; i++) {
			out.write((byte) chars[i]);
		}
		updateWritten(len);
	}

    public final void writeChar(int v) throws IOException {
		OutputStream out = this.out;
		assert out != null;
		out.write(v >> 8);
		out.write(v);
		updateWritten(2);
    }
    
    public final void writeChars(String s) throws IOException {
		OutputStream out = this.out;
		assert out != null;
		int len = s.offset+s.length();
		char[] c = s.chars;
		for(int i = s.offset; i < len; ++i) {
	   		out.write(c[i]>>8);
    		out.write(c[i]);
		}
		updateWritten(2 * len);
	}

    public final void writeDouble(double v) throws IOException {
    	this.writeLong(Double.doubleToLongBits(v)) ;
    }

    public final void writeFloat(float v) throws IOException {
    	this.writeInt(Float.floatToIntBits(v)) ;
    }

    public final void writeInt(int v) throws IOException {
		OutputStream out = this.out;
		assert out != null;
		out.write(v >> 24);
		out.write(v >> 16);
		out.write(v >> 8);
		out.write(v);
		updateWritten(4);
	}

	public final void writeLong(long v) throws IOException {
		this.writeInt((int) (v >> 32));
		this.writeInt((int) v);
	}
    
	public final void writeShort(int v) throws IOException {
		this.writeChar(v) ;
    }

    public final void writeUTF(String str) throws IOException {
		OutputStream out = this.out;
		assert out != null;
    	char[] chars = str.toCharArray() ;
    	byte[] arrayUTFString = new byte[3*chars.length]; //1 char is encoded at most by 3 bytes in UTF format
    	int i=-1, end=chars.length, count=-1  ;
    	
    	// first compute the size of utf8 string
     	while (++i < end) {
    		char c = chars[i] ;
    		if (c >= '\u0001' && c <= '\u007f') {
    			arrayUTFString[++count] = (byte)c ;
    		}
    		else if (c=='\u0000' || c <= '\u07ff') {
    			arrayUTFString[++count] = (byte)(0xc0 | (0x1f & (c >> 6))) ;
    			arrayUTFString[++count] = (byte)(0x80 | (0x3f & c)) ;
    		}
    		else { // from \u0800 through \uffff
    			arrayUTFString[++count] = (byte)(0xe0 | (0x0f & (c >> 12))) ;
    			arrayUTFString[++count] = (byte)(0x80 | (0x3f & (c >>  6))) ;
    			arrayUTFString[++count] = (byte)(0x80 | (0x3f & c)) ;
    		}
    	}
    	
    	if (++count > 65535) {
    		throw new UTFDataFormatException() ;
    	}
    	
    	writeShort(count) ;
    	out.write(arrayUTFString, 0, count);
    }
    
    private void updateWritten(int nb) {
    	final int written = this.written + nb;
    	this.written = written;
    	if(written < 0) {
    		// overflow
    		this.written = Integer.MAX_VALUE;
    	}
    }

}
