/*
 * Java
 *
 * Copyright 2004-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.vm.support.util;


import java.util.Calendar;
import java.util.TimeZone;

import ej.annotation.Nullable;

public class GregorianCalendar extends Calendar {
	// IMPLEMENTATION NOTES:
	// This GregorianCalendar is not fully compliant with Gregorian/Julian rules :
	// - October 4, 1582 was thus followed by October 15, 1582
	// - BC era has not been tested (not mandatory in CLDC-1.1? AD and BC fields are not defined in Calendar, but they are used in Timezone.getOffset())

	/**
     * Nb absolute days since year 0
     */
    static final long ZERO_NB_DAYS = (1970l*365l+((1970l/4l)-(1970l/100l)+1970l/400l));
    /**
     * Absolute time in millis of 1st Jan 1970 from year 0
     */
    static final long ZERO_TIME_MILLIS = ZERO_NB_DAYS*24l*60l*60l*1000l;

    /**
	 * Value of the <code>ERA</code> field indicating the common era (Anno Domini), also known as
	 * CE. The sequence of years at the transition from <code>BC</code> to <code>AD</code> is ..., 2
	 * BC, 1 BC, 1 AD, 2 AD,...
	 *
	 * @see #ERA
	 */
	public static final int AD = 1;

	/**
	 * Value of the <code>ERA</code> field indicating the period before the common era (before
	 * Christ), also known as BCE. The sequence of years at the transition from <code>BC</code> to
	 * <code>AD</code> is ..., 2 BC, 1 BC, 1 AD, 2 AD,...
	 *
	 * @see #ERA
	 */
	public static final int BC = 0;

	/**
	 * Value of the {@link #ERA} field indicating the period before the common era, the same value
	 * as {@link #BC}.
	 *
	 * @see #CE
	 */
	static final int BCE = 0;

	/**
	 * Value of the {@link #ERA} field indicating the common era, the same value as {@link #AD}.
	 *
	 * @see #BCE
	 */
	static final int CE = 1;

    static final long ONE_SECOND = 1000;
    static final long ONE_MINUTE = 60 * ONE_SECOND;
    static final long ONE_HOUR   = 60 * ONE_MINUTE;
    static final long ONE_DAY    = 24 * ONE_HOUR;

    static final int[] FIELDS_EPOCH_VALUES = {
    	CE, // ERA
    	1970, // YEAR
    	JANUARY, // MONTH
    	1, // WEEK_OF_YEAR
		1, // WEEK_OF_MONTH
		1, // DAY_OF_MONTH
		1, // DAY_OF_YEAR
		THURSDAY, // DAY_OF_WEEK
		1, // DAY_OF_WEEK_IN_MONTH
		AM, // AM_PM
		0, // HOUR
		0, // HOUR_OF_DAY
		0, // MINUTE
		0, // SECOND
		0, // MILLISECOND
		(int)ONE_HOUR, // ZONE_OFFSET
		0 // DST_OFFSET
    };

    static final int[] FIELDS_MIN_VALUES = {
    	BCE, // ERA
    	1, // YEAR
		JANUARY, // MONTH
		1, // WEEK_OF_YEAR
		0, // WEEK_OF_MONTH
		1, // DAY_OF_MONTH
		1, // DAY_OF_YEAR
		SUNDAY, // DAY_OF_WEEK
		1, // DAY_OF_WEEK_IN_MONTH
		AM, // AM_PM
		0, // HOUR
		0, // HOUR_OF_DAY
		0, // MINUTE
		0, // SECOND
		0, // MILLISECOND
		-13 * (int)ONE_HOUR, // ZONE_OFFSET
		0 // DST_OFFSET
    };

    static final int[] FIELDS_MAX_VALUES = {
    	CE, // ERA
    	292278994, // YEAR
		DECEMBER, // MONTH
		53, // WEEK_OF_YEAR
		6, // WEEK_OF_MONTH
		31, // DAY_OF_MONTH
		366, // DAY_OF_YEAR
		SATURDAY, // DAY_OF_WEEK
		6, // DAY_OF_WEEK_IN_MONTH
		PM, // AM_PM
		11, // HOUR
		23, // HOUR_OF_DAY
		59, // MINUTE
		59, // SECOND
		999, // MILLISECOND
		14 * (int)ONE_HOUR, // ZONE_OFFSET
		2 * (int)ONE_HOUR// DST_OFFSET
    };

    /*
     * Those values are valid if not taking Gregorian change date (in 1582) into account.
     * Else the following values are different:
     * WEEK_OF_YEAR: 50
     * WEEK_OF_MONTH: 3
     * DAY_OF_YEAR: 355
     * DAY_OF_WEEK_IN_MONTH: 3
     * Implementation may vary in other VMs.
     */
    static final int[] FIELDS_LEAST_MAX_VALUES = {
    	CE, // ERA
    	292269054, // YEAR
		DECEMBER, // MONTH
		52, // WEEK_OF_YEAR
		4, // WEEK_OF_MONTH
		28, // DAY_OF_MONTH
		365, // DAY_OF_YEAR
		SATURDAY, // DAY_OF_WEEK
		4, // DAY_OF_WEEK_IN_MONTH
		PM, // AM_PM
		11, // HOUR
		23, // HOUR_OF_DAY
		59, // MINUTE
		59, // SECOND
		999, // MILLISECOND
		14 * (int)ONE_HOUR, // ZONE_OFFSET
		20 * (int)ONE_MINUTE// DST_OFFSET
    };

    /*
     * Number of past days for each month.
     * Those values are for non bisextil years, else 1 must be added to months after February
     */
    static final int[] YEAR_PAST_DAYS_COUNT = {
    	0, //JANUARY
		31, // FEBRUARY
		59, // MARCH
		90, // APRIL
		120, // MAY
		151, // JUNE
		181, // JULY
		212, // AUGUST
		243, // SEPTEMBER
		273, // OCTOBER
		304, // NOVEMBER
		334, // DECEMBER
    };

	public GregorianCalendar() {
		super();
	}

	/**
	 * @accelerable
	 */
	private static boolean isBisextilYear(int year) {
		return ((year % 4 == 0) && (year % 100 !=0)) || (year % 400 ==0);
	}

	/**
	 * @accelerable
	 */
	private static int getNbDaysInYear(int  year) {
		return isBisextilYear(year) ? 366 : 365;
	}

	private int getIntern(int field) {
		/* Default value is equal to epoch, except for DAY_OF_WEEK */
		return isSet(field) ? fields[field] : ((field != DAY_OF_WEEK) ? FIELDS_EPOCH_VALUES[field] : getFirstDayOfWeek());
	}

	/**
	 * Modulo always returning positive results
	 * @return Positive modulo of value % modulo
	 */
	private int mod(int value, int modulo) {
		int result = value % modulo;

		return (result < 0) ? (result + modulo) : result;
	}

	/**
	 * Compute and fill {@link Calendar#YEAR}, {@link Calendar#MONTH}, {@link Calendar#DAY_OF_WEEK}, {@link Calendar#DAY_OF_MONTH} fields.
	 * @return the remaining number of milliseconds in the day
	 * @accelerable
	 */
	private int computeYearMonthDay(long time){
		long absTime = time + ZERO_TIME_MILLIS;
		/* Do not use absolute time when time is as it may have been overflown */
		long nbDays = time < 0 ? (absTime / ONE_DAY) : (time / ONE_DAY + ZERO_TIME_MILLIS / ONE_DAY);

		// compute the number of years and the number of days within the year
		// 1 year = 365 + 1/4 - 1/100 + 1/400 days
		// => 400 years = 365*400 + 100 - 4 + 1 days
		// => nbYears = (nbDays * 400) / (365*400 + 100 - 4 + 1)
		// Maybe 1 year is missing.
		// Example: Day 365 (January 1st, 1971) => nbYears = 0 because bisextil year is not yet encountered


		int year = (int)((nbDays*400) / (365*400+100-4+1));
		int lastCompleteYear = year-1;
		// compute the real number of days
		int nbMod4Years = lastCompleteYear/4;
		int nbMod100Years = lastCompleteYear/100;
		int nbMod400Years = lastCompleteYear/400;
		long nbRawDays = ((long)lastCompleteYear+1)*365+(nbMod4Years-nbMod100Years)+nbMod400Years;

		// Compute dayOfYear (>=0 from the beginning of the current year)
		int dayOfYear = (int)(nbDays-nbRawDays+1);
		int nbDaysInCurrentYear = getNbDaysInYear(year);
		if(dayOfYear > nbDaysInCurrentYear){
			dayOfYear -= nbDaysInCurrentYear;
			++year;
		}

		int[] fields = this.fields;
		fields[Calendar.DAY_OF_YEAR] = dayOfYear;
    	fields[Calendar.YEAR] = year;
    	fields[Calendar.ERA] = year < 0 ? BC : AD;

    	boolean bissextilYear = isBisextilYear(year);

    	// Compute the day of week (January 1st, 1970 was a Thursday)
    	int dayOfWeek = (int)((((THURSDAY-1)+((nbDays%7)-(int)(ZERO_NB_DAYS%7))) % 7) + 1);
    	fields[Calendar.DAY_OF_WEEK] = dayOfWeek;

    	// Compute the month
    	int month = (dayOfYear - 1) / 28; // approximation : month may be bigger.
    	// approximation : For dates like 30th of January, nbDaysInMonth will be negative and month will be adjusted in the next loop
    	if(month > DECEMBER) {
    		month = DECEMBER;
    	}

    	int nbDaysStartMonth = getNbOfDays(month, bissextilYear);
    	int nbDaysInMonth = dayOfYear-nbDaysStartMonth;
    	while (nbDaysInMonth <= 0) {
    		nbDaysStartMonth = getNbOfDays(--month, bissextilYear);
        	nbDaysInMonth = dayOfYear-nbDaysStartMonth;
    	}

    	fields[Calendar.MONTH] = month;
    	int dayOfMonth = nbDaysInMonth;
    	fields[Calendar.DAY_OF_MONTH] = dayOfMonth;

    	// Compute week related fields
    	int firstWeekDayOffset = getFirstWeekDayOffset(year, month);
    	int weekOfMonth = (dayOfMonth - firstWeekDayOffset) / 7 + 1;
    	fields[Calendar.WEEK_OF_MONTH] = weekOfMonth;

    	fields[Calendar.DAY_OF_WEEK_IN_MONTH] = ((dayOfMonth - 1) / 7) + 1;

    	int firstYearDayOffset = getFirstWeekDayOffset(year, Calendar.JANUARY);
    	fields[Calendar.WEEK_OF_YEAR] = (dayOfYear - firstYearDayOffset) / 7 + 1;

		/* Do not use absolute time when time is as it may have been overflown */
    	long millisInDay = time < 0 ? (absTime % ONE_DAY) : (time % ONE_DAY);

    	return (int) millisInDay; // = millisInDay
	}

	/**
	 * @param month between {@link Calendar#JANUARY} and {@link Calendar#DECEMBER}.
	 * @return the number of days at the beginning of the month
	 * @accelerable
	 */
	private static int getNbOfDays(int month, boolean bissextil){ //NOSONAR
		//Cyclomatic Complexity : Cyclomatic Complexity is 14 (max allowed is 10). is necessary

		// should be turned into an immutable array instead of a table switch
		int nbOfDays = YEAR_PAST_DAYS_COUNT[month];

		if(bissextil && month > Calendar.FEBRUARY) {
			return nbOfDays+1;
		}

		return nbOfDays;
	}

	/**
	 * Get the day of the 1st of a month in a year
	 * @param year The year
	 * @param month The month
	 * @return The day of the 1st of the month in a year
	 */
	private static int getFirstMonthDay(int year, int month) {
		int nbDays = year * (365*400+100-4+1) / 400;

		nbDays += getNbOfDays(month, isBisextilYear(year));

		// Compute the day of week (January 1st, 1970 was a Thursday)
    	return (((THURSDAY-1)+((nbDays%7)-(int)(ZERO_NB_DAYS%7))) % 7) + 1;
	}

	/**
	 * Get the first day of the first week of a month in a year
	 * @param year The year
	 * @param month The month
	 * @return The day of the month, or a negative value if the first day is in the past month
	 */
	private int getFirstWeekDayOffset(int year, int month) {
		int firstYearDay = getFirstMonthDay(year, month);

		/* How many days of the week are in the new year? */
		int weekDays = Calendar.SATURDAY - firstYearDay + getFirstDayOfWeek();

		/* Check if a sufficient number of days is in the first week */
		if (weekDays < getMinimalDaysInFirstWeek()) {
			return mod(getFirstDayOfWeek() - firstYearDay, 7) + Calendar.SUNDAY;
		}
		else {
			return getFirstDayOfWeek() - firstYearDay + Calendar.SUNDAY;
		}
	}

	/**
     * Converts the current millisecond time value time to field values in
     * fields[]. This allows you to sync up the time field values with a new
     * time that is set for the calendar.
     * @accelerable
     */
    @Override
	protected void computeFields() {
    	computeFields(true);
    }

    /**
     * Compute time from {@link Calendar#time}
     * @param utc Compute from UTC time (add timezone offset)
     * @accelerable
     */
    private void computeFields(boolean utc){
    	// First step is to compute YEAR/MONTH/DAY in the given timezone
    	long realTime;
    	final int[] fields = this.fields;
    	if(utc){
			TimeZone timeZone = this.timeZone;
			assert timeZone != null;
    		int rawOffset = isExternallySet(ZONE_OFFSET) ? fields[Calendar.ZONE_OFFSET] : timeZone.getRawOffset();
    		realTime = this.time + rawOffset;

    		// Then update with daylight savings time offset
    		int dstOffset = 0;
    		if (isExternallySet(DST_OFFSET)) {
    			dstOffset = fields[Calendar.DST_OFFSET];
    		}
    		else {
	    		int millisInDay = computeYearMonthDay(realTime);
	    		int offset = getTimezoneOffset(millisInDay);
	    		dstOffset = offset-rawOffset;
    		}

    		realTime += dstOffset;

    		/* Set fields */
    		fields[Calendar.ZONE_OFFSET] = rawOffset;
    		fields[Calendar.DST_OFFSET] = dstOffset;
    	}
    	else{
    		realTime = this.time;
    	}

    	// Recompute YEAR/MONTH/DAY
    	int millisInDay = computeYearMonthDay(realTime);

    	// Compute hours
    	int hour;
    	fields[Calendar.HOUR_OF_DAY] = (hour = (int) (millisInDay / ONE_HOUR));

		// update AM_PM and HOUR
    	fields[Calendar.HOUR] = hour % 12;
    	fields[Calendar.AM_PM] = hour / 12;

    	// Compute minutes
    	fields[Calendar.MINUTE] = (int) ((millisInDay -= (fields[Calendar.HOUR_OF_DAY] * ONE_HOUR)) / ONE_MINUTE);

    	// Compute seconds
    	fields[Calendar.SECOND] = (int)((millisInDay -= (fields[Calendar.MINUTE] * ONE_MINUTE)) / ONE_SECOND);

    	// Compute milliseconds
    	fields[Calendar.MILLISECOND] = (int) (millisInDay - (fields[Calendar.SECOND] * ONE_SECOND));
    }

    /**
     * Compute time from {@link Calendar#fields}
     * @param utc Compute UTC time (remove timezone offset).
     * Fields may be invalid
     * @accelerable
     */
    private void computeTime(boolean utc){
		long time = millisInDay();
		long timeWithZoneOffset = 0;

		int resolution = dateResolutionMethod();

		int year = 0;
		int month = 0;
		if (resolution == RESOLUTION_YEAR_DAY_OF_YEAR || resolution == RESOLUTION_YEAR_DAY_OF_WEEK_WEEK_OF_YEAR) {
			// Compute year only (month not needed)
			year = getIntern(Calendar.YEAR);
		}
		else {
			// Compute year and month (fields may be out of range)
			int nbMonths = getIntern(Calendar.YEAR)*12 + getIntern(Calendar.MONTH);
			year = nbMonths / 12;
			month = nbMonths % 12;
		}

		int dayOfYear = 0;

		switch (resolution) {
		case RESOLUTION_YEAR_MONTH_DAY_OF_MONTH:
			/* YEAR + MONTH + DAY_OF_MONTH */
			dayOfYear = (getIntern(Calendar.DAY_OF_MONTH) - 1) + getNbOfDays(month, isBisextilYear(year)) + 1;
			break;
		case RESOLUTION_YEAR_MONTH_WEEK_OF_MONTH_DAY_OF_WEEK:
			/* YEAR + MONTH + WEEK_OF_MONTH + DAY_OF_WEEK */
			dayOfYear = getNbOfDays(month, isBisextilYear(year)) + getFirstWeekDayOffset(year, month) +
				(getIntern(Calendar.WEEK_OF_MONTH) - 1) * 7 + (getIntern(Calendar.DAY_OF_WEEK) - 1);
			break;
		case RESOLUTION_YEAR_MONTH_DAY_OF_WEEK_IN_MONTH_DAY_OF_WEEK:
			/* YEAR + MONTH + DAY_OF_WEEK_IN_MONTH + DAY_OF_WEEK */
			dayOfYear = getNbOfDays(month, isBisextilYear(year)) + ((getIntern(Calendar.DAY_OF_WEEK) - getFirstMonthDay(year, month)) % 7) + 1 +
				(getIntern(Calendar.DAY_OF_WEEK_IN_MONTH) - 1) * 7;
			break;
		case RESOLUTION_YEAR_DAY_OF_YEAR:
			/* YEAR + DAY_OF_YEAR */
			dayOfYear = getIntern(Calendar.DAY_OF_YEAR);
			break;
		case RESOLUTION_YEAR_DAY_OF_WEEK_WEEK_OF_YEAR:
			/* YEAR + DAY_OF_WEEK + WEEK_OF_YEAR */
			int firstWeekDay = getFirstWeekDayOffset(year, Calendar.JANUARY);
			int weekDays = (getIntern(Calendar.WEEK_OF_YEAR) - 1) * 7 + (firstWeekDay - 1);
			dayOfYear = getIntern(Calendar.DAY_OF_WEEK) + weekDays;
			break;
		}

		/* Iterate if dayOfYear is negative to find the year it belongs to */
		while (dayOfYear <= 0) {
			year--;
			dayOfYear += getNbDaysInYear(year);
		}

		int lastCompleteYear = year-1;
		int nbMod4Years = lastCompleteYear/4;
		int nbMod100Years = lastCompleteYear/100;
		int nbMod400Years = lastCompleteYear/400;

		// Compute number of days on a long, as it may overflow for years near to maximum
		long nbOfDays = (((long)year)*365+(nbMod4Years-nbMod100Years)+nbMod400Years) + dayOfYear - 1;
		timeWithZoneOffset = time + (nbOfDays * ONE_DAY) - ZERO_TIME_MILLIS;

		if(utc){
			// Get UTC time by removing timezone and dst offset
			// Recompute fields in order to ensure they are valid before getting the timezone offset
			TimeZone timeZone = this.timeZone;
			assert timeZone != null;
			int tzOffset = isExternallySet(ZONE_OFFSET) ? fields[ZONE_OFFSET] : timeZone.getRawOffset();
			this.time = timeWithZoneOffset-tzOffset;
			computeFields(true);

			int dstOffset = 0;
			if (isExternallySet(DST_OFFSET)) {
				dstOffset = fields[DST_OFFSET];
			}
			else {
				long millisInDay = millisInDay();
				dstOffset = getTimezoneOffset((int)millisInDay); // fields are valid
			}

			this.time = timeWithZoneOffset - dstOffset;
		}
		else {
			this.time = timeWithZoneOffset;
		}
    }

    /**
     * Gets the timezone offset, with DST savings for the current date.
     * Fields {@link Calendar#YEAR}, {@link Calendar#MONTH}, {@link Calendar#DAY_OF_MONTH}, {@link Calendar#DAY_OF_WEEK} are assumed to be valid
     * @param millisInDay number of milliseconds within the day [0,86400[
     * @accelerable
     */
    private int getTimezoneOffset(int millisInDay) {
    	int[] fields = this.fields;
		TimeZone timeZone = this.timeZone;
		assert timeZone != null;
    	return timeZone.getOffset(fields[YEAR] < 0 ? BC : AD, fields[YEAR], fields[MONTH], fields[DAY_OF_MONTH], fields[DAY_OF_WEEK], millisInDay);
	}


    /**
     * check if both are gregorianCalendar and both are equals
     */
    @Override
	public boolean equals(@Nullable Object o ) { //NOSONAR
    	// no need to override hashCode() from Calendar since the behavior is the same
    	try {
    		return super.equals((GregorianCalendar)o);
    	}
    	catch (ClassCastException e) { //Object o is not a gregoriancalendar
    		return false;
    	}
    }



	/**
	 * Converts the current field values in fields[] to the millisecond time value time.
	 * @accelerable
	 */
	@Override
	protected void computeTime() {
		// Time to fields is easy
		// Fields to time is a little more subtle.
		// - Fields may be out of range (invalid) (CLDC-1.1 Calendar is considered to be lenient)
		// - Fields may represent a "virtual" Date which does not exist. Example 28 March 2010 2h00 CET does not exist (28 March 2010 3h00)
		// The idea is to first convert invalid fields to valid fields, within the same timezone (1)
		// Then compute offset delta before and after returning to UTC time (2)
		// If delta != 0, a DST is being crossed and the valid date was virtual

		// 1) INVALID FIELDS -> UTC TIME+TZ+DST -> VALID FIELDS
		// 27 March 2010 26h00 CET -> 28 March 2010 2h00 CET
		// At this step, this is impossible to know that this Date exists or not
		computeTime(false);
		computeFields(false);
		// 28 March 2010 2h00 CET: offset 7200000 (2 hours)
		long millisInDay1 = millisInDay();
		int offset1 = getTimezoneOffset((int)millisInDay1); // fields are valid

		// 2) VALID FIELDS -> UTC TIME -> VALID FIELDS
		// 28 March 2010 2h00 CET -> 28 March 2010 0h00 GMT -> 28 March 2010 1h CET
		computeTime(true);
		computeFields(true);
		// 28 March 2010 1h CET: offset 3600000 (1 hour)
		long millisInDay2 = millisInDay();
		int offset2 = getTimezoneOffset((int)millisInDay2); // fields are valid

		if(offset1 != offset2){
			// crossing DST (UTC time was computed from a "virtual" address
			// For example, in 28 March 2010 2h00 Europe/Paris => 28 March 2010 3h00 Europe/Paris
			time = time+(offset1-offset2);
			computeFields(true); // last
		}

	}

	/**
	 * Gets the number of milliseconds from fields within a day.
	 * Fields may be out of range (invalid).
	 * When fields are valid, the returned value is in the range [0, 86400000[.
	 * @accelerable
	 */
	private long millisInDay() {
		int resolution = timeResolutionMethod();

		int hourOfDay = 0;

		switch(resolution) {
		case RESOLUTION_HOUR_OF_DAY:
			hourOfDay = getIntern(Calendar.HOUR_OF_DAY);
			break;
		case RESOLUTION_AM_PM_HOUR:
		{
			int hour = getIntern(Calendar.HOUR);
			int am_pm = getIntern(Calendar.AM_PM);

			hourOfDay = hour + (am_pm == PM ? 12 : 0);
			break;
		}
		}

		int minute = getIntern(Calendar.MINUTE);
		int second = getIntern(Calendar.SECOND);
		int millisecond = getIntern(Calendar.MILLISECOND);

		return millisecond + second * ONE_SECOND + minute * ONE_MINUTE + hourOfDay * ONE_HOUR;
	}

//	// TODO WI 8378
//	public void add(int field, int amount) {
//		throw new RuntimeException();
//	}
//

	@Override
	public int getGreatestMinimum(int field) {
		/* For Gregorian calendar, there is no difference between getMinimum() and getGreatestMinimum() */
		return FIELDS_MIN_VALUES[field];
	}

	@Override
	public int getLeastMaximum(int field) {
		return FIELDS_LEAST_MAX_VALUES[field];
	}

	@Override
	public int getMaximum(int field) {
		return FIELDS_MAX_VALUES[field];
	}

	@Override
	public int getMinimum(int field) {
		/* For Gregorian calendar, there is no difference between getMinimum() and getGreatestMinimum() */
		return FIELDS_MIN_VALUES[field];
	}

//	// TODO WI 8378
//	public void roll(int field, boolean up) {
//		throw new RuntimeException();
//	}
}
