/*
 * Java
 *
 * Copyright 2007-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.vm.support.util;

import java.io.UnsupportedEncodingException;

/**
 * Internal APIs for encoding/decoding bytes to chars inside our CLDC implementations
 * Subclasses represent one encoding implementation
 */
public abstract class EncodingConversion {
	
	protected String encoding;
	
	protected EncodingConversion(String encoding) {
		this.encoding = encoding;
	}

	/**
	 *  Default encoding. CLDC 1.1 should support at least ISO_8859_1 (ISO-LATIN-1)
	 */
	public static final String ISO_8859_1 = "ISO-8859-1"; //$NON-NLS-1$
	public static final String US_ASCII = "US-ASCII"; //$NON-NLS-1$
	public static final EncodingConversion EncodingIso88591 = new EncUS_ASCII(ISO_8859_1);
	public static final EncodingConversion EncodingUUASCII = new EncUS_ASCII(US_ASCII);
	public static final EncodingConversion DefaultEncoding = EncodingIso88591;

	/**
	 * Get the current encoding.
	 */
	public String getEncoding() {
		return this.encoding;
	}

	/**
	 * Gets the maximum number of bytes necessary to encode a character with this encoding
	 */
	public abstract int getMaxBytesPerChar();
	
	/**
	 * Decode characters from byte array to char array
	 * @return the number of decoded characters (the maximum that can be decoded within the given range of bytes and range of chars)
	 */
	public abstract int decode(byte[] bytes, int[] bytesOffset, int bytesLength, char[] chars, int offset, int length);
	
	/**
	 * Encode <code>length</code> characters to <code>bytes[bytesOffset]</code>
	 * @return the number of encoded bytes (the maximum that can be encoded within the given range of bytes and range of chars)
	 */
	public abstract int encode(char[] chars, int[] charsOffset, int charsLength, byte[] bytes, int offset, int length);
	
	/**
	 * Get the EncodingConversion for the given encoding name
	 * 
	 * Class should start with EncodingConversion fully qualified name + encoding name with - replaced with _
	 * and transformed into its upper case form
	 * Example:
	 * <ul>
	 * 	<li>UTF-8 => EncUTF_8</li>
	 * 	<li>utf-8 => EncUTF_8</li>
	 * </ul>
	 * 
	 * @param enc as described in CLDC spec
	 * @throw UnsupportedEncodingException if encoding is unknown
	 */
	public static final EncodingConversion getEncoding(String enc) throws UnsupportedEncodingException{
		enc = enc.toUpperCase();
		if(enc.equals(ISO_8859_1)) {
			return EncodingIso88591;
		}
		else if(enc.equals(US_ASCII)) {
			return EncodingUUASCII;
		}
		
		// the package name is hard coded => no class.getName needs
		String expectedImplClassname = new StringBuilder("com.is2t.vm.support.util.").append("Enc").append(enc.replace('-', '_')).toString(); //$NON-NLS-1$ //$NON-NLS-2$
		try{
			return (EncodingConversion) (Class.forName(expectedImplClassname).newInstance());
		}
		catch(Exception e) {
			// Cause is not required here because it is impl. specific
			throw new UnsupportedEncodingException(enc); //NOSONAR
		}
	}
	
}
