/*
 * Java
 *
 * Copyright 2008-2023 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */ 
package com.is2t.vm.support.io;

import java.io.IOException;
import java.io.InputStream;

/**
 * VM known class used to access to raw memory as a stream
 */
public class MemoryInputStream extends InputStream
{
    int startAddress; //VM known
    int endAddress;//exclusive   //VM known
    int currentPtr;   //VM known
    int markPtr;      //VM known
    
    /**
     * Accesses to this field does not need to be synchronized or thread safe in Java.
     * It is used only for checks in methods with no native side effects ({@link #available()}, {@link #skip(long)}, ...).
     * <p>
     * In native, the access are thread safe by construction.
     * Check for closed status must be done in native for the read methods. 
     */
    boolean isClosed;	// VM known

   /**
    * return the number of available address in the MemoryInputStream
    */
    public int available() throws IOException
    {
    	if(isClosed){
    		return 0;
    	}
    	
        return endAddress - currentPtr;
    }
    
    public boolean markSupported()
    {
        return true;
    }

    public synchronized void mark(int readlimit)
    {
    	if (isClosed) {
			return;
		}
        markPtr = currentPtr;
    }

    public synchronized void reset() throws IOException
    {
    	if (isClosed) {
			throw new IOException();
		}
   	
        currentPtr = markPtr;
    }

    public int read(byte[] b) throws IOException
    {
        return readArray(b, 0, b.length);
    }

    public long skip(long n) throws IOException
    {
    	if(isClosed){
    		throw new IOException();
    	}
    	
    	int oldPtr =  currentPtr;
        int available = endAddress - oldPtr;
        
        if(n > available) {
        	n = available;
        }
        
        int newPtr = oldPtr + (int)n;
        if(newPtr < startAddress){
        	// n was negative and we skipped back too many bytes.
        	newPtr = startAddress;
        }
        
        this.currentPtr = newPtr;
        
        return newPtr - oldPtr;
    }
    
    public int read(byte[] b, int off, int len) throws IOException
    {
    	if(isClosed){
    		throw new IOException();
    	}
    	return readArray(b, off, len);
    }
    
    // Implementation known by the VM
    @Override
    public void close() throws IOException {
    	this.isClosed = true;
    }
    
    public native int read() throws IOException;
    public native int readArray(byte[] b, int off, int len) throws IOException;
    
}
