/*
 * Java
 *
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.event;

import java.lang.Thread.UncaughtExceptionHandler;

import ej.annotation.Nullable;
import ej.event.map.PackedMap;

/**
 * EventQueue is an asynchronous communication interface between the native
 * world and the Java world based on events. It allows users to send events from
 * the native world to the Java world.
 * <p>
 * A FIFO mechanism is implemented on the native side and is system specific.
 * The user can offer events to this FIFO.
 * <p>
 * Event notifications are handled using event listeners (Observer design
 * pattern). The application code has to register {@link EventQueueListener} to
 * be notified when new events are coming. To do so, the user can call
 * {@link EventQueue#registerListener(EventQueueListener, int)}.
 * <p>
 * Then the pump automatically retrieves new events pushed in the FIFO and
 * notifies the application via the {@link EventQueueListener}.
 * <p>
 * EventQueue runs on a dedicated Java thread to forward and process events.
 * Application event listener's calls are done in he context of this thread.
 * This thread is started when method {@link EventQueue#getInstance()} is
 * called.
 * <p>
 * Events reading operations are done using the SNI mechanism. The thread is
 * suspended if the events FIFO is empty. The thread resume is done by the
 * native part when a new event is sent if the events FIFO was previously empty.
 * <p>
 * <b> Event format </b>
 * <p>
 * An event is composed of a type and some data. The type identifies the
 * listener that will handle the event. The data is application specific and
 * passed to the listener. The items stored in the FIFO buffer are integers (4
 * bytes). There are two kinds of events that can be sent over the Event Queue:
 * <ul>
 * <li>Standard event: an event with data that fits on 24 bits. The event is
 * stored in the FIFO as a single 4 bytes item.</li>
 * <li>Extended event: an event with data that does not fit on 24 bits. The
 * event is stored in the FIFO as multiple 4 bytes items.</li>
 * </ul>
 *
 * <pre>
 * +--------------+----------+---------------------------------------------------+
 * | Extended (1) | Type (7) | Data (if Extended==0), Length (if Extended==1) (24) |
 * +--------------+----------+---------------------------------------------------+
 * ...
 * +-----------------------------------------------------------------------------+
 * |                  Extended Data for extended events (32)                     | (x integers)
 * +-----------------------------------------------------------------------------+
 * </pre>
 *
 * Format explanation:
 *
 * <ul>
 * <li>Extended (1 bit): event kind flag (0 for non-extended event, 1 for
 * extended event).</li>
 * <li>Type (7 bits): event type, which allows to find the corresponding event
 * listener.</li>
 * <li>Length (24 bits): length of the data in bytes (for extended events
 * only).</li>
 * <li>Data (24 bits): standard event data (for standard events only).</li>
 * <li>Extended data (`Length` bytes): extended event data (for extended events
 * only).</li>
 * </ul>
 *
 * <p>
 *
 * For more information, please have a look at our <a href=
 * "https://docs.microej.com/en/latest/VEEPortingGuide/eventQueue.html">Event
 * Queue documentation</a>.
 *
 */
public class EventQueue {
	private static final String THREAD_NAME = "EventQueue";
	private static final int MAX_TYPE_ID = 127;
	private static final int TYPE_OFFSET = 24;
	private static final int TYPE_MASK = 0x7F;
	private static final int INT_24_BITS = 0xFFFFFF;
	private static final int EXTENDED_EVENT_MASK = 0x7FFFFFFF;

	@Nullable
	private static EventQueue eventQueue = null;

	private final PackedMap<Integer, EventQueueListener> eventListeners;

	private EventQueueListener defaultListener;
	private UncaughtExceptionHandler uncaughtExceptionHandler;

	private int lastType;

	/**
	 * Creates a pump.
	 */
	private EventQueue() {
		this.eventListeners = new PackedMap<>();
		this.lastType = -1;
		this.defaultListener = new NullEventQueueListener();
		this.uncaughtExceptionHandler = new UncaughtExceptionHandler() {
			@Override
			public void uncaughtException(Thread t, Throwable e) {
				UncaughtExceptionHandler uncaughtExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
				if (uncaughtExceptionHandler != null) {
					uncaughtExceptionHandler.uncaughtException(t, e);
				} else {
					e.printStackTrace();
				}
			}
		};
	}

	/**
	 * Sets a handler for the exceptions that occur during event reading or
	 * execution.
	 *
	 * @param uncaughtExceptionHandler the uncaught exception handler to set.
	 */
	public void setUncaughtExceptionHandler(UncaughtExceptionHandler uncaughtExceptionHandler) {
		this.uncaughtExceptionHandler = uncaughtExceptionHandler;
	}

	/**
	 * Sets the default {@code listener}. It will receive all the events whose type
	 * is not handled by any registered listener.
	 *
	 * @param listener the listener to set.
	 */
	public void setDefaultListener(EventQueueListener listener) {
		this.defaultListener = listener;
	}

	/**
	 * Registers a {@code listener} that will received the events of a {@code type}.
	 * <p>
	 * The same listener can be registered for several types.
	 *
	 * @param listener the listener to register.
	 * @param type     the type handled by the {@code listener}.
	 * @throws IllegalArgumentException if the given {@code type} is lower than
	 *                                  <code>0</code> or higher than
	 *                                  <code>127</code>.
	 * @throws IllegalArgumentException if there is already a listener for the event
	 *                                  {@code type}.
	 */
	public synchronized void registerListener(EventQueueListener listener, int type) {
		// Check the type value
		checkTypeValue(type);
		// Check that there is not already a listener for the event type.
		if (this.eventListeners.containsKey(type)) {
			throw new IllegalArgumentException("There is already an EventListener for this type.");
		}

		// Register the EventListener and its type.
		this.eventListeners.put(type, listener);

	}

	/**
	 * Unregister listener for the provided event type
	 *
	 * @param type type of listener to be unregistered
	 */
	public synchronized void unregisterListener(int type) {
		this.eventListeners.remove(type);
	}

	private void checkTypeValue(int type) {
		if (type < 0 || type > MAX_TYPE_ID) {
			throw new IllegalArgumentException(
					"Invalid type value " + type + " (must be between 0 and " + MAX_TYPE_ID + ", inclusive)");
		}
	}

	/**
	 * Inserts an event to the FIFO.
	 * <p>
	 * The {@code data} must not exceed 24 bits. Otherwise use
	 * {@link #offerExtendedEvent(int, byte[])}.
	 *
	 * @param type the type of the event.
	 * @param data the data of the event.
	 * @throws IllegalStateException    if the FIFO is full.
	 * @throws IllegalArgumentException if the given {@code type} is lower than
	 *                                  <code>0</code>.
	 * @throws IllegalArgumentException if the {@code data} exceeds 24 bits.
	 */
	public void offerEvent(int type, int data) {
		// Check the type value
		checkTypeValue(type);

		if ((data & INT_24_BITS) != data) {
			throw new IllegalArgumentException("Data exceed 24-bit.");
		}
		if (!EventQueueNatives.offerEvent(type, data)) {
			throw new IllegalStateException("The FIFO is full.");
		}
	}

	/**
	 * Inserts an extended event to the FIFO.
	 * <p>
	 * When the data exceeds 24 bits, it can be split into several integers. These
	 * integers are pushed consecutively to the FIFO. The listener will receive all
	 * these integers at once and will be able to rebuild a complex data.
	 *
	 * @param type the type of the events.
	 * @param data the data of the events.
	 * @throws IllegalStateException    if the FIFO is full.
	 * @throws IllegalArgumentException if the given {@code type} is lower than
	 *                                  <code>0</code>.
	 * @throws IllegalArgumentException if the length of {@code data} buffer exceeds
	 *                                  24 bits.
	 */
	public void offerExtendedEvent(int type, byte[] data) {
		// Check the type value
		checkTypeValue(type);

		if ((data.length & INT_24_BITS) != data.length) {
			throw new IllegalArgumentException("Data exceeds 24 bits.");
		}
		if (!EventQueueNatives.offerExtendedEvent(type, data, data.length)) {
			throw new IllegalStateException("The FIFO is full.");
		}
	}

	/**
	 * Gets an instance of a started EventQueue.
	 *
	 * @return the instance of the queue.
	 */
	public static synchronized EventQueue getInstance() {
		EventQueue instance = EventQueue.eventQueue;
		if (instance == null) {
			instance = new EventQueue();
			EventQueue.eventQueue = instance;
			instance.start();
		}
		return instance;
	}

	/**
	 * Runs the pump on the current thread.
	 * <p>
	 * The pump loops on:
	 * <ol>
	 * <li>read a event from the FIFO (pop),</li>
	 * <li>handle this event by sending it to the listener that manages the type of
	 * the event (if any registered).</li>
	 * </ol>
	 */
	private void runThread() {

		while (true) { // NOSONAR the pump never stops.
			try {
				int event = EventQueueNatives.waitEvent();
				if (event >= 0) {
					// Not extended event
					handleEvent(event);
				} else {
					// Extended event.
					handleExtendedEvent(event & EXTENDED_EVENT_MASK);
				}
			} catch (Throwable t) { // NOSONAR Pump thread Errors should not be handled by the JVM.
				this.uncaughtExceptionHandler.uncaughtException(Thread.currentThread(), t);
			}
		}
	}

	/**
	 * Handles an event, checking whether a listener is registered for the event
	 * type. If yes, call {@link EventQueueListener#handleEvent(int, int)}. If not,
	 * call the default listener.
	 *
	 * @param event the event to handle.
	 */
	private void handleEvent(int event) {
		// Process the data of the event. 0 (1) | Type (7) | Data (24)
		int type = getEventType(event);

		// Get the listener corresponding to the type. If it exists, handle the event.
		// If not use the default listener.
		EventQueueListener eventListener;
		synchronized (this) {
			eventListener = this.eventListeners.get(type);
		}
		if (eventListener != null) {
			eventListener.handleEvent(type, getEventData(event));
		} else {
			this.defaultListener.handleEvent(type, getEventData(event));
		}
	}

	/**
	 * Handles an extended event.
	 *
	 * @param event the event to handle.
	 */
	private void handleExtendedEvent(int event) {
		// Process the data of the event. 1 (1) | Type (7) | Data Length (24)
		int type = getEventType(event);
		int dataLength = getEventData(event);

		// Create the Data reader related to the event.
		EventDataReader eventQueueDataReader = new EventQueueDataReader();

		// Get the listener corresponding to the type. If it exists, handle the event.
		// If not use the default listener.
		EventQueueListener eventListener;
		synchronized (this) {
			eventListener = eventListeners.get(type);
		}
		// Start to handle the extended data.
		EventQueueNatives.startReadExtendedData(dataLength);
		if (eventListener != null) {
			eventListener.handleExtendedEvent(type, eventQueueDataReader);
		} else {
			this.defaultListener.handleExtendedEvent(type, eventQueueDataReader);
		}
		// Check that the extended data has been read entirely.
		EventQueueNatives.endReadExtendedData();
	}

	/**
	 * Executes this {@link EventQueue} in a new thread.
	 */
	private void start() {
		EventQueueNatives.initialize();
		Thread t = new Thread(new Runnable() {

			@Override
			public void run() {
				runThread();

			}
		}, THREAD_NAME);
		t.setDaemon(true);
		t.start();

	}

	/**
	 * Get an unique Id to register a Listener.
	 *
	 * @return the unique Id corresponding to the name.
	 * @throws IllegalStateException if the maximum number of types
	 *                               (<code>127</code>) has been reached.
	 */
	public synchronized int getNewType() {
		if (this.lastType >= MAX_TYPE_ID) {
			throw new IllegalStateException(
					"Internal limits reached, you can not have more than " + MAX_TYPE_ID + " event types.");
		}
		this.lastType += 1;
		return this.lastType;
	}

	/**
	 * Gets the type of an event.
	 *
	 * @param event the event.
	 * @return the type of the event.
	 */
	private static int getEventType(int event) {
		return ((event >> TYPE_OFFSET) & TYPE_MASK);
	}

	/**
	 * Gets the data of an event.
	 *
	 * @param event the event.
	 * @return the data of the event.
	 */
	private static int getEventData(int event) {
		return event & INT_24_BITS;
	}

}
