/*
 * Java
 *
 * Copyright 2013-2022 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.kf;

import com.is2t.kf.KernelNatives;
import com.is2t.kf.ModuleData;
import com.is2t.schedcontrol.ScheduleContext;
import com.is2t.schedcontrol.ScheduleController;

/**
 * A {@link Module} is either Kernel or a Feature.
 */
public class Module {

	//WARNING: don't modify basetype or reference fields here.
	//The VM knows the structure of the Feature class that is a subclass of Module.

	protected ModuleData data; // VM Known
	protected ScheduleContext scheduleContext; // VM Known

	Module(ModuleData data){
		this.data = data;
		this.scheduleContext = newDefaultScheduleContext();
	}

	public String getName(){
		return new String(data.name);
	}

	public String getVersion(){
		return new String(data.version);
	}

	public Principal getProvider(){
		return data.owner;
	}

	public byte[] getUID(){
		return data.uid;
	}

	protected ScheduleContext newDefaultScheduleContext() {
		ScheduleContext context = ScheduleController.getScheduleController().newContext(-1);
		assert (context != null);
		return context;
	}

	public void setExecutionQuota(int quota) {
		scheduleContext.setQuantumQuota(quota);
		ScheduleController.getScheduleController().resetAllQuantumCounters(); // reset all counters and optionally enables the quantum counter, as indicated in the API)
	}

	public int getExecutionQuota() {
		return scheduleContext.getQuantumQuota();
	}

	public long getExecutionCounter() {
		return scheduleContext.getQuantumCounter();
	}

	/**
	 * Returns the amount of allocated memory by this module.
	 *
	 * <p>
	 * To get an accurate value of the allocated memory, please run the garbage
	 * collector before calling this method.
	 *
	 * @return the amount of allocated memory by this module, measured in bytes.
	 */
	public long getAllocatedMemory() {
		return KernelNatives.getAllocatedMemory(this);
	}

	@Override
	public String toString() {
		return getName();
	}
}
