/*
 * Java
 *
 * Copyright 2013-2023 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.kf;

import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;

import com.is2t.kf.DefaultExceptionHandler;
import com.is2t.kf.FeatureFinalizer;
import com.is2t.kf.IFeatureLoader;
import com.is2t.kf.KFResourceManager;
import com.is2t.kf.KernelNatives;
import com.is2t.kf.KernelNativesConstants;
import com.is2t.kf.KernelSupport;
import com.is2t.kf.ModuleData;
import com.is2t.tools.ArrayTools;

import ej.annotation.Nullable;
import ej.bon.Constants;
import ej.kf.Feature.State;
import ej.lang.ResourceManager;

public class Kernel extends Module {
	/**
	 * This field is synchronized using Kernel.class monitor
	 * May contains null entries
	 */
	private static Feature[] LoadedFeatures;

	/**
	 * This field is synchronized using Kernel.class monitor
	 */
	static FeatureStateListener[] Listeners;

	/**
	 * This field is synchronized using Kernel.class monitor
	 */
	static ResourceControlListener[] ResourceControlListeners;

	/**
	 * Sorted list of Converters : a deepest type appears before its super types.
	 * Last element is the default converter for <code>java.lang.Object</code>.
	 * No synchronization (// add or remove may be lost but will not let the pool of converters in an invalid state).
	 */
	static Converter<?>[] Converters;

	private static Kernel INSTANCE; // VM Known
	public static UncaughtExceptionHandler handler;

	/**
	 * Ordered list of Feature loaders
	 */
	private static IFeatureLoader[] FEATURE_LOADERS;

	public static final String KF_PROPERTY_PREFIX = "com.is2t.kf.";

	/**
	 * Timeout Constant name used for the call of {@link FeatureEntryPoint#stop()}, see the kernel.constants.list for the default value.
	 */
	public static final String STOP_TIMEOUT_CONSTANT_NAME = "com.microej.runtime.kf.waitstop.delay";


	/**
	 * Timeout for the call of {@link FeatureEntryPoint#stop()}.
	 */
	/*default*/ static final long STOP_TIMEOUT;


	static{
		STOP_TIMEOUT = Constants.getLong(Kernel.STOP_TIMEOUT_CONSTANT_NAME);
		Listeners = new FeatureStateListener[0];
		ResourceControlListeners = new ResourceControlListener[0];
		if(!KernelNatives.initialize()){
			// Should not occur: wrong library build
			throw new AssertionError();
		}

		LoadedFeatures = new Feature[0];

		INSTANCE = new Kernel(KernelNatives.getData(KernelNativesConstants.KERNEL_DESCRIPTOR));
		handler = new DefaultExceptionHandler();
		ResourceManager.setResourceManager(new KFResourceManager());

		String property = System.getProperty(new StringBuilder(KF_PROPERTY_PREFIX).append("dynamic.loader").toString());
		IFeatureLoader[] loaders;
		if(property != null){
			IFeatureLoader featureLoader;
			try{ featureLoader = (IFeatureLoader)Class.forName(property).newInstance(); }
			catch(Throwable e){
				// Should not occur: wrong library build
				throw new AssertionError(e);
			}

			loaders = new IFeatureLoader[] { featureLoader };
			try {
				featureLoader.loadInstalledFeatures();
			} catch (Throwable e) {
				throw new ExceptionInInitializerError(e);
			}
		}
		else{
			loaders = new IFeatureLoader[0];
		}
		FEATURE_LOADERS = loaders;

		// Initialize converters and install the default one (anonymous proxies on java.lang.Object)
		Converters = new Converter[0];
		addConverter(new Converter<Object>() {
			@Override
			public Object convert(Object source, Feature targetOwner) {
				Object convertedObject = bindProxy(source, null, targetOwner);
				assert (convertedObject != null);
				return convertedObject;
			}
			@Override
			public Class<Object> getType() {
				return Object.class;
			}
		});

		FeatureFinalizer.initialize();
	}

	private int minMemory;

	private Kernel(ModuleData data){
		super(data);
		// attach the current thread (the main thread that is executing clinit method to the current schedule context)
		this.scheduleContext.addThread(Thread.currentThread());
	}

	private static final int SIGNATURE_LENGTH = 4;

	public static Kernel getInstance(){
		return INSTANCE;
	}

	public static Feature load(InputStream is) throws IOException, InvalidFormatException, IncompatibleFeatureException, AlreadyLoadedFeatureException{
		Feature f = install(is);
		f.start();
		return f;
	}

	public static Feature install(InputStream is) throws IOException, InvalidFormatException, IncompatibleFeatureException, AlreadyLoadedFeatureException{
		IFeatureLoader[] loaders = FEATURE_LOADERS;
		int nbLoaders = loaders.length;
		byte[] signature = new byte[SIGNATURE_LENGTH];
		try{ new DataInputStream(is).readFully(signature); }
		catch(EOFException e){} // input stream lower than 4 bytes (most likely a static feature name)
		IFeatureLoader loader;
		findLoader:{
			for(int i=-1; ++i<nbLoaders;){ // ordered
				loader = loaders[i];
				if(loader.canLoad(signature)){
					break findLoader;
				}
			}
			throw new InvalidFormatException(InvalidFormatException.NO_ELIGIBLE_LOADER);
		}

		Feature f;
		synchronized (loader) { // see IFeatureLoader comment
			f = loader.load(signature, is);
		}

		// Here, the Feature has been successfully loaded and is in the INSTALLED state.
		addInstalledFeature(f);
		return f;
	}

	/**
	 * Add a rightly loaded Feature to the set of installed Features and notify
	 * {@link FeatureStateListener}
	 *
	 * @param f the loaded Feature
	 */
	public static void addInstalledFeature(Feature f) {
		// Add it to the loaded features list
		synchronized(Kernel.class){
			insertFeature:{
			for(int i=LoadedFeatures.length; --i>=0;){
				if(LoadedFeatures[i] == null){
					LoadedFeatures[i] = f;
					break insertFeature;
				}
			}
			// here, array is full: append at the end
			LoadedFeatures = (Feature[]) ArrayTools.add(LoadedFeatures, f);
		}
		}

		f.notifyStateChanged(null);
	}

	/**
	 * Start sequence of a Feature (in its thread)
	 */
	static void start(final Feature f) {

		Thread thread = new Thread(new Runnable() {

			@Override
			public void run() {
				// Initialize prior to any Feature code execution
				f.initialize();

				try {
					// Execute clinits
					KernelNatives.execClinitWrapper(f);
				} catch (RuntimeException t) {
					// all runtime exceptions that occurred in clinit are converted in
					// ExceptionInInitializerError
					// See jvms7.pdf - Section 5.5 - Initialization
					throw new ExceptionInInitializerError(t);
				}

				// instantiate entry point (this trigger constructor code)
				FeatureEntryPoint fep;
				try { fep = f.initEntryPoint(); }
				catch (InstantiationException e) {
					KernelSupport.uncaughtException(f, e);
					return;
				}
				catch (IllegalAccessException e) {
					KernelSupport.uncaughtException(f, e);
					return;
				}

				// run start() method
				try { fep.start(); }
				catch(Throwable e){
					KernelSupport.uncaughtException(f, e);
					return;
				}
			}}, f.getProvider().getValue(Principal.FIELD_CN));
		KernelNatives.setStarted(f);

		// set thread object owned by this Feature: this is the root thread of the Feature
		KernelNatives.setOwner(thread, f);

		// start (Feature context)
		f.scheduleContext.startThread(thread);
	}

	public static synchronized Feature[] getAllLoadedFeatures(){
		ArrayList<Feature> featuresVect = new ArrayList<Feature>();
		for(Feature f : LoadedFeatures){
			if(f != null) {
				featuresVect.add(f);
			}
		}
		Feature[] loadedFeatures = featuresVect.toArray(new Feature[featuresVect.size()]);
		assert (loadedFeatures != null);
		return loadedFeatures;
	}

	/**
	 * Get the owner of the given {@link Object}.
	 */
	public static synchronized Module getOwner(Object o){
		if(o == null) {
			throw new NullPointerException();
		}
		return KernelNatives.getOwner(o);
	}

	/**
	 * Get the owner of the current thread context.
	 */
	public static synchronized Module getContextOwner(){
		return KernelNatives.getContextOwner();
	}

	/**
	 * Get the {@link Feature} that owns the currently thread context.
	 * @return null if {@link #isInKernelMode()}
	 */
	@Nullable
	public static synchronized Feature getContextOwnerFeature(){
		Module owner = KernelNatives.getContextOwner();
		if(owner == INSTANCE){
			return null;
		} else {
			return (Feature) owner;
		}
	}

	/*
	 * WARNING: The number of levels to the native call is well known by the VM.
	 * Is it actually set to 2. If the number of calls change, KFNativesPool.enter() need to be updated.
	 */
	public static void enter(){
		KernelNatives.enter();
	}

	/*
	 * WARNING: The number of levels to the native call is well known by the VM.
	 * Is it actually set to 2. If the number of calls change, KFNativesPool.exit() need to be updated.
	 */
	public static void exit(){
		KernelNatives.exit();
	}

	public static void setUncaughtExceptionHandler(@Nullable UncaughtExceptionHandler handler) {
		if(handler == null) {
			handler = new DefaultExceptionHandler();
		}
		Kernel.handler = handler;
	}

	public static boolean isInKernelMode(){
		return Kernel.getContextOwner() == Kernel.getInstance();
	}

	/**
	 * Returns the Feature with the given section descriptor.
	 *
	 * @return null if not found
	 */
	@Nullable
	public static Feature getFeatureByDescriptor(long sectionDescriptor) {
		Feature[] loadedFeatures = LoadedFeatures;// take a snapshot. No synchronization required.
		for (Feature f : loadedFeatures) {
			if (f.getSectionDescriptor() == sectionDescriptor) {
				return f;
			}
		}
		return null;
	}

	/**
	 * Returns the first Feature with the given name.
	 * @return null if not found
	 */
	@Nullable
	public static Feature getFeatureByName(String name){
		Feature[] loadedFeatures = LoadedFeatures;// take a snapshot. No synchronization required.
		for (Feature f : loadedFeatures) {
			if(f.getName().equals(name)){
				return f;
			}
		}
		return null;
	}


	public static void uninstall(Feature f){
		f.uninstall();
	}

	public static boolean canUninstall(Feature f){
		return f.canUninstall();
	}

	/**
	 * Remove the given {@link Feature} from the loaded features list.
	 * @param featureToRemove the feature to remove
	 */
	/*default*/ synchronized static void removeFromLoadedFeatures(Feature featureToRemove){
		// remove reference from the global array
		for(int i=LoadedFeatures.length; --i>=0;){
			if(LoadedFeatures[i] == featureToRemove){
				LoadedFeatures[i] = null;
			}
		}
	}

	public static boolean unload(Feature f) throws UnknownFeatureException{
		return f.unload();
	}

	public static synchronized void addFeatureStateListener(FeatureStateListener listener){
		if(listener == null) {
			throw new NullPointerException();
		}
		Listeners = (FeatureStateListener[])ArrayTools.add(Listeners, listener);
	}

	/**
	 * Removes the {@link FeatureStateListener} to the list of listeners
	 * that are notified when the state of a Feature has changed.
	 * @param listener the listener to be removed
	 */
	public static synchronized void removeFeatureStateListener(FeatureStateListener listener){
		Listeners = (FeatureStateListener[]) ArrayTools.remove(Listeners, listener);
	}

	/**
	 * Adds the given {@link ResourceControlListener} to the list of listeners
	 * that are notified when a Feature is stopped by the Resource Control Manager.
	 * @param listener the new listener to add
	 * @throws NullPointerException if listener is <code>null</code>
	 */
	public static synchronized void addResourceControlListener(ResourceControlListener listener){
		if(listener == null) {
			throw new NullPointerException();
		}
		ResourceControlListeners = (ResourceControlListener[])ArrayTools.add(ResourceControlListeners, listener);
	}

	/**
	 * Removes the {@link ResourceControlListener} to the list of listeners
	 * that are notified when a Feature is stopped by the Resource Control Manager.
	 * <p>
	 * Does nothing if the listener is not registered or <code>null</code>.
	 * @param listener the listener to be removed
	 */
	public static synchronized void removeResourceControlListener(ResourceControlListener listener) {
		ResourceControlListeners = (ResourceControlListener[]) ArrayTools.remove(ResourceControlListeners, listener);
	}

	public static synchronized FeatureStateListener[] getAllFeatureStateListeners() {
		return Listeners.clone();
	}

	/**
	 * This API is known by the VM.
	 */
	@Nullable
	public static <T> T bind(@Nullable T o, Class<T> targetType, Feature targetOwner) {
		checkFeatureStarted(targetOwner);

		if (o == null) {
			return null;
		}

		if(Boolean.getBoolean(new StringBuilder(Kernel.KF_PROPERTY_PREFIX).append("s3.mode").toString())){
			// On Simulator, objects are returned directly (no proxys).
			// This is an approximation of the API since it does not return a subclass of Proxy.
			// But the MicroEJ libraries (Wadapps) do not rely on this assumption.
			// The property 'com.is2t.kf.s3.mode' is known by:
			// - the simulator that defines this property
			// - hokapp
			return o;
		}
		else if(Kernel.getOwner(targetType) == targetOwner){
			return bindProxy(o, targetType, targetOwner);
		}
		else{
			return convert(o, targetType, targetOwner);
		}
	}

	/**
	 * @param targetInterface may be null if building an anonymous {@link Proxy}.
	 */
	@SuppressWarnings("unchecked")
	@Nullable
	private static <T> T bindProxy(T source, @Nullable Class<T> targetInterface, Feature targetOwner) {
		Object o = source;
		if(o instanceof Proxy){
			// remove Proxy wrapper
			o = ((Proxy<?>)o).ref;
		}

		if (o == null) {
			return null;
		}

		if(getOwner(o) == targetOwner){
			if(targetInterface == null || targetInterface.isInstance(o)){
				return (T)o; // true for sure
			}
			else{
				throw new IllegalAccessError();
			}
		}
		return targetOwner.newProxy(o, targetInterface);
	}

	public static void runUnderContext(Module contextOwner, final Runnable runnable){
		// checkFeatureStarted is done by KernelNatives.setOwner().
		Kernel.enter();
		Module runnableOwner = getOwner(runnable);
		Kernel kernel = Kernel.getInstance();
		if(contextOwner == kernel){
			if(runnableOwner != kernel){
				throw new IllegalAccessError();
			}
			else{
				runnable.run();
			}
		}
		else{
			Feature feature = (Feature)contextOwner;

			if(runnableOwner != kernel){
				// runnable owned by a Feature
				if(runnableOwner != feature){
					throw new IllegalAccessError();
				}
				else{
					// direct call
					runnable.run();
				}
			}
			else{
				// runnable owned by the Kernel
				// create a wrapper to ensure the runnable has no reference fields
				Runnable runnableWrapper = new Runnable() {

					@Override
					public void run() {
						runnable.run();
					}
				};
				KernelNatives.setOwner(runnableWrapper, feature);
				runnableWrapper.run();
			}
		}
	}

	public static void setOwner(Object object, Module newOwner){
		// checkFeatureStarted is done by KernelNatives.setOwner()
		if(object == null || newOwner == null){
			throw new NullPointerException();
		}
		Feature featureOwner;
		if(newOwner == INSTANCE){
			featureOwner = null;
		}
		else {
			featureOwner = (Feature) newOwner;
		}
		KernelNatives.setOwner(object, featureOwner);
	}

	public static <T> T clone(T from, Module toOwner) throws CloneNotSupportedException {
		// checkFeatureStarted is done by KernelNatives.clone()
		if(from == null || toOwner == null){
			throw new NullPointerException();
		}
		if(from instanceof String){
			// cast String to T for sure
			@SuppressWarnings("unchecked")
			T clone = (T) cloneString((String)from, toOwner);
			return clone;
		}
		if(!(from instanceof Cloneable || from.getClass().isArray())) {
			throw new CloneNotSupportedException();
		}

		Feature toFeatureOwner;
		if(toOwner == INSTANCE){
			toFeatureOwner = null;
		}
		else {
			toFeatureOwner = (Feature) toOwner;
		}
		// cast ensured by native
		@SuppressWarnings("unchecked")
		T clone = (T) KernelNatives.clone(from, toFeatureOwner);
		return clone;
	}

	private static String cloneString(String from, Module toOwner) {
		Kernel.enter();

		final char[] chars = from.toCharArray(); // new allocation owned by from owner
		assert (chars != null);
		setOwner(chars, toOwner);

		final String[] resultTmp = new String[1];

		runUnderContext(toOwner, new Runnable() {

			@Override
			public void run() {
				String s = new String();
				s.chars = chars;
				s.offset = 0;
				s.length = chars.length;

				// store
				Kernel.enter();
				resultTmp[0] = s;
			}
		});
		String result = resultTmp[0];
		assert (result != null);
		return result;
	}

	public static void addConverter(Converter<?> converter){
		Class<?> type = converter.getType(); // throws NullPointerException if converter is null
		Converter<?>[] converters = Converters;
		int nbConverters = converters.length;
		int insertIndex = 0;
		for(int i=-1; ++i<nbConverters;){
			Converter<?> current = converters[i];
			Class<?> currentType = current.getType();
			if(currentType == type){
				throw new IllegalArgumentException(); // already a converter
			}
			if(currentType.isAssignableFrom(type)){
				// insert before
				insertIndex = i;
				break;
			}
		}

		Converter<?>[] newConverters = new Converter<?>[nbConverters+1];
		if(insertIndex > 0){
			System.arraycopy(converters, 0, newConverters, 0, insertIndex);
		}
		if(nbConverters-insertIndex > 0){
			System.arraycopy(converters, insertIndex, newConverters, insertIndex+1, nbConverters-insertIndex);
		}
		newConverters[insertIndex] = converter;
		Converters = newConverters;
	}

	public static void removeConverter(Converter<?> converter) {
		Converters = (Converter[]) ArrayTools.remove(Converters, converter);
	}

	/**
	 * @return null if not found
	 */
	@SuppressWarnings("unchecked")
	@Nullable
	public static <T> Converter<T> getConverter(Class<? extends T> runtimeType, Class<? extends T> compiletimeType){
		Converter<?>[] converters = Converters;
		int nbConverters = converters.length;
		for(int i=-1; ++i<nbConverters;){ // keep order to find the first closed converter
			Converter<?> current = converters[i];
			Class<?> currentType = current.getType();
			if(currentType.isAssignableFrom(runtimeType) && compiletimeType.isAssignableFrom(currentType)){
				return (Converter<T>)current;
			}
		}
		return null;
	}

	/**
	 * @param o a Feature object
	 * @param compiletimeType the compile time Kernel type declared in the shared interface
	 * @param targetOwner the owner of the converted object
	 * @throws IllegalAccessError if Kernel does not define any converter for this type, or if the converted Object is not owned by the target feature
	 */
	@SuppressWarnings("unchecked")
	@Nullable
	protected static <T> T convert(T o, Class<? extends T> compiletimeType, Feature targetOwner){
		if(compiletimeType.isInterface()){
			Class<?> si = KernelNatives.getImplementedSharedInterface(o.getClass(), compiletimeType);
			if(si != null){
				Class<T> targetSI = (Class<T>)KernelNatives.getTargetSharedInterface(si, compiletimeType, targetOwner);
				if(targetSI != null){
					// o implements a shared interface that extends the Kernel type and the shared interface exists to the targetOwner side
					// => convert using a proxy
					return bindProxy(o, targetSI, targetOwner);
				}
			}
		}

		// Kernel type conversion using a registered Kernel type converter
		Converter<T> converter = Kernel.getConverter((Class<T>)o.getClass(), compiletimeType);
		if(converter == null){
			throw new IllegalAccessError();
		}

		T result = converter.convert(o, targetOwner);
		// check the reference returned by the converter
		Module objectOwner = Kernel.getOwner(result);
		if(objectOwner != targetOwner){
			throw new IllegalAccessError();
		}
		return result;
	}

	public static boolean isSharedInterface(Class<?> c){
		return KernelNatives.isSharedInterface(c);
	}

	@SuppressWarnings("unchecked")
	public static <T> Proxy<T> newProxy(T ref, Module owner){
		// checkFeatureStarted is done by KernelNatives.newAnonymousProxy()
		Feature featureOwner;
		if(owner == INSTANCE){
			featureOwner = null;
		}
		else {
			featureOwner = (Feature) owner;
		}
		return (Proxy<T>) KernelNatives.newAnonymousProxy(ref, featureOwner);
	}

	public static boolean isAPI(Class<?> c){
		return KernelNatives.isAPI(c);
	}

	public static boolean areEquivalentSharedInterfaces(Class<?> si1, Class<?> si2){
		return KernelNatives.areEquivalentSharedInterfaces(si1, si2);
	}

	@Nullable
	public static Class<?> getSharedInterface(Class<?> si, Class<?> top, Feature target){
		if(!isSharedInterface(si) || (!isSharedInterface(top) && Kernel.getOwner(top) != INSTANCE) || !isAssignableFrom(top, si)){
			throw new IllegalArgumentException();
		}
		if(Kernel.getOwner(si) == target){
			return si;
		}
		return KernelNatives.getTargetSharedInterface(si, top, target);
	}

	@Nullable
	public static Class<?> getEquivalentSharedInterface(Class<?> si, Feature target){
		if(!isSharedInterface(si)){
			throw new IllegalArgumentException();
		}
		if(Kernel.getOwner(si) == target){
			return si;
		}
		return KernelNatives.getTargetSharedInterface(si, si, target);
	}

	@Nullable
	public static Class<?> getImplementedSharedInterface(Class<?> c, Class<?> topInterface){
		if(Kernel.getOwner(c) == INSTANCE || c.isArray() || !topInterface.isInterface() || !isAssignableFrom(topInterface, c)){
			throw new IllegalArgumentException();

		}
		return KernelNatives.getImplementedSharedInterface(c, topInterface);
	}

	/**
	 * Test <code>c1.isAssignablefrom(c2)</code> with classes that may be owned by 2 Features
	 * @param from
	 * @param to
	 * @return true if <code>c1.isAssignablefrom(c2)</code>, false otherwise
	 */
	private static boolean isAssignableFrom(Class<?> c1, Class<?> c2){
		try{
			return c1.isAssignableFrom(c2);
		}
		catch(IllegalAccessError e){
			// 2 classes from 2 different features (not assignable from)
			return false;
		}
	}

	private static void checkFeatureStarted(Feature feature) {
		State featureState = feature.getState();
		if (featureState != State.STARTED) {
			throw new IllegalStateException(featureState.toString());
		}
	}

	public boolean setReservedMemory(long size) {
		return KernelNatives.setReservedMemory(size);
	}

	public long getReservedMemory() {
		return KernelNatives.getReservedMemory();
	}
}