/*
 * Java
 *
 * Copyright 2013-2023 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.kf;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import com.is2t.kf.FeatureFinalizer;
import com.is2t.kf.IFeatureLoader;
import com.is2t.kf.KernelNatives;
import com.is2t.kf.KernelSupport;
import com.is2t.kf.ModuleData;
import com.is2t.kf.ProxyRef;

import ej.annotation.Nullable;
import ej.kf.ResourceControlListener.StopCause;
import ej.lang.Resource;



public class Feature extends Module{

	// WARNING: don't add instance field (reference or basetype)
	// without modifying, in the VM, the references to "VM Known" fields.

	// WARNING: don't synchronize on a Feature without calling checkNotInFeatureFinalizerThread()
	// before.

	/**
	 * The minimum criticality that a feature can have.
	 * @see #getCriticality()
	 */
	public static final int MIN_CRITICALITY = 0; // VM Known

	/**
	 * The default criticality that is assigned to a feature.
	 * @see #getCriticality()
	 */
	public static final int NORM_CRITICALITY = 5; // VM Known

	/**
	 * The maximum criticality that a feature can have.
	 * @see #getCriticality()
	 */
	public static final int MAX_CRITICALITY = 10; // VM Known

	public static enum State{
		INSTALLED,
		STARTED,
		STOPPED,
		UNINSTALLED,
	}

	// States in the native space:
	// Synchronize with values in FeatureSectionStructRam
	public static final int NATIVE_STATE_INSTALLED = 1; // VM Known
	public static final int NATIVE_STATE_STARTED = 2; // VM Known
	public static final int NATIVE_STATE_STOPPED = 3; // VM Known
	public static final int NATIVE_STATE_UNINSTALLED = 4; // VM Known

	// Stop cause values:
	/**
	 * The {@link Feature} has not been stopped.
	 */
	private static final int STOP_CAUSE_NOT_STOPPED = 0; // VM Known
	/**
	 * The {@link Feature} has been stopped explicitly with a call to {@link Feature#stop()}.
	 */
	private static final int STOP_CAUSE_KERNEL = 1; // VM Known
	/**
	 * The {@link Feature} has been stopped by the Memory Control.
	 */
	private static final int STOP_CAUSE_MEMORY = 2; // VM Known
	/**
	 * The {@link Feature} has been stopped by the Watchdog.
	 */
	private static final int STOP_CAUSE_WATCHDOG = 3; // VM Known

	// Stop step values:
	// WARNING: keep order (see waitForStopStep())
	/**
	 * The {@link Feature} has not been stopped ({@link #NATIVE_STATE_STARTED} or {@value #NATIVE_STATE_UNINSTALLED} state).
	 */
	private static final int STOP_STEP_NOT_STOPPED = 0; // VM Known
	/**
	 * The {@link Feature} has been stopped but the {@link FeatureStateListener}s have not been called.
	 */
	private static final int STOP_STEP_STOPPING = 1; // VM Known
	/**
	 * The {@link Feature} has been stopped and the {@link FeatureStateListener}s have been called.
	 */
	private static final int STOP_STEP_STOPPED = 2; // VM Known
	/**
	 * The {@link Feature} has moved from {@link #NATIVE_STATE_STOPPED} state to {@value #NATIVE_STATE_INSTALLED} but the {@link FeatureStateListener}s have not been called.
	 */
	private static final int STOP_STEP_REINSTALLING = 3; // VM Known
	/**
	 * The {@link Feature} has moved from {@link #NATIVE_STATE_STOPPED} state to {@value #NATIVE_STATE_INSTALLED} and the {@link FeatureStateListener}s have been called.
	 * Or the {@link Feature} has just been created and installed.
	 */
	private static final int STOP_STEP_INSTALLED = 4; // VM Known

	private static final int DEFAULT_PROXY_NB_KEYS = 32;

	/**
	 * Feature handle cannot be allocated
	 */
	public static final int FEATURE_HANDLE_NONE = 0; // VM Known

	/**
	 * Compile-time constant indicating the quota control is disabled (always
	 * eligible to scheduling).
	 */
	public static final int QUOTA_DISABLED = -1; // VM Known

	/**
	 * FeatureSectionStruct id in VM side. Most likely a native address (64 bits
	 * compatible). 0 is reserved.
	 * <p>
	 * This value is valid while the feature is not uninstalled. Set to -1 when the
	 * feature is uninstalled.
	 * <p>
	 * This field MUST not be used in the Java.
	 * <p>
	 * WARNING: VM Known
	 */
	private long _featureSectionDescriptor; //VM Known

	/**
	 * One of the <p>STOP_CAUSE_*</p> values.
	 */
	private byte stopCause; // VM Known (set by the VM)

	/**
	 * One of the <p>STOP_STEP_*</p> values.
	 */
	private byte stopStep; // VM Known

	/**
	 * The Feature handle
	 */
	private final int featureHandle; // VM Known

	/**
	 * Initialized during feature startup.<p>
	 * WARNING: This object is owned by the Feature and access to it may throw a {@link DeadFeatureException}.
	 * <p>
	 * WARNING: VM Known
	 */
	private @Nullable FeatureEntryPoint entryPoint; // VM Known

	/**
	 * The list of resources currently open by this Feature (owned by the Feature).<p>
	 * WARNING: This object is owned by the Feature and access to it may throw a {@link DeadFeatureException}.
	 * <p>
	 * WARNING: VM Known
	 */
	public @Nullable ArrayList<Resource> openResources; // VM Known

	/**
	 * Pool of proxies that have been allocated for this Feature.
	 * The index of the proxy (the key) is equals to the proxy target object hashcode % the length of the array.
	 * Proxies sharing the same key are chained.
	 * Lazily initialized.<p>
	 * WARNING: This object is owned by the Feature and access to it may throw a {@link DeadFeatureException}.
	 * <p>
	 * WARNING: VM Known
	 */
	public @Nullable ProxyRef[] proxies; // VM Known
	private final Object proxyMonitor;

	/**
	 * Loader of the feature
	 */
	private final IFeatureLoader loader;

	private final Object featureStateListenerMonitor;

	/**
	 * Initial capacity of the {@link ArrayList} created for {@link #openJavaResources}.
	 */
	private static final int OPENED_JAVA_RESOURCES_INITIAL_CAPACITY = 1;

	/**
	 * List of the Java resources opened with {@link #getResourceAsStream(String)}.
	 * These resources are automatically closed when this Feature is uninstalled.
	 * <p>
	 * WARNING: Every accesses to this list must be synchronized.
	 * <p>
	 * NOTE: use {@link ArrayList} instead of {@link List} to use the {@link ArrayList#trimToSize()} method.
	 */
	private final ArrayList<FeatureResourceInputStream> openJavaResources;

	public Feature(IFeatureLoader loader, long featureSectionDescriptor, int featureHandle) {
		super(ModuleData.kernelClone(KernelNatives.getData(featureSectionDescriptor)));
		assert featureHandle != Feature.FEATURE_HANDLE_NONE;
		KernelNatives.linkToNative(this, featureSectionDescriptor);
		this.loader = loader;
		this.proxyMonitor = new Object();
		this.featureStateListenerMonitor = new Object();
		this.featureHandle = featureHandle;
		this.openJavaResources = new ArrayList<FeatureResourceInputStream>(OPENED_JAVA_RESOURCES_INITIAL_CAPACITY);
	}

	/**
	 * Initialize method (in Feature context), before any Feature code execution.
	 */
	public void initialize(){
		ArrayList<Resource> openResources = new ArrayList<Resource>();
		// Register instances to Feature object
		Kernel.enter();
		this.openResources = openResources;
		Kernel.exit();
	}

	/**
	 * Creates the entry point of this feature and returns it. Must be called once.
	 * 	Called in the Feature context.
	 * @throws IllegalAccessException
	 * @throws InstantiationException
	 */
	/*default*/ FeatureEntryPoint initEntryPoint() throws InstantiationException, IllegalAccessException{
		if(entryPoint != null){
			throw new IllegalArgumentException();
		}


		// instantiate entry point (this trigger constructor code)
		Class<?> c = KernelNatives.getEntryPointClass(this);
		FeatureEntryPoint entryPoint = (FeatureEntryPoint)c.newInstance();
		assert (entryPoint != null);

		// Register instances to Feature object
		Kernel.enter();
		this.entryPoint = entryPoint;
		Kernel.exit();

		return entryPoint;
	}

	/**
	 * @return the entry point of this feature, null if the {@link #initEntryPoint(FeatureEntryPoint)} method has not been called with success.
	 */
	@Nullable
	/*default*/ FeatureEntryPoint getEntryPoint() {
		return this.entryPoint;
	}

	/**
	 * Notifies {@link FeatureStateListener} that the state of the feature has been changed.
	 */
	/* default */ void notifyStateChanged(@Nullable State previousState) {
		FeatureStateListener[] listeners = Kernel.Listeners; // take a snapshot (don't need synchronization)
		for(FeatureStateListener listener : listeners){
			try{ listener.stateChanged(this, previousState); }
			catch(Throwable e){
				e.printStackTrace(); // TODO log
			}
		}
	}

	/**
	 * Notifies {@link ResourceControlListener} that this feature has been stopped by the Resource Control Manager.
	 */
	private void notifyStoppedByResourceControlManager() {
		StopCause notifiedStopCause;
		switch(this.stopCause){
		case STOP_CAUSE_MEMORY:
			notifiedStopCause = StopCause.MEMORY;
			break;
		case STOP_CAUSE_WATCHDOG:
			notifiedStopCause = StopCause.WATCHDOG;
			break;
		default:
			// stop cause not managed: just ignore
			return;
		}

		ResourceControlListener[] listeners = Kernel.ResourceControlListeners; // take a snapshot (don't need synchronization)
		for(ResourceControlListener listener : listeners){
			try{
				listener.featureStopped(this, notifiedStopCause);
			}
			catch(Throwable e){
				e.printStackTrace(); // TODO log
			}
		}
	}

	/**
	 * @return the loader of this feature
	 */
	public IFeatureLoader getLoader() {
		return loader;
	}

	/**
	 * Take a snapshot of the Feature state (no synchronization)
	 */
	public State getState(){
		int nativeState = KernelNatives.getState(this);
		return getJavaStateFromNativeState(nativeState);
	}

	public void stop() {

		checkNotInFeatureFinalizerThread();

		synchronized(this){

			int nativeState = KernelNatives.getState(this);
			if(nativeState == NATIVE_STATE_INSTALLED || nativeState == NATIVE_STATE_UNINSTALLED) {
				throw new IllegalStateException(getJavaStateFromNativeState(nativeState).toString());
			}

			if(nativeState == NATIVE_STATE_STARTED){

				callStopHook();
				//Here the stop hook is done or a timeout occurs

				closeResources();

				KernelNatives.kill(this);

				// Allow the Feature to run (only throwDeadFeatureException() code)
				setExecutionQuota(QUOTA_DISABLED);

			}

			// Here we are in STOPPED or INSTALLED state
			tryToMoveFromStoppedToInstalled();
		}

	}

	/**
	 * Throws an {@link IllegalStateException} if the current thread is the FeatureFinalizer thread.<p>
	 * This method must be called before any synchronize on this feature to prevent deadlocks.
	 */
	private void checkNotInFeatureFinalizerThread() {
		// If we are in the FeatureFinalizer thread, we don't wait because
		// it is the FeatureFinalizer thread that will unlock the wait.
		// Otherwise we will end up in a dead lock.
		if(Thread.currentThread() == FeatureFinalizer.FeatureFinalizerThread){
			// This is not allowed!
			State state;
			if(stopStep == STOP_STEP_STOPPING){
				// If we are in the STOPPED step but still stopping,
				// then throw an error telling we are in the STARTED state.
				state = State.STARTED;
			}
			else if(stopStep == STOP_STEP_REINSTALLING){
				// If we are in the INSTALLED step but still reinstalling,
				// then throw an error telling we are in the STOPPED state.
				state = State.STOPPED;
			}
			else {
				// By default throw an error with the current state.
				state = this.getState();
			}
			throw new IllegalStateException(state.toString());
		}
	}

	private void callStopHook(){
		AsyncJob job = new AsyncJob() {
			@Override
			public void run() {
				try{
					FeatureEntryPoint fep = getEntryPoint();
					if(fep != null){
						fep.stop();
					}
				}
				catch(Throwable t){
					// The Feature may have been kill while executing this method,
					// so calling stop on the FeatureEntryPoint may throw a DeadFeatureException.
					// Furthermore, the FeatureEntryPoint.stop() implementation may throw any kind
					// of exceptions.
					KernelSupport.uncaughtException(Feature.this, t);
				}
				this.done();
			}
		};

		boolean result = job.start(this);
		if(result){
			job.waitJobEnd(Kernel.STOP_TIMEOUT);
		}
		else{
			// internal limits - not enough threads to execute Feature.stop()
			// => silently skip
		}
	}

	@SuppressWarnings("unchecked")
	private void closeResources() {
		ArrayList<Resource> resources = this.openResources;
		if(resources != null){
			try{
				synchronized (resources) {
					// avoid concurrent modification because reclaim can call directly resourceReclaimed()
					// => take a copy
					resources = (ArrayList<Resource>)resources.clone();
				}
				// Close resources
				for(Resource r : resources){
					try{ r.reclaim(); }
					catch(Throwable e){
						e.printStackTrace();
					}
				}
			}
			catch(Throwable t){
				// clone the resources object may throw a DeadFeatureException
				// if the feature dies while we are cloning it.
			}

		}
	}

	public void start() {
		checkNotInFeatureFinalizerThread();
		synchronized (this) {
			checkExitInstalledStateConditions();
			Kernel.start(this);
			notifyStateChanged(State.INSTALLED);
		}

	}

	/**
	 * Returns true if the feature has been uninstalled, false otherwise.
	 * @see Kernel#uninstall(Feature)
	 */
	/*default*/ boolean uninstall() {

		checkNotInFeatureFinalizerThread();

		synchronized (this) {

			checkExitInstalledStateConditions();

			closeAllJavaResources();

			IFeatureLoader loader = getLoader();
			boolean unloaded;
			synchronized (loader){// see IFeatureLoader comment
				unloaded = loader.unload(this);
			}
			if(unloaded){
				Kernel.removeFromLoadedFeatures(this);
				notifyStateChanged(State.INSTALLED);
			}
			return unloaded;
		}
	}

	public boolean canUninstall() {
		IFeatureLoader loader = getLoader();
		return loader.canUnload();
	}

	public boolean unload() throws UnknownFeatureException{

		checkNotInFeatureFinalizerThread();

		synchronized (this) {

			int nativeState = KernelNatives.getState(this);
			if(nativeState == NATIVE_STATE_UNINSTALLED){
				throw new IllegalStateException(State.UNINSTALLED.toString());
			}

			try{
				stop();
			}
			catch(IllegalStateException e){
				// Ignore exceptions related to a wrong state (installed state or uninstalled state)
				// The right check has been done previously.
			}

			try{
				return uninstall();
			}
			catch(IllegalStateException e){
				return false;
			}
		}
	}

	/**
	 * Checks if this feature can exit the {@link State#INSTALLED} state to move into
	 * the {@link State#STARTED} or {@link State#UNINSTALLED} state.<p>
	 * If this feature is in {@link State#STOPPED} state, then we try to move it into the
	 * {@link State#INSTALLED} state first.<p>
	 * This methods waits also for the execution of any pending {@link FeatureStateListener}s.
	 *
	 * @throws IllegalStateException if the conditions are not satisfied for this transition.
	 */
	/*default*/ void checkExitInstalledStateConditions(){
		int nativeState = KernelNatives.getState(this);
		if(nativeState == NATIVE_STATE_STOPPED){
			// The feature has been stopped but has not moved into the installed state: try to move it into the installed state
			nativeState = tryToMoveFromStoppedToInstalled();
		}
		if(nativeState != NATIVE_STATE_INSTALLED){
			throw new IllegalStateException(Feature.getJavaStateFromNativeState(nativeState).toString());
		}
	}

	/**
	 * Tries to move this feature from {@link State#STOPPED} state to {@link State#INSTALLED} state.<p>
	 * This method assumes this feature is in {@link State#STOPPED} state.
	 *
	 * @return the new state of this feature ({@link #NATIVE_STATE_STOPPED} or {@value #NATIVE_STATE_INSTALLED}).
	 */
	private int tryToMoveFromStoppedToInstalled(){
		// Wait for the feature state listener to be called
		waitForStopStep(STOP_STEP_STOPPED);
		// Try to move the feature from stopped to installed step by calling a GC
		System.gc();

		// Here the feature may have been moved into the installed state: get the current native state
		int nativeState = KernelNatives.getState(this);

		if(nativeState == NATIVE_STATE_INSTALLED){
			// The feature has moved to installed state: wait for the FeatureStateListener to be called.
			waitForStopStep(STOP_STEP_INSTALLED);
		}

		return nativeState;
	}

	public Thread[] getAllAliveThreads(){

		State state = this.getState();
		if(state != State.STARTED){
			throw new IllegalStateException(state.toString());
		}

		int nbThreads = Thread.activeCount();
		Thread[] threads = new Thread[nbThreads]; // created in the context of the caller
		ArrayList<Thread> featureThreads = new ArrayList<Thread>(); // created in the context of the caller
		Kernel.enter();
		try{
			nbThreads = Thread.enumerate(threads);
			for(int i=nbThreads; --i>=0;){
				Thread t = threads[i];
				assert (t != null);
				if(Kernel.getOwner(t) == this) {
					featureThreads.add(t);
				}
			}
		}
		finally{
			Kernel.exit();
		}
		Thread[] featureThreadsArray = featureThreads.toArray(new Thread[featureThreads.size()]);
		assert (featureThreadsArray != null);
		return featureThreadsArray;
	}

	/**
	 * This method must be called under Kernel context.
	 * @param o an object that is not owned by this Feature and which is not a {@link Proxy}.
	 * @param targetInterface must be owned by this Feature. Null for an anonymous {@link Proxy}.
	 * @see Kernel#bind(Object, Class)
	 */
	public <T> T newProxy(Object o, @Nullable Class<T> targetInterface) {
		synchronized (proxyMonitor) {
			return newProxy0(o, targetInterface);
		}
	}

	/**
	 * Caller must ensure this method is thread safe
	 * @see #newProxy(Object, Class)
	 */
	@SuppressWarnings("unchecked")
	private <T> T newProxy0(Object o, @Nullable Class<T> targetInterface) {
		// - Initialize array of proxies
		final int length;
		ProxyRef[] proxies = this.proxies;
		if(proxies == null){
			// lazy init
			length = Integer.getInteger(new StringBuilder(Kernel.KF_PROPERTY_PREFIX).append("proxy.nbKeys").toString(), DEFAULT_PROXY_NB_KEYS);
			// allocate the array of Proxies under Feature context
			Kernel.runUnderContext(this, new Runnable() {

				@Override
				public void run() {
					ProxyRef[] p = new ProxyRef[length];
					Kernel.enter();
					Feature.this.proxies = p;
				}
			});
			proxies = this.proxies;
			assert (proxies != null);
		}
		else{
			length = proxies.length;
		}
		int key = Math.abs(System.identityHashCode(o) % length);

		// - Find potential existing proxy + "incremental" cleanup of the chained list during the search.
		// ProxyRefs with null references are removed
		//	=> No more need to keep identity because the current Feature does not refer to the Proxy anymore. (a new Proxy will be allocated if the target object is retrieved again later)
		// Proxies with null references are removed
		// 	=> No more need to keep identity because the targetted object does not exits anymore
		ProxyRef currentProxyRef = proxies[key];
		ProxyRef previousProxyRef = null;
		while(currentProxyRef != null){
			Proxy<?> currentProxy = currentProxyRef.ref;
			Object target;
			ProxyRef nextProxyRef = currentProxyRef.next;
			if(currentProxy == null || (target=currentProxy.ref) == null){
				// cleanup
				if(previousProxyRef == null){
					proxies[key] = nextProxyRef;
				}
				else{
					previousProxyRef.next = nextProxyRef;
				}
			}
			else{
				if(target == o){
					if(targetInterface == null || targetInterface.isInstance(currentProxy)){
						return (T)currentProxy;
					}
					else{
						// [SPEC] No unicity for the same object across multiple interfaces (no interface aggregate)
						// This proxy does not implement the expected interface.
						// Object may have been registered multiple times with different interfaces => continue
					}
				}
				previousProxyRef = currentProxyRef;
			}
			currentProxyRef = nextProxyRef;
		}

		// - Here, this is the first time this object is encountered
		// => Create a new Proxy and store it
		Proxy<?> proxy;
		if(targetInterface == null){
			proxy = KernelNatives.newAnonymousProxy(o, this);
		}
		else{
			proxy = KernelNatives.bind(o, targetInterface);
		}
		ProxyRef proxyRef = new ProxyRef(proxy);
		Kernel.setOwner(proxyRef, this);

		ProxyRef previousHead = proxies[key];
		assert (previousHead != null); // Eclipse false positive
		proxyRef.next = previousHead;
		proxies[key] = proxyRef;

		return (T)proxy;
	}

	public static State getJavaStateFromNativeState(int nativeState){
		switch(nativeState){
		case NATIVE_STATE_INSTALLED :
			return State.INSTALLED;
		case NATIVE_STATE_STARTED :
			return State.STARTED;
		case NATIVE_STATE_STOPPED :
			return State.STOPPED;
		case NATIVE_STATE_UNINSTALLED :
			return State.UNINSTALLED;
		}
		throw new InternalError("Invalid native state: "+nativeState);
	}

	/**
	 * Returns the criticality of this feature.<p>
	 * The criticality is used by the system to determine which feature to kill when there is memory pressure. The features with the
	 * lowest criticality will be killed first. The features with a criticality set to {@link #MAX_CRITICALITY} are not
	 * killed automatically by the system when there is memory pressure.<p>
	 * The default criticality of a feature is {@link #NORM_CRITICALITY}.
	 *
	 * @return the criticality.
	 *
	 * @throws IllegalStateException if this feature is in {@link State#UNINSTALLED} state.
	 */
	public int getCriticality(){
		return KernelNatives.getCriticality(this);
	}

	/**
	 * Sets the criticality of this feature.<p>
	 *
	 * @param criticality the new criticality of this feature.
	 *
	 * @throws IllegalArgumentException If the criticality is not in the range {@link #MIN_CRITICALITY} to {@link #MAX_CRITICALITY}.
	 *
	 * @see #getCriticality()
	 */
	public void setCriticality(int criticality){
		if(criticality > MAX_CRITICALITY || criticality < MIN_CRITICALITY){
			throw new IllegalArgumentException(Integer.toString(criticality));
		}
		KernelNatives.setCriticality(this, criticality);
	}

	/**
	 * Wait for the feature to reach the given stop step.
	 * @param waitedStopStep stop state we want to wait for (one of the <p>STOP_STEP_*</p> value)
	 */
	public void waitForStopStep(int waitedStopStep) {
		boolean interrupted = false;
		synchronized (featureStateListenerMonitor) {

			// While the waited stop step has not been reached.
			while(this.stopStep < waitedStopStep){
				//Final check before entering in a potential endless wait if this method is called
				//from within the FeatureFinalizer thread.
				checkNotInFeatureFinalizerThread();
				try {
					featureStateListenerMonitor.wait();
				} catch (InterruptedException e) {
					interrupted = true;
				}
			}

		}
		// Set again interrupt flag if this thread has been interrupted.
		if(interrupted){
			Thread.currentThread().interrupt();
		}

	}


	/**
	 * Depending on the current stop step, this method calls the {@link FeatureStateListener}
	 * and {@link ResourceControlListener} and updates the stop step.
	 */
	public void updateStopStep() {
		int stopStep = this.stopStep;
		if(stopStep == STOP_STEP_STOPPING){
			notifyStoppedByResourceControlManager();
			notifyStateChanged(State.STARTED);

			synchronized (featureStateListenerMonitor) {
				this.stopStep = (byte) STOP_STEP_STOPPED;
				// If we have NOT been stopped explicitly (by calling Feature.stop())
				// try to move into the installed state now (execute a GC that will try
				// to do that).
				// If we have been stopped explicitly (by calling Feature.stop())
				// the method tryToMoveFromStoppedToInstalled() will call the GC.
				if(stopCause != STOP_CAUSE_KERNEL){
					System.gc();
				}
				featureStateListenerMonitor.notifyAll();
			}
		}
		else if(stopStep == STOP_STEP_REINSTALLING){
			notifyStateChanged(State.STOPPED);

			synchronized (featureStateListenerMonitor) {
				this.stopStep = (byte) STOP_STEP_INSTALLED;
				featureStateListenerMonitor.notifyAll();
			}
		}


	}

	public long getSectionDescriptor() {
		return KernelNatives.getSectionDescriptor(this);
	}

	/**
	 * IMPLEMENTATION NOTE: VM knows the number of calls from this API to native method (2 calls)
	 * Changing the number of calls need to be reported to KFNativesPool implementation of com_is2t_kf_KernelNatives_getResourceAsStream
	 * <p>
	 * Keep this method synchronized to keep it thread safe with {@link #uninstall()} and {@link FeatureResourceInputStream#close()}.
	 */
	@Nullable
	public synchronized InputStream getResourceAsStream(String name) {

		int nativeState = KernelNatives.getState(this);
		if(nativeState == NATIVE_STATE_UNINSTALLED){
			throw new IllegalStateException(State.UNINSTALLED.toString());
		}

		// Only absolute path are accepted.
		if(!name.startsWith("/")){
			return null;
		}
		// The native assumes an absolute path that does not start with a '/'
		String nameWithoutSlash = name.substring(1);
		assert nameWithoutSlash != null;
		InputStream inputStream = KernelNatives.getResourceAsStream(this, nameWithoutSlash);
		if(inputStream != null){
			inputStream = new FeatureResourceInputStream(inputStream);
		}
		return inputStream;
	}

	/**
	 * WARNING: this method is not thread safe and should be called while synchronized on the same monitor
	 * as the one used by {@link FeatureResourceInputStream#close()} and {@link FeatureResourceInputStream#FeatureResourceInputStream(InputStream)}.
	 */
	private void closeAllJavaResources(){

		ArrayList<FeatureResourceInputStream> openedJavaResources = this.openJavaResources;

		for(FeatureResourceInputStream javaResource : openedJavaResources){
			if(javaResource != null) {
				try {
					javaResource.closeWithoutNotifyingFeature();
				} catch (IOException e) {
					// Nothing to do
				}
			}
		}

		openedJavaResources.clear();
		openedJavaResources.trimToSize();
	}

	/**
	 * Wrapper of an {@link InputStream} that is automatically registered in the resources
	 * opened with {@link Feature#getResourceAsStream(String)}.
	 * <p>
	 * When this {@link InputStream} is closed, it is removed from the Feature's resources list.
	 */
	class FeatureResourceInputStream extends FilterInputStream {

		protected FeatureResourceInputStream(InputStream in) {
			super(in);
			synchronized (Feature.this) {
				openJavaResources.add(this);
			}
		}

		@Override
		public void close() throws IOException {
			super.close();

			synchronized (Feature.this) {
				openJavaResources.remove(this);
			}
		}

		public void closeWithoutNotifyingFeature() throws IOException {
			super.close();
		}
	}

	public boolean setMemoryLimit(long size) {
		return KernelNatives.setMemoryLimit(this, size);
	}

	public long getMemoryLimit() {
		return KernelNatives.getMemoryLimit(this);
	}

}
