/*
 * Java
 *
 * Copyright 2013-2023 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.kf;

import java.io.InputStream;

import ej.annotation.Nullable;
import ej.kf.DeadFeatureException;
import ej.kf.Feature;
import ej.kf.FeatureStateListener;
import ej.kf.Kernel;
import ej.kf.Module;
import ej.kf.Proxy;



public class KernelNatives {

	/**
	 * Initialize within JVM
	 * @return true on success, false on error
	 */
	public static native boolean initialize();

	/**
	 * Gets module data of a Feature or Kernel {@link #KERNEL_DESCRIPTOR}.<p>
	 * WARNING: the object returned by this native is owned by the given module.
	 *
	 * @param featureSectionDescriptor the descriptor of the feature in the VM or {@link KernelNativesConstants#KERNEL_DESCRIPTOR} for the kernel (assumed to be valid (no checks))
	 */
	public static native ModuleData getData(long featureSectionDescriptor);

	/**
	 * Returns null if the feature is not installed.
	 * @param feature native assumes it is not null (no check)
	 */
	public static native Class<?> getEntryPointClass(Feature feature);

	/**
	 *
	 * Access rules are checked inside the VM. CheckPerform
	 * @param o must be non null (no checks)
	 * @param feature the feature that will own the object or null if the owner is the kernel
	 * @throws IllegalAccessError if owner cannot be changed without breaking access rules.
	 * @throws IllegalStateException if the feature is not in started state
	 */
	public static native void setOwner(Object o, @Nullable Feature feature);

	/**
	 * Get the runtime descriptor of the owner of the given Object.
	 * @param o must not be null
	 */
	public static native Module getOwner(Object o);
	/**
	 * Get the runtime descriptor of the owner of the current context
	 */
	public static native Module getContextOwner();


	public static native void enter();
	public static native void exit();

	/**
	 * @param feature native assumes it is not null (no check)
	 */
	public static native void execClinitWrapper(Feature feature);

	/**
	 * Link the given feature with the feature descriptor in the VM.<p>
	 * This method set the {@link Feature#_featureSectionDescriptor} field.
	 * @param feature the feature to link
	 * @param featureSectionDescriptor the descriptor of the feature in the VM.
	 */
	public static native void linkToNative(Feature feature, long featureSectionDescriptor);

	/**
	 * Check if Kernel can load the given signature and endianness.
	 *
	 * @param expectedKernelSignature
	 *            the Kernel signature of the Feature to load
	 * @param expectedKernelEndianness
	 *            the endianness of the Feature to load
	 *            (<code>true<code> for little-endian, <code>false<code> for big-endian)
	 * @return <code>true<code> if the currently running Kernel can load the Feature, <code>false</code>
	 *         otherwise
	 */
	public static native boolean checkKernel(byte[] expectedKernelSignature, boolean expectedKernelEndianness);

	/**
	 * Kill the given feature.<p>
	 * The given feature may already been stopped.
	 * @param feature native assumes it is not null (no check)
	 */
	public static native void kill(Feature feature);

	/**
	 * Access rules are checked inside the VM. CheckPerform
	 * @param o must be non null (no checks)
	 * @param feature the feature that will own the object or null if the owner is the kernel
	 * @throws IllegalAccessError if object cannot be allocated without breaking access rules.
	 * @throws IllegalStateException if the feature is not in started state
	 */
	public static native Object clone(Object from, @Nullable Feature feature);

	/**
	 * Allocate a new {@link Proxy} that implements the given target interface and references the given object.
	 * @param o an object that is not owned by the targetInterface owner Feature
	 * @param targetInterface an interface owned by a Feature
	 * @return a new {@link Proxy} to the given object owned by the targetInterface owner
	 * @throws {@link IllegalAccessError} if the given object cannot be bounded to the given interface
	 * @throws {@link IllegalArgumentException} if the given type is not a shared interface
	 */
	public static native Proxy<?> bind(Object o, Class<?> targetInterface);

	/**
	 * Allocate a new anonymous {@link Proxy} to the given {@link Object}.
	 * @param o an object that is not owned by the target owner
	 * @param feature the feature that will own the object or null if the owner is the kernel
	 * @throws IllegalStateException if the feature is not in started state
	 */
	public static native Proxy<?> newAnonymousProxy(Object o, @Nullable Feature feature);

	/**
	 * @return true if both classes are equivalent shared interfaces, false otherwise
	 */
	public static native boolean areEquivalentSharedInterfaces(Class<?> targetRef, Class<?> otherTargetRef);

	/**
	 * @param c may be null
	 *
	 * @return true if the class is a shared interface, false otherwise or if c is null.
	 */
	public static native boolean isSharedInterface(Class<?> c);

	/**
	 * @return true if the class is a Kernel API, false otherwise
	 */
	public static native boolean isAPI(Class<?> c);

	/**
	 * Get the deepest interface implemented by c under the hierarchy of <code>topInterface</code>. May return <code>topInterface</code> if it is the deepest shared interface implemented by c.
	 * @param c a Feature class or interface that implements <code>topInterface</code>
	 * @param rootInterface an interface accessible by the owner of c
	 * @return shared interface type as described or null if no shared interface found
	 */
	@Nullable
	public static native Class<?> getImplementedSharedInterface(Class<?> c, Class<?> topInterface);

	/**
	 * From a shared interface, get the closest target shared interface in the target Feature.
	 * @param si a shared interface that extends <code>topInterface</code>
	 * @param topInterface a shared interface owned by the owner of <code>si</code> or the Kernel
	 * @param targetFeature the target Feature where to find the closest equivalent shared interface (which is not the owner of <code>si</code>). May be null.
	 * @return the shared interface as described, null if not found.
	 */
	@Nullable
	public static native Class<?> getTargetSharedInterface(Class<?> si, Class<?> topInterface, Feature targetFeature);

	/**
	 * Notify the VM that the given feature has been started.
	 * @param feature native assumes it is not null (no check)
	 */
	public static native void setStarted(Feature feature);

	/**
	 * Returns the state of the feature identified by <code>featureVmDescriptor</code>.
	 * @param feature native assumes it is not null (no check)
	 * @return the current state of the feature
	 */
	public static native int getState(Feature feature);

	/**
	 * @param feature native assumes it is not null (no check)
	 *
	 * @see Feature#getCriticality()
	 */
	public static native int getCriticality(Feature feature);

	/**
	 * @param feature native assumes it is not null (no check)
	 * @param criticality native assumes it is between MIN_CRITICALITY and MAX_CRITICALITY (no check)
	 *
	 * @see Feature#setCriticality(int)
	 */
	public static native void setCriticality(Feature feature, int criticality);

	/**
	 * Waits for a stopped feature that needs to call the {@link FeatureStateListener}.
	 * <p>
	 * If there is no available feature, waits for a Feature to finalize and returns null after waiting.
	 * Call this method again to get the feature.
	 * @return a feature on which we must call {@link Feature#updateStopStep()} or null
	 */
	@Nullable
	public static native Feature getNextStoppedFeatureToUpdate();


	/**
	 * Referenced by the VM when a feature is killed.
	 * <p>
	 * Throws a {@link DeadFeatureException}.
	 */
	public static native void throwDeadFeatureException();

	/**
	 * @return the Feature section descriptor
	 */
	public static native long getSectionDescriptor(Feature feature);

	/**
	 * @return the heap memory size allocated by the given module.
	 */
	public static native long getAllocatedMemory(Module module);

	/**
	 * See {@link Feature#getResourceAsStream(String)}
	 */
	@Nullable
	public static native InputStream getResourceAsStream(Feature feature, String name);

	/**
	 * See {@link Feature#setMemoryLimit(long)} API.
	 */
	public static native boolean setMemoryLimit(Feature feature, long size);

	/**
	 * See {@link Feature#getMemoryLimit()} API.
	 */
	public static native long getMemoryLimit(Feature feature);

	/**
	 * See {@link Kernel#setReservedMemory(long)} API.
	 */
	public static native boolean setReservedMemory(long size);

	/**
	 * See {@link Kernel#getReservedMemory()} API.
	 */
	public static native long getReservedMemory();

}
