/*
 * Java
 *
 * Copyright 2015-2023 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.kf;

import java.util.ArrayList;

import ej.bon.Constants;
import ej.kf.Feature;
import ej.kf.Kernel;
import ej.lang.Resource;
import ej.lang.ResourceManager;

/**
 * {@link KFResourceManager} is responsible for registering/unregistering created resource on Features.
 * These resources are closed before killing the Feature.
 */
public class KFResourceManager extends ResourceManager {

	@Override
	public void resourceCreated(Resource resource) {
		// ResourceManager is deprecated and should not be used when KF RAM Control is enabled.
		// SNI native resources must be used as a replacement.
		// Reason why: when KF RAM Control is enabled, the Core Engine can stop a Feature abruptly
		// without triggering calls to Resource.reclaim(). This could lead to memory or resource leaks.
		if(Constants.getBoolean("com.microej.runtime.kf.ramcontrol.enabled")) {
			throw new AssertionError(
					"Legacy Resource Manager is deprecated and disabled when RAM Control is enabled. Please update the Foundation Library implementation that created the following resource: "
							+ resource);
		}
		else {
			super.resourceCreated(resource);
			Feature f = Kernel.getContextOwnerFeature();
			if(f != null){
				ArrayList<Resource> resources = f.openResources;
				assert (resources != null);
				synchronized (resources) {
					resources.add(resource);
				}
			}
		}
	}

	@Override
	public void resourceReclaimed(Resource resource) {
		super.resourceReclaimed(resource);
		Feature f = Kernel.getContextOwnerFeature();
		if(f != null){
			ArrayList<Resource> resources = f.openResources;
			assert (resources != null);
			synchronized (resources) {
				resources.remove(resource);
			}
		}
	}
}
