/*
 * Java
 *
 * Copyright 2018-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.bluetooth;

import ej.annotation.Nullable;
import ej.bluetooth.tools.HexTools;
import ej.bon.ByteArray;
import ej.bon.Immutables;

public class BluetoothUuid {

	/**
	 * Not in API.
	 */
	public static final int LENGTH = 16;

	private static final int STRING_LENGTH = 36; // xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx

	private static final String BASE_UUID_IMMUTABLE = "ej.bluetooth.BASE_UUID";
	private static final byte[] BASE_UUID = (byte[]) Immutables.get(BASE_UUID_IMMUTABLE);

	private final byte[] bytes;

	public BluetoothUuid(byte[] bytes, int offset) {
		this.bytes = new byte[LENGTH];
		System.arraycopy(bytes, offset, this.bytes, 0, LENGTH);
	}

	public BluetoothUuid(String string) {
		if (string.length() != STRING_LENGTH) {
			throw new IllegalArgumentException();
		}

		this.bytes = new byte[LENGTH];

		int stringIndex = 0;
		for (int i = 0; i < LENGTH; i++) {
			this.bytes[i] = HexTools.stringToByte(string, stringIndex);
			stringIndex += 2;

			if (i == 3 || i == 5 || i == 7 || i == 9) {
				if (string.charAt(stringIndex) != '-') {
					throw new IllegalArgumentException();
				}
				stringIndex++;
			}
		}
	}

	public BluetoothUuid(int value) {
		this.bytes = new byte[LENGTH];
		System.arraycopy(BASE_UUID, 0, this.bytes, 0, LENGTH);
		ByteArray.writeShort(this.bytes, 2, value, ByteArray.BIG_ENDIAN);
	}

	/**
	 * Not in API.
	 */
	public BluetoothUuid(BluetoothUuid uuid) {
		this(uuid.bytes, 0);
	}

	public void getBytes(byte[] buffer, int offset) {
		System.arraycopy(this.bytes, 0, buffer, offset, LENGTH);
	}

	public boolean is16Bit() {
		for (int i = 0; i < LENGTH; i++) {
			// skip i = 2 and i = 3
			if (i != 2 && i != 3) {
				if (this.bytes[i] != BASE_UUID[i]) {
					return false;
				}
			}
		}
		return true;
	}

	public short get16BitValue() {
		if (!is16Bit()) {
			throw new IllegalStateException();
		}
		return ByteArray.readShort(this.bytes, 2, ByteArray.BIG_ENDIAN);
	}

	@Override
	public String toString() {
		char[] charArray = new char[STRING_LENGTH];
		int stringIndex = 0;
		for (int i = 0; i < LENGTH; i++) {
			HexTools.byteToString(this.bytes[i], charArray, stringIndex);
			stringIndex += 2;

			if (i == 3 || i == 5 || i == 7 || i == 9) {
				charArray[stringIndex] = '-';
				stringIndex++;
			}
		}
		return new String(charArray);
	}

	@Override
	public boolean equals(@Nullable Object object) {
		if (object == this) {
			return true;
		}

		if (!(object instanceof BluetoothUuid)) {
			return false;
		}

		BluetoothUuid uuid = (BluetoothUuid) object;
		for (int i = 0; i < LENGTH; i++) {
			if (uuid.bytes[i] != this.bytes[i]) {
				return false;
			}
		}

		return true;
	}

	@Override
	public int hashCode() {
		int result = 1;
		for (byte element : this.bytes) {
			result = 31 * result + element;
		}
		return result;
	}
}
