/*
 * Java
 *
 * Copyright 2018-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.bluetooth;

import ej.bluetooth.tools.ArrayTools;
import ej.kf.Kernel;

public class BluetoothServiceDefinition {

	private static class BluetoothAttributeDefinition {

		public final byte type;
		public final BluetoothUuid uuid;
		public final byte permissions;
		public final byte properties;

		public BluetoothAttributeDefinition(byte type, BluetoothUuid uuid, byte permissions, byte properties) {
			this.type = type;
			this.uuid = uuid;
			this.permissions = permissions;
			this.properties = properties;
		}
	}

	private static final byte ATTRIBUTE_TYPE_CHARACTERISTIC = 0;
	private static final byte ATTRIBUTE_TYPE_DESCRIPTOR = 1;

	private static final int SERVICE_HEADER_SIZE = 18;
	private static final int ATTRIBUTE_SIZE = 20;

	private final BluetoothUuid serviceUuid;
	private BluetoothAttributeDefinition[] attributes;

	public BluetoothServiceDefinition(BluetoothUuid serviceUuid) {
		this.serviceUuid = serviceUuid;
		this.attributes = new BluetoothAttributeDefinition[0];
	}

	public void addCharacteristic(BluetoothUuid uuid, int properties, int permissions) {
		addAttribute(ATTRIBUTE_TYPE_CHARACTERISTIC, uuid, (byte) permissions, (byte) properties);
	}

	public void addDescriptor(BluetoothUuid uuid, int permissions) {
		addAttribute(ATTRIBUTE_TYPE_DESCRIPTOR, uuid, (byte) permissions, (byte) 0);
	}

	private void addAttribute(byte type, BluetoothUuid uuid, byte permissions, byte properties) {
		BluetoothAttributeDefinition attribute = new BluetoothAttributeDefinition(type, uuid, permissions, properties);
		this.attributes = ArrayTools.append(this.attributes, attribute);
	}

	/**
	 * Not in API.
	 */
	public int getNumAttributes() {
		return this.attributes.length;
	}

	/**
	 * Not in API.
	 */
	public int getSerializedSize() {
		return SERVICE_HEADER_SIZE + this.attributes.length * ATTRIBUTE_SIZE;
	}

	/**
	 * Not in API.
	 */
	public static void serialize(BluetoothServiceDefinition definition, byte[] buffer, int offset) {
		int numCharacteristics = 0;
		int numDescriptors = 0;

		int attrOffset = offset + SERVICE_HEADER_SIZE;
		for (BluetoothAttributeDefinition attribute : definition.attributes) {
			// write attribute data
			buffer[attrOffset] = attribute.type;
			attribute.uuid.getBytes(buffer, attrOffset + 2);
			buffer[attrOffset + 18] = attribute.permissions;
			buffer[attrOffset + 19] = attribute.properties;
			attrOffset += ATTRIBUTE_SIZE;

			// update attributes count
			if (attribute.type == ATTRIBUTE_TYPE_CHARACTERISTIC) {
				numCharacteristics++;
			} else if (attribute.type == ATTRIBUTE_TYPE_DESCRIPTOR) {
				numDescriptors++;
			}
		}

		// write service data
		definition.serviceUuid.getBytes(buffer, offset);
		buffer[offset + 16] = (byte) numCharacteristics;
		buffer[offset + 17] = (byte) numDescriptors;
	}

	/**
	 * Not in API.
	 */
	public static BluetoothService createService(BluetoothServiceDefinition definition, short[] handles) {
		// read service handle
		int handleIndex = 0;
		short serviceHandle = handles[handleIndex++];

		// create service
		BluetoothService service = new BluetoothService(cloneUuid(definition.serviceUuid), serviceHandle, true);

		BluetoothCharacteristic lastCharacteristic = null;
		for (BluetoothAttributeDefinition attribute : definition.attributes) {
			// read attribute handle
			short attrHandle = handles[handleIndex++];

			if (attribute.type == ATTRIBUTE_TYPE_CHARACTERISTIC) {
				// create characteristic and add it to service
				lastCharacteristic = new BluetoothCharacteristic(cloneUuid(attribute.uuid), attribute.properties,
						attrHandle, service);
				service.addCharacteristic(lastCharacteristic);
			} else if (attribute.type == ATTRIBUTE_TYPE_DESCRIPTOR) {
				if (lastCharacteristic != null) {
					// create descriptor and add it to last characteristic
					BluetoothDescriptor descriptor = new BluetoothDescriptor(cloneUuid(attribute.uuid), attrHandle,
							lastCharacteristic);
					lastCharacteristic.addDescriptor(descriptor);
				}
			}
		}

		return service;
	}

	private static BluetoothUuid cloneUuid(BluetoothUuid uuid) {
		// only create a new object if the original one is not owned by the kernel
		if (Kernel.getOwner(uuid) == Kernel.getInstance()) {
			return uuid;
		} else {
			return new BluetoothUuid(uuid);
		}
	}
}
