/*
 * Java
 *
 * Copyright 2018-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.bluetooth;

import ej.bluetooth.listeners.LocalServiceListener;
import ej.bluetooth.listeners.RemoteServiceListener;
import ej.bluetooth.listeners.impl.DefaultLocalServiceListener;
import ej.bluetooth.listeners.impl.DefaultRemoteServiceListener;
import ej.bluetooth.tools.ArrayTools;
import ej.bon.IllegalStateException;
import ej.kf.Feature;
import ej.kf.Kernel;

public class BluetoothService {

	private static final RemoteServiceListener DEFAULT_REMOTE_LISTENER = new DefaultRemoteServiceListener();
	private static final LocalServiceListener DEFAULT_LOCAL_LISTENER = new DefaultLocalServiceListener();

	private final BluetoothUuid uuid;
	private final short handle;
	private final boolean isLocal;

	private BluetoothCharacteristic[] characteristics;
	private RemoteServiceListener remoteListener;
	private LocalServiceListener localListener;

	/**
	 * Not in API.
	 */
	public BluetoothService(BluetoothUuid uuid, short handle, boolean isLocal) {
		this.uuid = uuid;
		this.handle = handle;
		this.isLocal = isLocal;

		this.characteristics = new BluetoothCharacteristic[0];
		this.remoteListener = DEFAULT_REMOTE_LISTENER;
		this.localListener = DEFAULT_LOCAL_LISTENER;
	}

	/**
	 * Not in API.
	 */
	public void addCharacteristic(BluetoothCharacteristic characteristic) {
		this.characteristics = ArrayTools.append(this.characteristics, characteristic);
	}

	public BluetoothUuid getUuid() {
		return this.uuid;
	}

	/**
	 * Not in API.
	 */
	public short getHandle() {
		return this.handle;
	}

	/**
	 * Not in API.
	 */
	public boolean isLocal() {
		return this.isLocal;
	}

	public int getNumCharacteristics() {
		return this.characteristics.length;
	}

	public BluetoothCharacteristic getCharacteristic(int index) {
		BluetoothCharacteristic characteristic = this.characteristics[index];
		assert (characteristic != null);
		return characteristic;
	}

	/**
	 * Not in API.
	 */
	public BluetoothCharacteristic[] getCharacteristics() {
		return this.characteristics; // NOSONAR - don't get a copy
	}

	/**
	 * Not in API.
	 */
	public RemoteServiceListener getRemoteListener() {
		return this.remoteListener;
	}

	/**
	 * Not in API.
	 */
	public LocalServiceListener getLocalListener() {
		return this.localListener;
	}

	public void setRemoteListener(RemoteServiceListener remoteListener) {
		if (this.isLocal) {
			throw new IllegalStateException();
		}
		Kernel.enter();
		this.remoteListener = remoteListener;
	}

	public void setLocalListener(LocalServiceListener localListener) {
		if (!this.isLocal) {
			throw new IllegalStateException();
		}
		Kernel.enter();
		this.localListener = localListener;
	}

	/**
	 * Not in API.
	 */
	public void cleanupFeatureListeners(Feature feature) {
		if (this.isLocal) {
			if (Kernel.getOwner(this.localListener) == feature) {
				this.localListener = DEFAULT_LOCAL_LISTENER;
			}
		} else {
			if (Kernel.getOwner(this.remoteListener) == feature) {
				this.remoteListener = DEFAULT_REMOTE_LISTENER;
			}
		}
	}
}
