/*
 * Java
 *
 * Copyright 2018-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.bluetooth;

import ej.annotation.Nullable;
import ej.bluetooth.tools.ArrayTools;
import ej.bon.ByteArray;

/**
 * Not in API.
 */
public class BluetoothPump {

	private static final byte ATTRIBUTE_TYPE_CHARACTERISTIC = 0;
	private static final byte ATTRIBUTE_TYPE_DESCRIPTOR = 1;

	private static final int SERVICE_HEADER_SIZE = 22;
	private static final int ATTRIBUTE_SIZE = 22;

	private static final String BUFFER_SIZE_PROPERTY = "bluetooth.pump.buffer.size";
	private static final int DEFAULT_BUFFER_SIZE = 600;

	private static final String THREAD_NAME = "BluetoothPump";

	private final BluetoothAdapter adapter;

	private @Nullable Thread thread;
	private boolean stopRequested;

	public BluetoothPump(BluetoothAdapter adapter) {
		this.adapter = adapter;
		this.thread = null;
	}

	public void start() {
		this.stopRequested = false;

		if (this.thread == null) {
			this.thread = new Thread(new Runnable() {
				@Override
				public void run() {
					runThread();
				}
			}, THREAD_NAME);

			this.thread.start();
		}
	}

	public void stop() {
		this.stopRequested = true;
	}

	private void runThread() {
		try {
			int bufferSize = getBufferSize();
			byte[] buffer = new byte[bufferSize];

			while (!this.stopRequested) {
				int eventSize = BluetoothNatives.waitEvent(buffer, buffer.length);
				if (eventSize > 0) {
					handleEvent(buffer);
				}
			}
		} finally {
			this.thread = null;
		}
	}

	private void handleEvent(byte[] data) { // NOSONAR - method complexity is acceptable
		int eventType = data[0];
		switch (eventType) {
		case BluetoothEventTypes.GAP_SCAN_RESULT:
			handleScanResult(data);
			break;
		case BluetoothEventTypes.GAP_SCAN_COMPLETED:
			handleScanCompleted();
			break;
		case BluetoothEventTypes.GAP_ADVERTISEMENT_COMPLETED:
			handleAdvertisementCompleted();
			break;
		case BluetoothEventTypes.GAP_CONNECT_FAILED:
			handleConnectFailed(data);
			break;
		case BluetoothEventTypes.GAP_CONNECTED:
			handleConnected(data);
			break;
		case BluetoothEventTypes.GAP_DISCONNECTED:
			handleDisconnected(data);
			break;
		case BluetoothEventTypes.GAP_PAIR_REQUEST:
			handlePairRequest(data);
			break;
		case BluetoothEventTypes.GAP_PAIR_COMPLETED:
			handlePairCompleted(data);
			break;
		case BluetoothEventTypes.GAP_PASSKEY_REQUEST:
			handlePasskeyRequest(data);
			break;
		case BluetoothEventTypes.GAP_PASSKEY_GENERATED:
			handlePasskeyGenerated(data);
			break;
		case BluetoothEventTypes.GATTC_DISCOVERY_RESULT:
			handleDiscoveryResult(data);
			break;
		case BluetoothEventTypes.GATTC_DISCOVERY_COMPLETED:
			handleDiscoveryCompleted(data);
			break;
		case BluetoothEventTypes.GATTC_READ_COMPLETED:
			handleReadCompleted(data);
			break;
		case BluetoothEventTypes.GATTC_WRITE_COMPLETED:
			handleWriteCompleted(data);
			break;
		case BluetoothEventTypes.GATTC_NOTIFICATION_RECEIVED:
			handleNotificationReceived(data);
			break;
		case BluetoothEventTypes.GATTS_READ_REQUEST:
			handleReadRequest(data);
			break;
		case BluetoothEventTypes.GATTS_WRITE_REQUEST:
			handleWriteRequest(data);
			break;
		case BluetoothEventTypes.GATTS_NOTIFICATION_SENT:
			handleNotificationSent(data);
			break;
		case BluetoothEventTypes.GATTS_EXECUTE_WRITE_REQUEST:
			handleExecuteWriteRequest(data);
			break;
		default:
			// unknown event type
		}
	}

	private void handleScanResult(byte[] data) {
		BluetoothAddress deviceAddress = BluetoothAddress.deserialize(data, 1);
		int rssi = data[8];
		int advertisementDataSize = data[9];
		byte[] advertisementData = ArrayTools.readBytes(data, 10, advertisementDataSize);

		this.adapter.handleScanResult(deviceAddress, advertisementData, rssi);
	}

	private void handleScanCompleted() {
		this.adapter.handleScanCompleted();
	}

	private void handleAdvertisementCompleted() {
		this.adapter.handleAdvertisementCompleted();
	}

	private void handleConnectFailed(byte[] data) {
		BluetoothAddress deviceAddress = BluetoothAddress.deserialize(data, 1);

		this.adapter.handleConnectFailed(deviceAddress);
	}

	private void handleConnected(byte[] data) {
		BluetoothAddress deviceAddress = BluetoothAddress.deserialize(data, 1);
		short connHandle = ByteArray.readShort(data, 8);

		this.adapter.handleConnected(connHandle, deviceAddress);
	}

	private void handleDisconnected(byte[] data) {
		short connHandle = ByteArray.readShort(data, 2);

		this.adapter.handleDisconnected(connHandle);
	}

	private void handlePairRequest(byte[] data) {
		short connHandle = ByteArray.readShort(data, 2);

		this.adapter.handlePairRequest(connHandle);
	}

	private void handlePairCompleted(byte[] data) {
		short connHandle = ByteArray.readShort(data, 2);
		boolean success = (data[1] == 1);

		this.adapter.handlePairCompleted(connHandle, success);
	}

	private void handlePasskeyRequest(byte[] data) {
		short connHandle = ByteArray.readShort(data, 2);

		this.adapter.handlePasskeyRequest(connHandle);
	}

	private void handlePasskeyGenerated(byte[] data) {
		short connHandle = ByteArray.readShort(data, 2);
		int passkey = ByteArray.readInt(data, 4);

		this.adapter.handlePasskeyGenerated(connHandle, passkey);
	}

	private void handleDiscoveryResult(byte[] data) {
		// read conn handle
		short connHandle = ByteArray.readShort(data, 2);

		// read service data
		int numAttributes = data[1];
		short serviceHandle = ByteArray.readShort(data, 4);
		BluetoothUuid serviceUuid = new BluetoothUuid(data, 6);

		// create service
		BluetoothService service = new BluetoothService(serviceUuid, serviceHandle, false);

		int attrOffset = SERVICE_HEADER_SIZE;
		BluetoothCharacteristic lastCharacteristic = null;
		for (int i = 0; i < numAttributes; i++) {
			// read attribute data
			byte attrType = data[attrOffset];
			BluetoothUuid attrUuid = new BluetoothUuid(data, attrOffset + 2);
			short attrHandle = ByteArray.readShort(data, attrOffset + 18);

			if (attrType == ATTRIBUTE_TYPE_CHARACTERISTIC) {
				// read characteristic data
				byte properties = data[attrOffset + 20];

				// create characteristic and add it to service
				lastCharacteristic = new BluetoothCharacteristic(attrUuid, properties, attrHandle, service);
				service.addCharacteristic(lastCharacteristic);
			} else if (attrType == ATTRIBUTE_TYPE_DESCRIPTOR) {
				if (lastCharacteristic != null) {
					// create descriptor and add it to last characteristic
					BluetoothDescriptor descriptor = new BluetoothDescriptor(attrUuid, attrHandle, lastCharacteristic);
					lastCharacteristic.addDescriptor(descriptor);
				}
			}

			attrOffset += ATTRIBUTE_SIZE;
		}

		// handle event in adapter
		this.adapter.handleDiscoveryResult(connHandle, service);
	}

	private void handleDiscoveryCompleted(byte[] data) {
		short connHandle = ByteArray.readShort(data, 2);

		this.adapter.handleDiscoveryCompleted(connHandle);
	}

	private void handleReadCompleted(byte[] data) {
		byte status = data[1];
		short connHandle = ByteArray.readShort(data, 2);
		short attributeHandle = ByteArray.readShort(data, 4);
		short valueSize = ByteArray.readShort(data, 6);
		byte[] value = ArrayTools.readBytes(data, 8, valueSize);

		this.adapter.handleReadCompleted(connHandle, attributeHandle, status, value);
	}

	private void handleWriteCompleted(byte[] data) {
		byte status = data[1];
		short connHandle = ByteArray.readShort(data, 2);
		short attributeHandle = ByteArray.readShort(data, 4);

		this.adapter.handleWriteCompleted(connHandle, attributeHandle, status);
	}

	private void handleNotificationReceived(byte[] data) {
		short connHandle = ByteArray.readShort(data, 2);
		short attributeHandle = ByteArray.readShort(data, 4);
		short valueSize = ByteArray.readShort(data, 6);
		byte[] value = ArrayTools.readBytes(data, 8, valueSize);

		this.adapter.handleNotificationReceived(connHandle, attributeHandle, value);
	}

	private void handleReadRequest(byte[] data) {
		short connHandle = ByteArray.readShort(data, 2);
		short attributeHandle = ByteArray.readShort(data, 4);
		short offset = ByteArray.readShort(data, 6);

		this.adapter.handleReadRequest(connHandle, attributeHandle, offset);
	}

	private void handleWriteRequest(byte[] data) {
		boolean prepare = (data[1] == 1);
		short connHandle = ByteArray.readShort(data, 2);
		short attributeHandle = ByteArray.readShort(data, 4);
		short offset = ByteArray.readShort(data, 6);
		short valueSize = ByteArray.readShort(data, 8);
		byte[] value = ArrayTools.readBytes(data, 10, valueSize);

		this.adapter.handleWriteRequest(connHandle, attributeHandle, value, offset, prepare);
	}

	private void handleExecuteWriteRequest(byte[] data) {
		boolean execute = (data[1] == 1);
		short connHandle = ByteArray.readShort(data, 2);
		short attributeHandle = ByteArray.readShort(data, 4);

		this.adapter.handleExecuteWriteRequest(connHandle, attributeHandle, execute);
	}

	private void handleNotificationSent(byte[] data) {
		boolean success = (data[1] == 1);
		short connHandle = ByteArray.readShort(data, 2);
		short attributeHandle = ByteArray.readShort(data, 4);

		this.adapter.handleNotificationSent(connHandle, attributeHandle, success);
	}

	private static int getBufferSize() {
		String value = System.getProperty(BUFFER_SIZE_PROPERTY);
		if (value != null) {
			// throw an exception if the string does not represent an integer
			return Integer.decode(value).intValue();
		} else {
			return DEFAULT_BUFFER_SIZE;
		}
	}
}
