/*
 * Java
 *
 * Copyright 2018-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.bluetooth;

import ej.annotation.Nullable;
import ej.bluetooth.tools.ArrayTools;
import ej.kf.Feature;

/**
 * Not in API.
 */
public class BluetoothDatabase {

	private BluetoothService[] services;
	private BluetoothAttribute[] attributes;

	public BluetoothDatabase() {
		this.services = new BluetoothService[0];
		this.attributes = new BluetoothAttribute[0];
	}

	public void clear() {
		this.services = new BluetoothService[0];
		this.attributes = new BluetoothAttribute[0];
	}

	public void cleanupFeatureListeners(Feature feature) {
		for (BluetoothService service : this.services) {
			service.cleanupFeatureListeners(feature);
		}
	}

	public boolean containsService(BluetoothService service) {
		for (BluetoothService s : this.services) {
			if (s == service) {
				return true;
			}
		}
		return false;
	}

	public @Nullable BluetoothService getService(short handle) {
		for (BluetoothService service : this.services) {
			if (service.getHandle() == handle) {
				return service;
			}
		}
		return null;
	}

	public @Nullable BluetoothAttribute getAttribute(short handle) {
		BluetoothAttribute[] attributes = this.attributes;
		int index = binarySearch(attributes, handle);
		return (index < 0 ? null : attributes[index]);
	}

	public void addService(BluetoothService service) {
		// add service to list
		this.services = ArrayTools.append(this.services, service);

		// add characteristics to attributes list
		for (BluetoothCharacteristic characteristic : service.getCharacteristics()) {
			assert (characteristic != null);
			addAttribute(characteristic);

			// add descriptors to attributes list
			for (BluetoothDescriptor descriptor : characteristic.getDescriptors()) {
				assert (descriptor != null);
				addAttribute(descriptor);
			}
		}
	}

	private void addAttribute(BluetoothAttribute attribute) {
		BluetoothAttribute[] attributes = this.attributes;
		int index = binarySearch(attributes, attribute.getHandle());
		if (index < 0) {
			int insertionIndex = -index - 1;
			this.attributes = ArrayTools.insert(attributes, attribute, insertionIndex);
		}
	}

	private static int binarySearch(BluetoothAttribute[] array, short key) {
		int low = 0;
		int high = array.length - 1;

		while (low <= high) {
			int mid = (low + high) >>> 1;
		int comp = array[mid].getHandle() - key;

		if (comp < 0) {
			low = mid + 1;
		} else if (comp > 0) {
			high = mid - 1;
		} else {
			return mid; // key found
		}
		}

		return -(low + 1); // key not found
	}
}
