/*
 * Java
 *
 * Copyright 2018-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.bluetooth;

import ej.annotation.Nullable;
import ej.kf.Feature;

public class BluetoothConnection {

	private final BluetoothAddress address;
	private final BluetoothDatabase database;
	private final short connHandle;

	/**
	 * Not in API.
	 */
	public BluetoothConnection(BluetoothAddress address, short connHandle) {
		this.address = address;
		this.database = new BluetoothDatabase();
		this.connHandle = connHandle;
	}

	public BluetoothAddress getAddress() {
		return this.address;
	}

	/**
	 * Not in API.
	 */
	public void cleanupFeatureListeners(Feature feature) {
		this.database.cleanupFeatureListeners(feature);
	}

	/**
	 * Not in API.
	 */
	public void addService(BluetoothService service) {
		this.database.addService(service);
	}

	/**
	 * Not in API.
	 */
	public @Nullable BluetoothService getService(short serviceHandle) {
		return this.database.getService(serviceHandle);
	}

	/**
	 * Not in API.
	 */
	public @Nullable BluetoothAttribute getAttribute(short attributeHandle) {
		return this.database.getAttribute(attributeHandle);
	}

	/**
	 * Not in API.
	 */
	public short getHandle() {
		return this.connHandle;
	}

	public boolean disconnect() {
		return BluetoothNatives.disconnect(this.connHandle);
	}

	public boolean sendPairRequest() {
		return BluetoothNatives.sendPairRequest(this.connHandle);
	}

	public boolean sendPairResponse(boolean accept) {
		return BluetoothNatives.sendPairResponse(this.connHandle, accept);
	}

	public boolean sendPasskeyResponse(boolean accept, int passkey) {
		return BluetoothNatives.sendPasskeyResponse(this.connHandle, accept, passkey);
	}

	public boolean discoverServices() {
		return BluetoothNatives.discoverServices(this.connHandle, null);
	}

	public boolean discoverService(BluetoothUuid uuid) {
		byte[] buffer = new byte[BluetoothUuid.LENGTH];
		uuid.getBytes(buffer, 0);
		return BluetoothNatives.discoverServices(this.connHandle, buffer);
	}

	public boolean sendReadRequest(BluetoothAttribute attribute) {
		BluetoothService service = attribute.getService();
		if (service.isLocal() || !this.database.containsService(service)) {
			throw new IllegalArgumentException();
		}

		return BluetoothNatives.sendReadRequest(this.connHandle, attribute.getHandle());
	}

	public boolean sendWriteRequest(BluetoothAttribute attribute, byte[] value) {
		BluetoothService service = attribute.getService();
		if (service.isLocal() || !this.database.containsService(service)) {
			throw new IllegalArgumentException();
		}

		boolean noResponse = attribute.useWriteWithoutResponse();

		return BluetoothNatives.sendWriteRequest(this.connHandle, attribute.getHandle(), value, value.length,
				noResponse);
	}

	public boolean sendReadResponse(BluetoothAttribute attribute, byte status, byte[] value) {
		BluetoothService service = attribute.getService();
		if (!service.isLocal()) {
			throw new IllegalArgumentException();
		}

		return BluetoothNatives.sendReadResponse(this.connHandle, attribute.getHandle(), status, value,
				value.length);
	}

	public boolean sendWriteResponse(BluetoothAttribute attribute, byte status) {
		BluetoothService service = attribute.getService();
		if (!service.isLocal()) {
			throw new IllegalArgumentException();
		}

		return BluetoothNatives.sendWriteResponse(this.connHandle, attribute.getHandle(), status);
	}

	public boolean sendPrepareWriteResponse(BluetoothAttribute attribute, byte status, byte[] value, int offset) {
		BluetoothService service = attribute.getService();
		if (!service.isLocal()) {
			throw new IllegalArgumentException();
		}

		return BluetoothNatives.sendPrepareWriteResponse(this.connHandle, attribute.getHandle(), status, value,
				value.length, offset);
	}

	public boolean sendExecuteWriteResponse(BluetoothAttribute attribute, byte status) {
		BluetoothService service = attribute.getService();
		if (!service.isLocal()) {
			throw new IllegalArgumentException();
		}

		return BluetoothNatives.sendExecuteWriteResponse(this.connHandle, attribute.getHandle(), status);
	}

	public boolean sendNotification(BluetoothCharacteristic characteristic, byte[] value, boolean confirm) {
		BluetoothService service = characteristic.getService();
		if (!service.isLocal()) {
			throw new IllegalArgumentException();
		}

		return BluetoothNatives.sendNotification(this.connHandle, characteristic.getHandle(), value, value.length,
				confirm);
	}
}
