/*
 * Java
 *
 * Copyright 2018-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.bluetooth;

import ej.annotation.Nullable;
import ej.bluetooth.tools.HexTools;

public class BluetoothAddress {

	/**
	 * Not in API.
	 */
	public static final int LENGTH = 7; // type (1) + addr (6)

	private static final byte TYPE_PUBLIC = 0x00;
	private static final byte TYPE_PRIVATE = 0x01;

	private static final int STRING_LENGTH = 17; // XX:XX:XX:XX:XX:XX

	private final byte[] bytes;

	public BluetoothAddress(byte[] bytes, int offset, boolean isPublic) {
		this.bytes = new byte[LENGTH];
		this.bytes[0] = (isPublic ? TYPE_PUBLIC : TYPE_PRIVATE);
		System.arraycopy(bytes, offset, this.bytes, 1, LENGTH - 1);
	}

	public BluetoothAddress(String string, boolean isPublic) {
		if (string.length() != STRING_LENGTH) {
			throw new IllegalArgumentException();
		}

		this.bytes = new byte[LENGTH];
		this.bytes[0] = (isPublic ? TYPE_PUBLIC : TYPE_PRIVATE);

		int stringIndex = 0;
		for (int i = 1; i < LENGTH; i++) {
			this.bytes[i] = HexTools.stringToByte(string, stringIndex);
			stringIndex += 2;

			if (i != LENGTH - 1) {
				if (string.charAt(stringIndex) != ':') {
					throw new IllegalArgumentException();
				}
				stringIndex++;
			}
		}
	}

	/**
	 * Not in API.
	 */
	public static BluetoothAddress deserialize(byte[] data, int offset) {
		byte type = data[offset];
		if (type != TYPE_PUBLIC && type != TYPE_PRIVATE) {
			throw new IllegalArgumentException();
		}
		return new BluetoothAddress(data, offset + 1, type == TYPE_PUBLIC);
	}

	/**
	 * Not in API.
	 */
	public static void serialize(BluetoothAddress address, byte[] buffer, int offset) {
		System.arraycopy(address.bytes, 0, buffer, offset, LENGTH);
	}

	public void getBytes(byte[] buffer, int offset) {
		System.arraycopy(this.bytes, 1, buffer, offset, LENGTH - 1);
	}

	public boolean isPublic() {
		return (this.bytes[0] == TYPE_PUBLIC);
	}

	@Override
	public String toString() {
		char[] charArray = new char[STRING_LENGTH];
		int stringIndex = 0;
		for (int i = 1; i < LENGTH; i++) {
			HexTools.byteToString(this.bytes[i], charArray, stringIndex);
			stringIndex += 2;

			if (i != LENGTH - 1) {
				charArray[stringIndex] = ':';
				stringIndex++;
			}
		}
		return new String(charArray);
	}

	@Override
	public boolean equals(@Nullable Object object) {
		if (object == this) {
			return true;
		}

		if (!(object instanceof BluetoothAddress)) {
			return false;
		}

		BluetoothAddress other = (BluetoothAddress) object;
		for (int i = 0; i < LENGTH; i++) {
			if (other.bytes[i] != this.bytes[i]) {
				return false;
			}
		}

		return true;
	}

	@Override
	public int hashCode() {
		int result = 1;
		for (byte element : this.bytes) {
			result = 31 * result + element;
		}
		return result;
	}
}
