/*
 * Java
 *
 * Copyright 2018-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.bluetooth;

import java.lang.Thread.UncaughtExceptionHandler;

import ej.annotation.Nullable;
import ej.bluetooth.listeners.ConnectionListener;
import ej.bluetooth.listeners.LocalServiceListener;
import ej.bluetooth.listeners.RemoteServiceListener;
import ej.bluetooth.listeners.impl.DefaultConnectionListener;
import ej.bluetooth.tools.ArrayTools;
import ej.kf.Feature;
import ej.kf.Feature.State;
import ej.kf.FeatureStateListener;
import ej.kf.Kernel;

public class BluetoothAdapter implements FeatureStateListener {

	private static final ConnectionListener DEFAULT_CONNECTION_LISTENER = new DefaultConnectionListener();

	private static @Nullable BluetoothAdapter instance;

	private final BluetoothPump pump;
	private final BluetoothDatabase database;

	private boolean enabled;
	private BluetoothConnection[] connections;
	private ConnectionListener connectionListener;

	private BluetoothAdapter() {
		this.pump = new BluetoothPump(this);
		this.database = new BluetoothDatabase();
		this.enabled = false;
		this.connections = new BluetoothConnection[0];
		this.connectionListener = DEFAULT_CONNECTION_LISTENER;
	}

	public static BluetoothAdapter getAdapter() {
		Kernel.enter();

		BluetoothAdapter adapter = instance;
		if (adapter == null) {
			adapter = new BluetoothAdapter();
			instance = adapter;
		}
		return adapter;
	}

	public boolean enable() {
		Kernel.enter();

		boolean enabled = this.enabled;
		if (!enabled) {
			enabled = BluetoothNatives.enable();
			if (enabled) {
				this.pump.start();
				Kernel.addFeatureStateListener(this);
				this.enabled = true;
			}
		}
		return enabled;
	}

	public void disable() {
		Kernel.enter();

		if (this.enabled) {
			Kernel.removeFeatureStateListener(this);
			this.pump.stop();
			BluetoothNatives.disable();

			this.database.clear();
			this.enabled = false;
			this.connections = new BluetoothConnection[0];
			this.connectionListener = DEFAULT_CONNECTION_LISTENER;
		}
	}

	@Override
	public void stateChanged(@Nullable Feature feature, @Nullable State previousState) {
		if (feature != null && feature.getState() == State.STOPPED) {
			// cleanup connection listener
			if (Kernel.getOwner(this.connectionListener) == feature) {
				this.connectionListener = DEFAULT_CONNECTION_LISTENER;
			}

			// cleanup local service listeners
			this.database.cleanupFeatureListeners(feature);

			// cleanup remote service listeners
			for (BluetoothConnection connection : this.connections) {
				connection.cleanupFeatureListeners(feature);
			}
		}
	}

	public @Nullable BluetoothService addService(BluetoothServiceDefinition definition) {
		Kernel.enter();

		// serialize definition
		int definitionSize = definition.getSerializedSize();
		byte[] data = new byte[definitionSize];
		BluetoothServiceDefinition.serialize(definition, data, 0);

		// create handles buffer
		int numHandles = 1 + definition.getNumAttributes();
		short[] handles = new short[numHandles];

		// add service
		if (!BluetoothNatives.addService(data, handles)) {
			return null;
		}

		// create service
		BluetoothService service = BluetoothServiceDefinition.createService(definition, handles);

		// add service to database
		this.database.addService(service);

		return service;
	}

	public void setConnectionListener(ConnectionListener connectionListener) {
		Kernel.enter();
		this.connectionListener = connectionListener;
	}

	public boolean startAdvertising(byte[] advertisementData) {
		return BluetoothNatives.startAdvertising(advertisementData, advertisementData.length);
	}

	public boolean stopAdvertising() {
		return BluetoothNatives.stopAdvertising();
	}

	public boolean startScanning(BluetoothScanFilter scanFilter) {
		byte filterAction = scanFilter.getAction();
		byte filterType = scanFilter.getType();
		byte[] filterData = scanFilter.getData();
		return BluetoothNatives.startScanning(filterAction, filterType, filterData, filterData.length);
	}

	public boolean stopScanning() {
		return BluetoothNatives.stopScanning();
	}

	public boolean connect(BluetoothAddress address) {
		byte[] buffer = new byte[BluetoothAddress.LENGTH];
		BluetoothAddress.serialize(address, buffer, 0);
		return BluetoothNatives.connect(buffer);
	}

	/**
	 * Not in API.
	 */
	public void handleScanResult(BluetoothAddress deviceAddress, byte[] advertisementData, int rssi) {
		try {
			this.connectionListener.onScanResult(deviceAddress, advertisementData, rssi);
		} catch (Exception e) {
			handleUncaughtException(e);
		}
	}

	/**
	 * Not in API.
	 */
	public void handleScanCompleted() {
		try {
			this.connectionListener.onScanCompleted();
		} catch (Exception e) {
			handleUncaughtException(e);
		}
	}

	/**
	 * Not in API.
	 */
	public void handleAdvertisementCompleted() {
		try {
			this.connectionListener.onAdvertisementCompleted();
		} catch (Exception e) {
			handleUncaughtException(e);
		}
	}

	/**
	 * Not in API.
	 */
	public void handleConnectFailed(BluetoothAddress deviceAddress) {
		try {
			this.connectionListener.onConnectFailed(deviceAddress);
		} catch (Exception e) {
			handleUncaughtException(e);
		}
	}

	/**
	 * Not in API.
	 */
	public void handleConnected(short connHandle, BluetoothAddress deviceAddress) {
		BluetoothConnection[] connections = this.connections;
		int index = getConnectionIndex(connections, connHandle);
		if (index == -1) {
			BluetoothConnection connection = new BluetoothConnection(deviceAddress, connHandle);
			this.connections = ArrayTools.append(connections, connection);

			try {
				this.connectionListener.onConnected(connection);
			} catch (Exception e) {
				handleUncaughtException(e);
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handleDisconnected(short connHandle) {
		BluetoothConnection[] connections = this.connections;
		int index = getConnectionIndex(connections, connHandle);
		if (index != -1) {
			BluetoothConnection connection = connections[index];
			assert (connection != null);
			this.connections = ArrayTools.remove(connections, index);

			try {
				this.connectionListener.onDisconnected(connection);
			} catch (Exception e) {
				handleUncaughtException(e);
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handlePairRequest(short connHandle) {
		BluetoothConnection connection = getConnection(connHandle);
		if (connection != null) {
			try {
				this.connectionListener.onPairRequest(connection);
			} catch (Exception e) {
				handleUncaughtException(e);
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handlePairCompleted(short connHandle, boolean success) {
		BluetoothConnection connection = getConnection(connHandle);
		if (connection != null) {
			try {
				this.connectionListener.onPairCompleted(connection, success);
			} catch (Exception e) {
				handleUncaughtException(e);
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handlePasskeyRequest(short connHandle) {
		BluetoothConnection connection = getConnection(connHandle);
		if (connection != null) {
			try {
				this.connectionListener.onPasskeyRequest(connection);
			} catch (Exception e) {
				handleUncaughtException(e);
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handlePasskeyGenerated(short connHandle, int passkey) {
		BluetoothConnection connection = getConnection(connHandle);
		if (connection != null) {
			try {
				this.connectionListener.onPasskeyGenerated(connection, passkey);
			} catch (Exception e) {
				handleUncaughtException(e);
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handleDiscoveryResult(short connHandle, BluetoothService service) {
		BluetoothConnection connection = getConnection(connHandle);
		if (connection != null) {
			// check there is no service with this handle already in the database
			BluetoothService identicalService = connection.getService(service.getHandle());
			if (identicalService == null) { // new service
				connection.addService(service);
			} else { // existing service
				service = identicalService;
			}

			try {
				this.connectionListener.onDiscoveryResult(connection, service);
			} catch (Exception e) {
				handleUncaughtException(e);
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handleDiscoveryCompleted(short connHandle) {
		BluetoothConnection connection = getConnection(connHandle);
		if (connection != null) {
			try {
				this.connectionListener.onDiscoveryCompleted(connection);
			} catch (Exception e) {
				handleUncaughtException(e);
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handleReadCompleted(short connHandle, short attributeHandle, byte status, byte[] value) {
		BluetoothConnection connection = getConnection(connHandle);
		if (connection != null) {
			BluetoothAttribute attribute = connection.getAttribute(attributeHandle);
			if (attribute != null) {
				BluetoothService service = attribute.getService();
				assert (service != null);
				RemoteServiceListener listener = service.getRemoteListener();

				try {
					listener.onReadCompleted(connection, attribute, status, value);
				} catch (Exception e) {
					handleUncaughtException(e);
				}
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handleWriteCompleted(short connHandle, short attributeHandle, byte status) {
		BluetoothConnection connection = getConnection(connHandle);
		if (connection != null) {
			BluetoothAttribute attribute = connection.getAttribute(attributeHandle);
			if (attribute != null) {
				BluetoothService service = attribute.getService();
				assert (service != null);
				RemoteServiceListener listener = service.getRemoteListener();

				try {
					listener.onWriteCompleted(connection, attribute, status);
				} catch (Exception e) {
					handleUncaughtException(e);
				}
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handleNotificationReceived(short connHandle, short attributeHandle, byte[] value) {
		BluetoothConnection connection = getConnection(connHandle);
		if (connection != null) {
			BluetoothAttribute attribute = connection.getAttribute(attributeHandle);
			if (attribute != null) {
				if (attribute instanceof BluetoothCharacteristic) {
					BluetoothService service = attribute.getService();
					assert (service != null);
					RemoteServiceListener listener = service.getRemoteListener();

					try {
						listener.onNotificationReceived(connection, (BluetoothCharacteristic) attribute, value);
					} catch (Exception e) {
						handleUncaughtException(e);
					}
				}
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handleReadRequest(short connHandle, short attributeHandle, int offset) {
		BluetoothConnection connection = getConnection(connHandle);
		BluetoothAttribute attribute = this.database.getAttribute(attributeHandle);
		if (connection != null && attribute != null) {
			BluetoothService service = attribute.getService();
			assert (service != null);
			LocalServiceListener listener = service.getLocalListener();

			try {
				if (offset > 0) {
					listener.onReadBlobRequest(connection, attribute, offset);
				} else {
					listener.onReadRequest(connection, attribute);
				}
			} catch (Exception e) {
				handleUncaughtException(e);
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handleWriteRequest(short connHandle, short attributeHandle, byte[] value, short offset,
			boolean prepare) {
		BluetoothConnection connection = getConnection(connHandle);
		BluetoothAttribute attribute = this.database.getAttribute(attributeHandle);
		if (connection != null && attribute != null) {
			BluetoothService service = attribute.getService();
			assert (service != null);
			LocalServiceListener listener = service.getLocalListener();

			try {
				if (prepare) {
					listener.onPrepareWriteRequest(connection, attribute, value, offset);
				} else {
					listener.onWriteRequest(connection, attribute, value);
				}
			} catch (Exception e) {
				handleUncaughtException(e);
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handleExecuteWriteRequest(short connHandle, short attributeHandle, boolean execute) {
		BluetoothConnection connection = getConnection(connHandle);
		BluetoothAttribute attribute = this.database.getAttribute(attributeHandle);
		if (connection != null && attribute != null) {
			BluetoothService service = attribute.getService();
			assert (service != null);
			LocalServiceListener listener = service.getLocalListener();

			try {
				listener.onExecuteWriteRequest(connection, attribute, execute);
			} catch (Exception e) {
				handleUncaughtException(e);
			}
		}
	}

	/**
	 * Not in API.
	 */
	public void handleNotificationSent(short connHandle, short attributeHandle, boolean success) {
		BluetoothConnection connection = getConnection(connHandle);
		BluetoothAttribute attribute = this.database.getAttribute(attributeHandle);
		if (connection != null && attribute instanceof BluetoothCharacteristic) {
			BluetoothService service = attribute.getService();
			assert (service != null);
			LocalServiceListener listener = service.getLocalListener();

			try {
				listener.onNotificationSent(connection, (BluetoothCharacteristic) attribute, success);
			} catch (Exception e) {
				handleUncaughtException(e);
			}
		}
	}

	private @Nullable BluetoothConnection getConnection(short connHandle) {
		BluetoothConnection[] connections = this.connections;
		int index = getConnectionIndex(connections, connHandle);
		return (index == -1 ? null : connections[index]);
	}

	private static int getConnectionIndex(BluetoothConnection[] connections, short connHandle) {
		for (int i = 0; i < connections.length; i++) {
			if (connections[i].getHandle() == connHandle) {
				return i;
			}
		}
		return -1;
	}

	private static void handleUncaughtException(Exception e) {
		// get uncaught exception handler
		Thread currentThread = Thread.currentThread();
		UncaughtExceptionHandler handler = currentThread.getUncaughtExceptionHandler();
		if (handler == null) {
			handler = Thread.getDefaultUncaughtExceptionHandler();
		}

		// handle exception
		if (handler != null) {
			handler.uncaughtException(currentThread, e);
		} else {
			e.printStackTrace();
		}
	}
}
