/*
 * Java
 *
 * Copyright 2014 IS2T. All rights reserved
 * IS2T PROPRIETARY/CONFIDENTIAL Use is subject to license terms.
 */
package com.is2t.schedcontrol;

import java.util.ArrayList;
import java.util.List;

/**
 * The philosophy is to be able to ensure a minimum execution time to some threads depending on a time quota.
 * A thread is associated to a {@link ScheduleContext} that controls its executions.<p>
 * 
 * Each {@link ScheduleContext} has a time quota expressed in quanta of time. A quantum is not directly related to any time unit (such as millisecond) and is defined by the VM execution.<p>
 * 
 * When a {@link ScheduleContext} is created its quantum counter is initialized with it quantum quota. When a thread is running (i.e. execute some code), the VM decrement the quantum counter
 * of its {@link ScheduleContext} regularly. When the quantum counter of a {@link ScheduleContext} reaches 0, the threads in this {@link ScheduleContext} are no more eligible to scheduling.<p>
 * 
 * If there is no thread eligible to scheduling (because they are in waiting state or the quantum counter of their {@link ScheduleContext} is at 0) then the quantum counter of each
 * {@link ScheduleContext} is reset to its quantum quota.<p>
 *
 * A {@link ScheduleContext} with a quantum quota at 0 will never be scheduled. A {@link ScheduleContext} with a quantum quota at -1 is always eligible to scheduling.<p>
 * 
 * The {@link ScheduleContext} time quota can be dynamically defined with {@link ScheduleContext#setQuantumQuota(int)}. A thread can change its {@link ScheduleContext} dynamically with
 * {@link ScheduleContext#addThread(Thread)}.<p>
 * 
 * For monitoring purpose, a global monitoring quantum counter is defined in the system and each {@link ScheduleContext} has a monitoring quantum counter. These counters are incremented by the VM.<p>
 * 
 * By default, a thread is associated to the {@link ScheduleContext} of the thread that starts it. The main thread is associated to a context with a quantum quota at -1.
 */
public class ScheduleController {

	/*
	 * WARNING: VM knows this class.
	 * 
	 * This class must be added as entry point to enable schedule control.
	 */
	
	
	//VM assumes that when this field is set, the scheduling control is enabled and all the threads have a schedule context.
	private static ScheduleController Singleton = new ScheduleController(); // VM Knows this field (don't move it)
	
	/**
	 * Total number of quantum consumed by the system.
	 */
	volatile private long consumedQuantum; // VM Knows this field (don't move it)
	
	/**
	 * Used by {@link #startThread(Thread, ScheduleContext)}
	 */
	volatile private Thread startingThread;// VM Knows this field (don't move it)
	/**
	 * Used by {@link #startThread(Thread, ScheduleContext)}
	 */
	volatile private ScheduleContext startingContext;// VM Knows this field (don't move it)
	
	/**
	 * Linked list of references to context.
	 * Don't reference directly the contexts to permit their garbage collection.<br>
	 * Elements in the list that have been GC (i.e. WeakRef reference null) will be removed
	 * by the VM. When browsing or modifying this list, take care to disable contexts cleanup
	 * by using {@link #setEnabledContextsCleanup()}
	 */
	private ScheduleContextWeakRef contexts;// VM Knows this field (don't move it)
		
	
	private ScheduleController(){
		//This constructor is called in the clinit by the main thread. Set its context (see comment of the Singleton static).
		newContext(-1).addThread(Thread.currentThread());
	}
	
	/**
	 * @return the {@link ScheduleController} of the system
	 */
	public static ScheduleController getScheduleController(){
		return Singleton;
	}
	
	/**
	 * Create a new {@link ScheduleContext} with the given quota.
	 * @param quota the quota to set to the new {@link ScheduleContext}.
	 * @return a new {@link ScheduleContext}.
	 * @see ScheduleContext#setQuantumQuota(int)
	 * 
	 * @throws IllegalArgumentException if the given quota is lower than -1.
	 */
	synchronized public ScheduleContext newContext(int quota){
		ScheduleContext context = new ScheduleContext(quota);
		setEnabledContextsCleanup(false);//We will browse the linked list -> disable contexts cleanup
		try{
			ScheduleContextWeakRef nextContextInList = contexts;
			ScheduleContextWeakRef rootContextInList = new ScheduleContextWeakRef(context);
			//First link the new root with the next context and then update the root field in order
			//to always keep a good linked list of context.
			//This linked list may be browsed by the VM.
			rootContextInList.nextScheduleContextWeakRef = nextContextInList;
			this.contexts = rootContextInList;
		}
		finally {
			setEnabledContextsCleanup(true);
		}
		return context;
	}
	
	/**
	 * @return the {@link ScheduleContext} of the given thread. Null if the given thread is not started.
	 */
	native public ScheduleContext getContext(Thread t);
	
	/**
	 * @return the total amount of quantum consumed by the system.
	 */
	public long getQuantumCounter(){
		return consumedQuantum;
	}
	
	/**
	 * Reset all the monitoring quantum counters of every {@link ScheduleContext} and global monitoring quantum counter.
	 */
	public void resetAllQuantumCounters(){
		//We will browse the linked list -> disable contexts cleanup
		setEnabledContextsCleanup(false);
		//disable monitoring counters in order to get all the counters set to 0 at the same moment.
		setEnabledMonitoringCounters(false);
		try{
			consumedQuantum = 0;
			ScheduleContextWeakRef contextRef = this.contexts;
			while(contextRef != null){
				ScheduleContext sc = contextRef.get();
				if(sc != null){
					sc.consumedQuantum = 0;
				}
				contextRef = contextRef.nextScheduleContextWeakRef;
			}
		}
		finally {
			setEnabledMonitoringCounters(true);
			setEnabledContextsCleanup(true);
		}
		
	}
	
	/**
	 * @return a list of all the currently used contexts.
	 */
	public List<ScheduleContext> getAllContexts(){
		ArrayList<ScheduleContext> contextsList = new ArrayList<>();
		
		setEnabledContextsCleanup(false);//We will browse the linked list -> disable contexts cleanup
		try{
			ScheduleContextWeakRef contextRef = this.contexts;
			while(contextRef != null){
				ScheduleContext sc = contextRef.get();
				if(sc != null){
					contextsList.add(sc);
				}
				contextRef = contextRef.nextScheduleContextWeakRef;
			}
		}
		finally {
			setEnabledContextsCleanup(true);
		}
		
		return contextsList;
	}
	
	/**
	 * Enables or disables the incrementation of consumedQuantum counters used for monitoring.<p>
	 * By default monitoring counters are enabled.
	 */
	native private void setEnabledMonitoringCounters(boolean b);

	/**
	 * Enables or disables the automatic cleanup of GC element in the {@link #contexts} linked list.<p>
	 * By default cleanup is enabled.
	 */
	native private void setEnabledContextsCleanup(boolean b);
	
	/**
	 * @see ScheduleContext#startThread(Thread)
	 */
	/* default */ synchronized void startThread(Thread thread, ScheduleContext context){
		
		//Store in the controller the thread to start and the context of this thread.
		//These values will be used in the native code to check if the thread use a
		//specific context.
		//We need to store both the thread and context because another thread may be
		//started in parallel without using this method. The native code will check
		//that it assigns the context to the right thread.
		//
		//WARNING: keep this method synchronized to avoid concurrent accesses to these
		//two fields.
		
		this.startingThread = thread;
		this.startingContext = context;
		thread.start();
		this.startingContext = null;
		this.startingThread = null;
	}
	

	/**
	 * Reset the quantum of all the contexts.
	 */
	native /*default*/ static void resetQuantumOfAllContexts();
}
