/*
 * Java
 *
 * Copyright 2014 IS2T. All rights reserved
 * IS2T PROPRIETARY/CONFIDENTIAL Use is subject to license terms.
 */
package com.is2t.schedcontrol;

/**
 * A {@link ScheduleContext} is a scheduling policy that will be applied to a set of threads.
 * A thread can be associated to only one {@link ScheduleContext}.<p>
 * When a thread is running, the quantum counter of its {@link ScheduleContext} is decremented by the VM.
 * 
 * @see ScheduleController
 */
public class ScheduleContext {

	/*
	 * WARNING: VM knows this class.
	 */

	/**
	 * Maximum number of quantum that can be consumed by this context.
	 */
	volatile private int quota;// VM Knows this field (don't move it)
	
	/**
	 * Remaining quantum for this context.
	 */
	volatile private int counter;// VM Knows this field (don't move it)
	
	/**
	 * Total number of quantum consumed by this context.
	 */
	volatile /*default*/ long consumedQuantum;// VM Knows this field (don't move it)
	
	
	private ScheduleContext() {
		throw new RuntimeException();
	}
	
	/*default*/ ScheduleContext(int quota){
		this.setQuantumQuotaIntern(quota);
		this.counter = quota;
	}
	
	/**
	 * Set the time quota of the threads of this context. This quota is expressed in quanta of time.<p>
	 * A {@link ScheduleContext} with a quantum quota at 0 will never be scheduled.<p>
	 * A {@link ScheduleContext} with a quantum quota at -1 is always eligible to scheduling.<p>
	 * Calling this method induces a global reset of the quantum of all the contexts.
	 * @param quota the quota to set to this {@link ScheduleContext}.
	 * 
	 * @throws IllegalArgumentException if the given quota is lower than -1.
	 */
	public void setQuantumQuota(int quota){
		//Set the quota and launch a quantum reset
		setQuantumQuotaIntern(quota);
		ScheduleController.resetQuantumOfAllContexts();
	}
	
	/**
	 * Just set the quota value
	 */
	private void setQuantumQuotaIntern(int quota){
		if(quota < -1){
			throw new IllegalArgumentException(Integer.toString(quota));
		}
		
		this.quota = quota;
	}
	
	/**
	 * @return the quantum quota of this {@link ScheduleContext}.
	 */
	public int getQuantumQuota(){
		return quota;
	}
	
	/**
	 * @return the total amount of quantum that have been consumed by this {@link ScheduleContext}.
	 */
	public long getQuantumCounter(){
		return consumedQuantum;
	}
	
	/**
	 * Add the given thread in this {@link ScheduleContext}.
	 * @param thread the thread to add in the context.
	 * @return the previous context of the thread. Null if the thread was not associated to any context.
	 * 
	 * @throws IllegalThreadStateException if the thread is not started.
	 */
	native public ScheduleContext addThread(Thread thread);
	
	/**
	 * Starts the given thread in this context.
	 * @param thread the thread to start
	 * 
	 * @throws IllegalThreadStateException if the thread was already started.
	 * 
	 * @see Thread#start()
	 */
	public void startThread(Thread thread){
		ScheduleController.getScheduleController().startThread(thread, this);
	}
	
}
