/*
 * Java
 *
 * Copyright 2010-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.led;

import com.is2t.microbsp.microui.natives.NSystemLedsImpl;

/**
 * microUI-API
 * This class is used to manage all LEDs available on the platform. The available number of LEDs is
 * known thanks to the method {@link #getNumberOfLeds()}. A LED is identified by
 * its id. The ids range from 0 to <code>getNumberOfLeds()-1</code>.
 */
public class Leds {

	/* *****************************************************
	 * Constants
	 *******************************************************/

	/**
	 * microUI-API
	 * Intensity value to turn off a LED.
	 */
	public static final int MIN_INTENSITY	= 0;

	/**
	 * microUI-API
	 * Maximum intensity that a LED can handle. If a LED does not handle intensity, any
	 * valid intensity different from {@link #MIN_INTENSITY} turns the LED on.
	 */
	public static final int MAX_INTENSITY	= 255;

	/* *****************************************************
	 * Fields
	 *******************************************************/

	private static int NbLeds; // 0 init

	/* *****************************************************
	 * API
	 *******************************************************/

	/**
	 * microUI-API
	 * Forbidden constructor: cannot instantiate a <code>Leds</code> object.
	 */
	private Leds() {
	}

	/**
	 * microUI-API
	 * Returns the available number of LEDs. The range of valid led ids is [0..Leds.getNumberOfLeds()-1].
	 * @return the number of leds
	 */
	public static int getNumberOfLeds() {
		return NbLeds;
	}

	/**
	 * microUI-API
	 * Controls the intensity of the specified LED. If the id is invalid (out of range) the
	 * method has no effect.
	 * @param ledId the led identifier
	 * @param intensity the intensity to set on the led
	 */
	public static void setLedIntensity(int ledId, int intensity) {

		if (ledId < 0 || ledId >= NbLeds)
		{
			return;	// invalid led id -> spec: do nothing
		}

		if (intensity < MIN_INTENSITY) {
			intensity = MIN_INTENSITY;
		} else if (intensity > MAX_INTENSITY) {
			intensity = MAX_INTENSITY;
		}

		//Instance cannot be null because NbLeds > 0
		NSystemLedsImpl.setLedIntensityNative(ledId, intensity);
	}

	/**
	 * microUI-API
	 * Gets the intensity of the specified LED. If the id is invalid (out of range) the
	 * method returns 0.
	 * @param ledId the led identifier
	 * @return the led intensity
	 */
	public static int getLedIntensity(int ledId) {

		if (ledId < 0 || ledId >= NbLeds)
		{
			return 0;	// invalid led id -> spec: do nothing
		}

		return NSystemLedsImpl.getLedIntensityNative(ledId);
	}

	/**
	 * microUI-API
	 * Turns on the given LED.
	 * The effect is identical to <code>Leds.setLedIntensity(ledId, MAX_INTENSITY)</code>.
	 * @param ledId the led identifier
	 */
	public static void setLedOn(int ledId) {
		setLedIntensity(ledId, MAX_INTENSITY);
	}

	/**
	 * microUI-API
	 * Turns off the given LED.
	 * The effect is identical to <code>Leds.setLedIntensity(ledId, MIN_INTENSITY)</code>.
	 * @param ledId the led identifier
	 */
	public static void setLedOff(int ledId) {
		setLedIntensity(ledId, MIN_INTENSITY);
	}

	/* *****************************************************
	 * NOT IN API
	 *******************************************************/

	static {
		NbLeds = Math.max(0, NSystemLedsImpl.getNbLeds());
	}
}
