/*
 * Java
 *
 * Copyright 2012-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event.generator;

import ej.microui.event.Event;
import ej.microui.event.EventGenerator;

/**
 * microUI-API
 */
public class States extends EventGenerator {

	/**
	 * microUI-API
	 */
	public static final int EVENT_TYPE = 0x03;

	private static final int VALUE_MASK = 0x0000ff00;
	private static final int VALUE_SHIFT = 8;
	private static final int STATEID_MASK = 0x000000ff;
	private static final int STATEID_SHIFT = 0;
	private static final int STATES_MAX = 255;

	private final byte[] states;
	private final byte[] nbValues;

	/**
	 * microUI-API
	 *
	 * @param nbValues
	 *            microUI-API
	 * @param initialValues
	 *            microUI-API
	 */
	public States(int[] nbValues, int[] initialValues) {
		assert nbValues != null && initialValues != null;
		int length;
		if ((length = initialValues.length) != nbValues.length) {
			throw new IllegalArgumentException();
		}

		if (length > STATES_MAX) {
			throw new IndexOutOfBoundsException();
		}

		this.states = new byte[length];
		this.nbValues = new byte[length];
		for (int i = length; --i >= 0;) {
			int nb = nbValues[i];
			if (nb < 0 || nb > STATES_MAX) {
				throw new IndexOutOfBoundsException();
			}
			this.nbValues[i] = (byte) nb;
			int init = initialValues[i];
			if (init < 0 || init >= nb) {
				throw new IndexOutOfBoundsException();
			}
			this.states[i] = (byte) init;
		}
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static int getStateId(int event) {
		return (event & STATEID_MASK) >>> STATEID_SHIFT;
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static int getStateValue(int event) {
		return (event & VALUE_MASK) >>> VALUE_SHIFT;
	}

	@Override
	public int getEventType() {
		return EVENT_TYPE;
	}

	/**
	 * microUI-API
	 *
	 * @param stateId
	 *            microUI-API
	 * @return microUI-API
	 */
	public int getNumberOfValues(int stateId) {
		return this.nbValues[stateId] & 0xFF; // ArrayIndexOutOfBoundsException instanceof IndexOutOfBoundsException
	}

	/**
	 * microUI-API
	 *
	 * @param stateId
	 *            microUI-API
	 * @return microUI-API
	 */
	public int getCurrentValue(int stateId) {
		return this.states[stateId] & 0xFF; // ArrayIndexOutOfBoundsException instanceof IndexOutOfBoundsException
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getNumberOfStates() {
		return this.states.length;
	}

	/**
	 * microUI-API
	 *
	 * @param stateId
	 *            microUI-API
	 * @param value
	 *            microUI-API
	 * @return microUI-API
	 */
	public int buildEvent(int stateId, int value) {
		return Event.buildEvent(getEventType(), this, (stateId << STATEID_SHIFT) | (value << VALUE_SHIFT));
	}

	/**
	 * microUI-API
	 *
	 * @param stateId
	 *            microUI-API
	 * @param value
	 *            microUI-API
	 */
	public void send(int stateId, int value) {
		update(stateId, value);
		sendEvent(buildEvent(stateId, value), getEventHandler());
	}

	/**
	 * Updates state action
	 *
	 * @param stateId
	 *            the state identifier value.
	 * @param value
	 *            the value to set
	 */
	protected void update(int stateId, int value) {
		if (value < 0 || value >= getNumberOfValues(stateId)) {
			throw new IndexOutOfBoundsException();
		}
		this.states[stateId] = (byte) value;
	}
}
