/*
 * Java
 *
 * Copyright 2010-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event.generator;

import ej.annotation.Nullable;
import ej.bon.XMath;
import ej.microui.display.Display;
import ej.microui.event.EventHandler;

/**
 * microUI-API
 */
public class Pointer extends Buttons {

	/**
	 * microUI-API
	 */
	public static final int EVENT_TYPE = 0x02;

	/**
	 * microUI-API
	 */
	public static final int MOVED = 0x06;

	/**
	 * microUI-API
	 */
	public static final int DRAGGED = 0x07;

	private final int physicalWidth;
	private final int physicalHeight;
	private int lastX;
	private int lastY;
	private int lastAbsoluteX;
	private int lastAbsoluteY;
	private int scaledWidth;
	private int scaledHeight;
	private int x0;
	private int y0;
	private int buttonsState;

	/**
	 * microUI-API
	 *
	 * @param nbButtons
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 */
	public Pointer(int nbButtons, int width, int height) {
		super(nbButtons);
		assert width > 0 && height > 0;
		this.physicalWidth = this.scaledWidth = width;
		this.physicalHeight = this.scaledHeight = height;
	}

	/**
	 * microUI-API
	 *
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 */
	public Pointer(int width, int height) {
		this(0, width, height);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getAbsoluteWidth() {
		return this.physicalWidth;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getAbsoluteHeight() {
		return this.physicalHeight;
	}

	/**
	 * microUI-API microUI-API
	 *
	 * @return microUI-API
	 */
	@Override
	public int getEventType() {
		return EVENT_TYPE;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getX() {
		return this.lastX;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getY() {
		return this.lastY;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getAbsoluteX() {
		return this.lastAbsoluteX;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getAbsoluteY() {
		return this.lastAbsoluteY;
	}

	/**
	 * not in API
	 * <p>
	 * Kept for backward compatibility (called by system microui generator)
	 *
	 * @param display
	 *            microUI-API
	 */
	public void setScale(Display display) {
		setScale(display.getWidth(), display.getHeight());
	}

	/**
	 * microUI-API
	 *
	 * @param areaWidth
	 *            microUI-API
	 * @param areaHeight
	 *            microUI-API
	 */
	public void setScale(int areaWidth, int areaHeight) {
		assert areaWidth > 0 && areaHeight > 0;
		this.scaledWidth = areaWidth;
		this.scaledHeight = areaHeight;
	}

	/**
	 * microUI-API
	 *
	 * @param x0
	 *            microUI-API
	 * @param y0
	 *            microUI-API
	 */
	public void setOrigin(int x0, int y0) {

		if (x0 >= this.physicalWidth) {
			x0 = this.physicalWidth - 1;
		}
		if (y0 >= this.physicalHeight) {
			y0 = this.physicalHeight - 1;
		}
		if (x0 < 0) {
			x0 = 0;
		}
		if (y0 < 0) {
			y0 = 0;
		}

		this.x0 = x0;
		this.y0 = y0;

		update(this.lastAbsoluteX, this.lastAbsoluteY);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getWidth() {
		return this.scaledWidth;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getHeight() {
		return this.scaledHeight;
	}

	/**
	 * microUI-API
	 *
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 */
	public void move(int x, int y) {
		update(x, y);
		sendEvent(buildEvent(this.buttonsState == 0 ? MOVED : DRAGGED, 0), getEventHandler()); // 0: no data
	}

	/**
	 * microUI-API
	 *
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 */
	public void reset(int x, int y) {
		update(x, y);
	}

	@Override
	protected void send(int event, @Nullable EventHandler eventHandler) {
		// update button state
		int action = getAction(event);
		if (action == PRESSED) {
			this.buttonsState |= (0x1 << getButtonId(event));
		} else if (action == RELEASED) {
			this.buttonsState &= ~(0x1 << getButtonId(event));
		}
		super.send(event, eventHandler);
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static boolean isMoved(int event) {
		return getAction(event) == MOVED;
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static boolean isDragged(int event) {
		return getAction(event) == DRAGGED;
	}

	/**
	 * Gets the current buttons state. A button is pressed when (state & (1 << buttonId) != 0).
	 *
	 * @return 0 means all buttons are released.
	 */
	protected int getButtonsState() {
		return this.buttonsState;
	}

	private void update(int x, int y) {
		// pointer clip (sanity check)
		this.lastAbsoluteX = XMath.limit(x, 0, this.physicalWidth - 1);
		this.lastAbsoluteY = XMath.limit(y, 0, this.physicalHeight - 1);

		// new scaled position
		this.lastX = ((this.lastAbsoluteX - this.x0) * this.scaledWidth) / this.physicalWidth;
		this.lastY = ((this.lastAbsoluteY - this.y0) * this.scaledHeight) / this.physicalHeight;
	}
}
