/*
 * Java
 *
 * Copyright 2012-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event.generator;

import ej.microui.MicroUIPump;
import ej.microui.event.EventGenerator;
import ej.microui.event.EventHandler;

/**
 * Generic event generator.
 * <p>
 * Generic communication to receive events.
 */
public abstract class GenericEventGenerator extends EventGenerator {

	// masks & shifts
	private static final int DRIVER_LENGTH_SHIFT = 0;
	private static final int DRIVER_LENGTH_MASK = 0xff << DRIVER_LENGTH_SHIFT;

	/**
	 * Given event handler is the current displayable (may be null) when the event
	 * generator's handler is the display pump. Otherwise it is the generator
	 * returned by {@link #getEventHandler()}.
	 */
	@Override
	public final void convert(MicroUIPump pump, int inputEvent, EventHandler eventHandler) throws InterruptedException {

		// get the number of custom events
		int nbEvents = (inputEvent & DRIVER_LENGTH_MASK) >>> DRIVER_LENGTH_SHIFT;
		if (nbEvents == 1) {
			// only one event
			eventReceived(pump.read(), eventHandler);
		} else {
			// several events
			int[] events = new int[nbEvents];

			// read all events
			for (int i = -1; ++i < nbEvents;) { // respect the order
				events[i] = pump.read();
			}

			eventsReceived(events, eventHandler);
		}
	}

	/**
	 * microUI-API
	 *
	 * @param name
	 *            microUI-API
	 * @param value
	 *            microUI-API
	 */
	public abstract void setProperty(String name, String value);

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @param handler
	 *            microUI-API
	 */
	protected void eventReceived(int event, EventHandler handler) {
		eventReceived(event); // MicroUI spec
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 */
	protected abstract void eventReceived(int event);

	/**
	 * microUI-API
	 *
	 * @param events
	 *            microUI-API
	 * @param handler
	 *            microUI-API
	 */
	protected void eventsReceived(int[] events, EventHandler handler) {
		eventsReceived(events); // MicroUI spec
	}

	/**
	 * microUI-API
	 *
	 * @param events
	 *            microUI-API
	 */
	protected abstract void eventsReceived(int[] events);
}
