/*
 * Java
 *
 * Copyright 2010-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event.generator;

import ej.microui.event.Event;
import ej.microui.event.EventGenerator;

/**
 * microUI-API <code>Command</code> is an event generator that generates
 * application-level events. Unlike io generators that are related to some
 * hardware input format, the command generator defines its own input format.
 * Basically input and output event are the same. This allows the generation of
 * commands from within MicroUI without relying on an underlying input event
 * format.<br>
 * <br>
 * This class defines constants for a set of basic commands. The commands
 * defined in this class are typical application-level effects of input events.
 * The advantage of using commands rather than specific input events in an
 * application is that the application can be more portable: it is not tied to
 * specific input devices.
 */

public class Command extends EventGenerator {

	/**
	 * microUI-API
	 */
	public static final int EVENT_TYPE = 0x00;

	/**
	 * microUI-API The "escape" command constant.<br>
	 * The value <code>0x0000</code> is assigned to <code>ESC</code>.
	 */
	public static final int ESC = 0x0000;
	/**
	 * microUI-API The "back" command constant.<br>
	 * The value <code>0x0001</code> is assigned to <code>BACK</code>.
	 */
	public static final int BACK = 0x0001;
	/**
	 * microUI-API The "up" command constant.<br>
	 * The value <code>0x0002</code> is assigned to <code>UP</code>.
	 */
	public static final int UP = 0x0002;
	/**
	 * microUI-API The "down" command constant.<br>
	 * The value <code>0x0003</code> is assigned to <code>DOWN</code>.
	 */
	public static final int DOWN = 0x0003;
	/**
	 * microUI-API The "left" command constant.<br>
	 * The value <code>0x0004</code> is assigned to <code>LEFT</code>.
	 */
	public static final int LEFT = 0x0004;
	/**
	 * microUI-API The "right" command constant.<br>
	 * The value <code>0x0005</code> is assigned to <code>RIGHT</code>.
	 */
	public static final int RIGHT = 0x0005;
	/**
	 * microUI-API The "select" command constant.<br>
	 * The value <code>0x0006</code> is assigned to <code>SELECT</code>.
	 */
	public static final int SELECT = 0x0006;
	/**
	 * microUI-API The "cancel" command constant.<br>
	 * The value <code>0x0007</code> is assigned to <code>CANCEL</code>.
	 */
	public static final int CANCEL = 0x0007;
	/**
	 * microUI-API The "help" command constant.<br>
	 * The value <code>0x0008</code> is assigned to <code>HELP</code>.
	 */
	public static final int HELP = 0x0008;

	/**
	 * microUI-API The "menu" command constant.<br>
	 * The value <code>0x0009</code> is assigned to <code>MENU</code>.
	 */
	public static final int MENU = 0x0009;

	/**
	 * microUI-API The "exit" command constant.<br>
	 * The value <code>0x000A</code> is assigned to <code>EXIT</code>.
	 */
	public static final int EXIT = 0x000A;

	/**
	 * microUI-API The "start" command constant.<br>
	 * The value <code>0x000B</code> is assigned to <code>START</code>.
	 */
	public static final int START = 0x000B;

	/**
	 * microUI-API The "stop" command constant.<br>
	 * The value <code>0x000C</code> is assigned to <code>STOP</code>.
	 */
	public static final int STOP = 0x000C;

	/**
	 * microUI-API The "pause" command constant.<br>
	 * The value <code>0x000D</code> is assigned to <code>PAUSE</code>.
	 */
	public static final int PAUSE = 0x000D;

	/**
	 * microUI-API The "resume" command constant.<br>
	 * The value <code>0x000E</code> is assigned to <code>RESUME</code>.
	 */
	public static final int RESUME = 0x000E;

	/**
	 * microUI-API The "copy" command constant.<br>
	 * The value <code>0x000F</code> is assigned to <code>COPY</code>.
	 */
	public static final int COPY = 0x000F;

	/**
	 * microUI-API The "cut" command constant.<br>
	 * The value <code>0x0010</code> is assigned to <code>CUT</code>.
	 */
	public static final int CUT = 0x0010;

	/**
	 * microUI-API The "paste" command constant.<br>
	 * The value <code>0x0011</code> is assigned to <code>PASTE</code>.
	 */
	public static final int PASTE = 0x0011;
	/**
	 * microUI-API The "clockwise" command constant.<br>
	 * The value <code>0x0012</code> is assigned to <code>CLOCKWISE</code>.
	 */
	public static final int CLOCKWISE = 0x0012;
	/**
	 * microUI-API The "anti-clockwise" command constant.<br>
	 * The value <code>0x0013</code> is assigned to <code>ANTICLOCKWISE</code>.
	 */
	public static final int ANTICLOCKWISE = 0x0013;
	/**
	 * microUI-API The "previous" command constant.<br>
	 * The value <code>0x0014</code> is assigned to <code>PREVIOUS</code>.
	 */
	public static final int PREVIOUS = 0x0014;
	/**
	 * microUI-API The "next" command constant.<br>
	 * The value <code>0x0015</code> is assigned to <code>NEXT</code>.
	 */
	public static final int NEXT = 0x0015;
	/**
	 * microUI-API The "display" command constant.<br>
	 * The value <code>0x0016</code> is assigned to <code>DISPLAY</code>.
	 */
	public static final int DISPLAY = 0x0016;

	/**
	 * microUI-API Creates a new command <code>EventGenerator</code>.
	 */
	public Command() {
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	@Override
	public int getEventType() {
		return EVENT_TYPE;
	}

	/**
	 * microUI-API
	 *
	 * @param command
	 *            microUI-API
	 * @return microUI-API
	 */
	public int buildEvent(int command) {
		return Event.buildEvent(getEventType(), this, command);
	}

	/**
	 * microUI-API
	 *
	 * @param command
	 *            microUI-API
	 */
	public void send(int command) {
		sendEvent(buildEvent(command), getEventHandler());
	}
}
