/*
 * Java
 *
 * Copyright 2010-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event.generator;

import ej.annotation.Nullable;
import ej.bon.Util;
import ej.microui.event.Event;
import ej.microui.event.EventGenerator;
import ej.microui.event.EventHandler;

/**
 * microUI-API
 */
public class Buttons extends EventGenerator {

	/**
	 * microUI-API
	 */
	public static final int EVENT_TYPE = 0x01;

	/**
	 * microUI-API
	 */
	public static final int PRESSED = 0x00;

	/**
	 * microUI-API
	 */
	public static final int RELEASED = 0x01;

	/**
	 * microUI-API
	 */
	public static final int LONG = 0x02;

	/**
	 * microUI-API
	 */
	public static final int REPEATED = 0x03;

	/**
	 * microUI-API
	 */
	public static final int CLICKED = 0x04;

	/**
	 * microUI-API
	 */
	public static final int DOUBLE_CLICKED = 0x05;

	private static final int ACTION_MASK = 0x0000ff00;
	private static final int ACTION_SHIFT = 8;
	private static final int BUTTON_MASK = 0x000000ff;
	private static final int BUTTON_SHIFT = 0;

	private static final int DEFAULT_DOUBLE_CLICK = 200;// ms
	private static final int DOUBLE_CLICK_DISABLED = -1;
	private static final int BUTTONS_MAX = 255;

	// all these arrays have the same size (see supportsExtendedFeatures),
	private final long[] elapsedTimes;
	private final long[] eventTimes;
	private final int[] doubleClicks;
	private final boolean[] clicksOff;

	/**
	 * microUI-API
	 *
	 * @param nbButtons
	 *            microUI-API
	 */
	public Buttons(int nbButtons) {

		// sanity check
		if (nbButtons < 0 || nbButtons > BUTTONS_MAX) {
			nbButtons = 0;
		}

		this.elapsedTimes = new long[nbButtons];
		this.eventTimes = new long[nbButtons];
		this.clicksOff = new boolean[nbButtons];
		this.doubleClicks = new int[nbButtons];
		for (int i = nbButtons; --i >= 0;) {
			this.doubleClicks[i] = DEFAULT_DOUBLE_CLICK; // 200ms
			this.eventTimes[i] = Integer.MIN_VALUE;
		}
	}

	/**
	 * microUI-API
	 */
	public Buttons() {
		this(0); // no extra support
	}

	/**
	 * microUI-API
	 *
	 * @param enable
	 *            microUI-API
	 * @param buttonId
	 *            microUI-API
	 */
	public void enableClick(boolean enable, int buttonId) {
		if (supportsExtendedFeatures(buttonId)) {
			this.clicksOff[buttonId] = !enable;
		}
	}

	/**
	 * microUI-API
	 *
	 * @param enable
	 *            microUI-API
	 * @param click
	 *            microUI-API
	 * @param buttonId
	 *            microUI-API
	 */
	public void enableDoubleClick(boolean enable, int click, int buttonId) {
		if (supportsExtendedFeatures(buttonId)) {
			if (click < 0) {
				click = DEFAULT_DOUBLE_CLICK; // sanity (not spec required)
			}
			if (!enable) {
				click = DOUBLE_CLICK_DISABLED;
			} else {
				// force enable click
				this.clicksOff[buttonId] = false;
			}
			this.doubleClicks[buttonId] = click;
		}
	}

	/**
	 * microUI-API
	 *
	 * @param buttonId
	 *            microUI-API
	 * @return microUI-API
	 */
	public boolean supportsExtendedFeatures(int buttonId) {
		return buttonId < this.elapsedTimes.length;
	}

	/**
	 * microUI-API
	 *
	 * @param buttonId
	 *            microUI-API
	 * @return microUI-API
	 */
	public boolean clickEnabled(int buttonId) {
		if (supportsExtendedFeatures(buttonId)) {
			return !this.clicksOff[buttonId];
		}
		return false; // invalid buttonId
	}

	/**
	 * microUI-API
	 *
	 * @param buttonId
	 *            microUI-API
	 * @return microUI-API
	 */
	public boolean doubleClickEnabled(int buttonId) {
		if (supportsExtendedFeatures(buttonId)) {
			int click = this.doubleClicks[buttonId];
			return click != DOUBLE_CLICK_DISABLED;
		}
		return false; // invalid buttonId
	}

	/**
	 * microUI-API
	 */
	@Override
	public int getEventType() {
		return EVENT_TYPE;
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static boolean isPressed(int event) {
		return getAction(event) == PRESSED;
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static boolean isReleased(int event) {
		return getAction(event) == RELEASED;
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static boolean isRepeated(int event) {
		return getAction(event) == REPEATED;
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static boolean isLong(int event) {
		return getAction(event) == LONG;
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static boolean isClicked(int event) {
		return getAction(event) == CLICKED;
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static boolean isDoubleClicked(int event) {
		return getAction(event) == DOUBLE_CLICKED;
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static int getButtonId(int event) {
		return ((event & BUTTON_MASK) >>> BUTTON_SHIFT);
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static int getAction(int event) {
		return (event & ACTION_MASK) >>> ACTION_SHIFT;
	}

	/**
	 * microUI-API
	 *
	 * @param buttonId
	 *            microUI-API
	 * @return microUI-API
	 */
	public long elapsedTime(int buttonId) {
		if (supportsExtendedFeatures(buttonId)) {
			long elapsedTime = this.elapsedTimes[buttonId];
			if (elapsedTime == this.eventTimes[buttonId]) {
				return -1; // the first occurrence
			}
			return elapsedTime;
		}
		return -1;
	}

	/**
	 * microUI-API
	 *
	 * @param action
	 *            microUI-API
	 * @param buttonId
	 *            microUI-API
	 * @return microUI-API
	 */
	public int buildEvent(int action, int buttonId) {
		return Event.buildEvent(getEventType(), this, buildEventData(action, buttonId));
	}

	/**
	 * microUI-API
	 *
	 * @param action
	 *            microUI-API
	 * @param buttonId
	 *            microUI-API
	 */
	public void send(int action, int buttonId) {
		send(buildEvent(action, buttonId), getEventHandler());
	}

	/**
	 * Makes and sends a button action to the given event handler
	 *
	 * @param event
	 *            the button event (action and button ID)
	 * @param eventHandler
	 *            the receiver
	 */
	protected /* SystemButton */ void send(int event, @Nullable EventHandler eventHandler) {

		int buttonId = getButtonId(event);
		boolean extra = supportsExtendedFeatures(buttonId);
		boolean press = false;

		if (extra) {
			press = PRESSED == getAction(event);
			if (press) {
				long t = Util.platformTimeMillis();
				this.elapsedTimes[buttonId] = t - this.eventTimes[buttonId];
				this.eventTimes[buttonId] = t;
			}
		}

		sendEvent(event, eventHandler);

		if (extra && !this.clicksOff[buttonId] && press) { // the array access check is done by extra
			// generate the click on pressed if requested (allow for long click to be
			// treated)
			sendEvent(buildEvent(CLICKED, buttonId), eventHandler);
			if (this.elapsedTimes[buttonId] <= this.doubleClicks[buttonId]) { // doubleClick is -1 for disable (see
				// enableDoubleClick)
				sendEvent(buildEvent(DOUBLE_CLICKED, buttonId), eventHandler);
			}
		}
	}

	private static int buildEventData(int action, int buttonId) {
		return ((action << ACTION_SHIFT) & ACTION_MASK) | ((buttonId << BUTTON_SHIFT) & BUTTON_MASK);
	}

}
